/* This file is part of the KDE project
   Copyright (C) 2002 Werner Trobin <trobin@kde.org>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public License
   along with this library; see the file COPYING.LIB.  If not, write to
   the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
*/

#ifndef _kotoolbutton_h_
#define _kotoolbutton_h_

#include <ktoolbarbutton.h>
#include <tqmap.h>
#include <tqpoint.h>

class TQPopupMenu;

class KoColorPanel : public TQWidget
{
    Q_OBJECT
  TQ_OBJECT
public:
    KoColorPanel( TQWidget* parent = 0, const char* name = 0 );
    virtual ~KoColorPanel();

    virtual TQSize sizeHint() const;
    virtual TQSize minimumSizeHint() const;

    enum MenuStyle { Plain, CustomColors };
    static TQPopupMenu* createColorPopup( MenuStyle style, const TQColor& defaultColor,
                                         const TQObject* receiver, const char* slot,
                                         TQWidget* parent, const char* name );

public slots:
    void clear();
    void insertColor( const TQColor& color );
    void insertColor( const TQColor& color, const TQString& toolTip );
    void insertDefaultColors();

signals:
    void colorSelected( const TQColor& color );

protected:
    virtual void mousePressEvent( TQMouseEvent* e );
    virtual void mouseReleaseEvent( TQMouseEvent* e );
    virtual void mouseMoveEvent( TQMouseEvent* e );
    virtual void paintEvent( TQPaintEvent* e );
    virtual void keyPressEvent( TQKeyEvent* e );
    virtual void focusInEvent( TQFocusEvent* e );
    virtual void dragEnterEvent( TQDragEnterEvent* e );
    virtual void dropEvent( TQDropEvent* e );

private:
    // The position of the 16x16 tiles in "tile steps"
    struct Position {
        Position() : x( -1 ), y( -1 ) {}
        Position( short x_, short y_ ) : x( x_ ), y( y_ ) {}

        short x;
        short y;
    };
    friend bool operator<( const KoColorPanel::Position& lhs, const KoColorPanel::Position& rhs );

    void finalizeInsertion( const Position& pos );
    bool insertColor( const TQColor& color, bool checking );
    bool insertColor( const TQColor& color, const TQString& toolTip, bool checking );
    bool isAvailable( const TQColor& color );

    Position mapToPosition( const TQPoint& point ) const;
    TQColor mapToColor( const TQPoint& point ) const;
    TQColor mapToColor( const Position& position ) const;
    TQRect mapFromPosition( const Position& position ) const;
    Position validPosition( const Position& position );

    int lines() const;
    void paintArea( const TQRect& rect, int& startRow, int& endRow, int& startCol, int& endCol ) const;
    void updateFocusPosition( const Position& newPosition );
    void paint( const Position& position );
    void init();

    Position m_nextPosition, m_focusPosition;
    TQMap<Position, TQColor> m_colorMap;
    TQPoint m_pressedPos;
    bool m_defaultsAdded;
};

// Needed for the use of KoColorPanel::Position in TQMap
bool operator<( const KoColorPanel::Position& lhs, const KoColorPanel::Position& rhs );


// A tiny class needed to emit the correct signal when the default
// color item in the color-panel popup is activated. Additionally
// it's used to provide the color select dialog and manages the recent
// colors... hacky
class KoColorPopupProxy : public TQObject
{
    Q_OBJECT
  TQ_OBJECT
public:
    KoColorPopupProxy( const TQColor& defaultColor, KoColorPanel* recentColors, TQObject* parent, const char* name );
    virtual ~KoColorPopupProxy() {}

    void setRecentColorPanel( KoColorPanel* recentColors );

public slots:
    void slotDefaultColor();
    void slotMoreColors();

signals:
    void colorSelected( const TQColor& color );

private:
    TQColor m_defaultColor;
    KoColorPanel* m_recentColors;
};


// Parts of the code are from KToolBarButton
class KoToolButton : public KToolBarButton
{
    Q_OBJECT
  TQ_OBJECT
public:
    /**
     * Construct a button with an icon loaded by the button itself.
     * This will trust the button to load the correct icon with the
     * correct size.
     *
     * @param icon   Name of icon to load (may be absolute or relative)
     * @param id     Id of this button
     * @param parent This button's parent
     * @param name   This button's internal name
     * @param txt    This button's text (in a tooltip or otherwise)
     */
    KoToolButton( const TQString& icon, int id, TQWidget* parent,
                  const char* name = 0L, const TQString& txt = TQString(),
                  KInstance* _instance = KGlobal::instance() );

    /**
     * Construct a button with an existing pixmap.  It is not
     * recommended that you use this as the internal icon loading code
     * will almost always get it "right".
     *
     * @param icon   Name of icon to load (may be absolute or relative)
     * @param id     Id of this button
     * @param parent This button's parent
     * @param name   This button's internal name
     * @param txt    This button's text (in a tooltip or otherwise)
     */
    KoToolButton( const TQPixmap& pixmap, int id, TQWidget* parent,
                  const char* name = 0L, const TQString& txt = TQString() );

    virtual ~KoToolButton();

    virtual TQSize sizeHint() const;
    virtual TQSize minimumSizeHint() const;
    virtual TQSize minimumSize() const;

public slots:
    void colorSelected( const TQColor& color );

protected:
    virtual void drawButton(TQPainter *p);
    virtual bool eventFilter( TQObject* o, TQEvent* e );

private:
    void init();
    void buttonShift( int& dx, int& dy );
    bool hitArrow( const TQPoint& pos );

    TQPopupMenu* m_popup;
    bool m_arrowPressed;
};

#endif // _kotoolbutton_h_
