/* This file is part of the KDE project
   Copyright (C) 1998, 1999 Torben Weis <weis@kde.org>
   Copyright (C) 2005-2006 Stefan Nikolaus <stefan.nikolaus@kdemail.net>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public License
   along with this library; see the file COPYING.LIB.  If not, write to
   the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
*/

#ifndef KSPREAD_SELECTION
#define KSPREAD_SELECTION

#include <tqobject.h>

#include <kdebug.h>

#include "region.h"

namespace KSpread
{
class View;

class Selection : public TQObject, public Region
{
  Q_OBJECT
  TQ_OBJECT
public:
  Selection(View*);
  Selection(const Selection&);
  ~Selection();

  /**
   * Sets the selection to the given point
   */
  void initialize(const TQPoint&, Sheet* sheet = 0);
  /**
   * Sets the selection to the given range
   */
  void initialize(const TQRect&, Sheet* sheet = 0);
  /**
   * Sets the selection to the given region
   */
  void initialize(const Region&, Sheet* sheet = 0);

  /**
   * Emits signal changed(const Region&)
   */
  void update();
  /**
   * Uses the anchor as starting point
   */
  void update(const TQPoint&);

  /**
   * Extends the current selection with an additional point
   */
  void extend(const TQPoint&, Sheet* sheet = 0);
  /**
   * Extends the current selection with an additional range
   */
  void extend(const TQRect&, Sheet* sheet = 0);
  /**
   * Extends the current selection with an additional region
   */
  void extend(const Region&);

  /**
   * @param point the point's location
   * @param sheet the sheet the point belongs to
   */
  virtual Element* eor(const TQPoint& point, Sheet* sheet = 0);

  /**
   * The anchor is the starting point of a range. For points marker and anchor are the same
   */
  const TQPoint& anchor() const;
  /**
   * The cursor represents the cursor position. This needed for obscured cells
   */
  const TQPoint& cursor() const;
  /**
   * The marker is the end point of a range. For points marker and anchor are the same
   */
  const TQPoint& marker() const;

  /**
   * Checks wether the region consists only of one point
   */
  bool isSingular() const;
  /**
   * @return the area that the 'handle' of the selection is located in painting coordinates
   */
  TQRect selectionHandleArea() const;

  /**
   * @return the name of the region (e.g. "A1:A2")
   */
  TQString name(Sheet* originSheet = 0) const;

  /**
   * Sets the selection's origin sheet.
   * @param sheet the sheet from which the selection starts
   */
  void setSheet(Sheet* sheet);
  /**
   * @return the selection's origin sheet
   */
  Sheet* sheet() const;

  /**
   * Sets the element, which has @p point as anchor, as active
   */
  void setActiveElement(const TQPoint& point);
  /**
   * Sets the @p number 'th element as active
   */
  void setActiveElement(uint number);
  /**
   * @return the active element
   */
  Element* activeElement() const;
  /**
   * Sets the starting position and the length of a subregion in a multiple
   * selection
   */
  void setActiveSubRegion(uint start, uint length);
  /**
   *
   */
  TQString activeSubRegionName() const;
  /**
   * Clears the elements of the subregion
   */
  void clearSubRegion();
  /**
   * fix subregion dimensions
   */
  void fixSubRegionDimension();
  /**
   * Deletes all elements of the region. The result is an empty region.
   */
  virtual void clear();


  /**
   * Sets, wether multiple selections are allowed.
   */
  void setMultipleSelection(bool state);

  TQRect lastRange(bool extend = true) const;
  TQRect extendToMergedAreas(TQRect area) const;

  const TQValueList<TQColor>& colors() const;

  // TODO Stefan #2: replace this
  TQRect selection(bool extend = true) const;

signals:
  void changed(const Region&);

protected:
  class Point;
  class Range;

  virtual Region::Point* createPoint(const TQPoint&) const;
  virtual Region::Point* createPoint(const TQString&) const;
  virtual Region::Point* createPoint(const Point&) const;
  virtual Region::Range* createRange(const TQRect&) const;
  virtual Region::Range* createRange(const TQString&) const;
  virtual Region::Range* createRange(const Range&) const;

private:
  class Private;
  Private *d;
};

/***************************************************************************
  class Selection::Point
****************************************************************************/

class Selection::Point : public Region::Point
{
public:
  Point(const TQPoint& point);
  Point(const TQString& string);

  void setColor(const TQColor& color) { m_color = color; }
  virtual const TQColor& color() const { return m_color; }

  bool columnFixed() const { return m_columnFixed; }
  bool rowFixed() const { return m_rowFixed; }

protected:

private:
  TQColor m_color;
  bool m_columnFixed : 1;
  bool m_rowFixed    : 1;
};

/***************************************************************************
  class Selection::Range
****************************************************************************/

class Selection::Range : public Region::Range
{
public:
  Range(const TQRect& rect);
  Range(const TQString& string);

  virtual bool isColorizable() const { return true; }

  void setColor(const TQColor& color) { m_color = color; }
  const TQColor& color() const { return m_color; }

protected:

private:
  TQColor m_color;
  bool m_leftFixed   : 1;
  bool m_rightFixed  : 1;
  bool m_topFixed    : 1;
  bool m_bottomFixed : 1;
};

} // namespace KSpread


/***************************************************************************
  kdDebug support
****************************************************************************/

inline kdbgstream operator<<( kdbgstream str, const KSpread::Selection& s )
{
  str << "Selection = " << s.name();
  return str;
}

inline kndbgstream operator<<( kndbgstream str, const KSpread::Selection& )
{
  return str;
}

#endif // KSPREAD_SELECTION
