// -*- Mode: c++; c-basic-offset: 4; indent-tabs-mode: nil; tab-width: 4; -*-
/* This file is part of the KDE project
   Copyright (C) 2001 Toshitaka Fujioka <fujioka@kde.org>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public License
   along with this library; see the file COPYING.LIB.  If not, write to
   the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
*/


#include <tqlayout.h>
#include <tqpainter.h>
#include <tqpaintdevicemetrics.h>
#include <tqsimplerichtext.h>
#include <tqlabel.h>

#include <kglobalsettings.h>
#include <kprinter.h>
#include <kdebug.h>
#include <ktextedit.h>
#include <klocale.h>

#include "KPrNoteBar.h"
#include "KPrView.h"
#include "KPrDocument.h"
#include "KPrPage.h"


KPrNoteBar::KPrNoteBar( TQWidget *_parent, KPrView *_view )
    : TQWidget( _parent ),
      view( _view ),
      initialize( true )
{
    TQBoxLayout *topLayout = new TQVBoxLayout( this, KDialog::marginHint(), KDialog::spacingHint() );

    label = new TQLabel(i18n("Note"), this);

    textEdit = new KTextEdit( this );

    TQFont font = KoGlobal::defaultFont();
    textEdit->setCurrentFont( font );

    int currentPageNum = view->getCurrentPresPage(); // 1 base.
    TQString text=TQString();
    if (currentPageNum!=-1)
        text = view->kPresenterDoc()->pageList().at(currentPageNum - 1)->noteText( );
    textEdit->setText( text );

    connect( textEdit, TQT_SIGNAL( textChanged() ),
             this, TQT_SLOT( slotTextChanged() ) );

    connect( textEdit, TQT_SIGNAL( selectionChanged() ),
             this, TQT_SLOT( slotSelectionChanged() ) );

    connect( textEdit, TQT_SIGNAL( copyAvailable( bool ) ),
             this, TQT_SLOT( slotSelectionChanged() ) );

    connect( textEdit, TQT_SIGNAL( undoAvailable( bool ) ),
             this, TQT_SLOT( slotUndoAvailable( bool ) ) );

    connect( textEdit, TQT_SIGNAL( redoAvailable( bool ) ),
             this, TQT_SLOT( slotRedoAvailable( bool ) ) );

    topLayout->addWidget( label );
    topLayout->addWidget( textEdit );
}

KPrNoteBar::~KPrNoteBar()
{
    delete textEdit;
}

void KPrNoteBar::setCurrentNoteText( const TQString &_text )
{
    initialize = true;
    textEdit->setText( _text );
    initialize = false;
}

void KPrNoteBar::slotTextChanged()
{
    int currentPageNum = view->getCurrPgNum(); // 1 base.
    if ( currentPageNum > 0 && !initialize ) {
        if ( view->editMaster() )
            view->kPresenterDoc()->refreshAllNoteBarMasterPage(textEdit->text() , view);
        else
            view->kPresenterDoc()->refreshAllNoteBar(currentPageNum -1,textEdit->text() , view);
        textEdit->setModified( true );
    }
}

void KPrNoteBar::slotSelectionChanged()
{
    kdDebug(33001) << "slotSelectionChanged(): " << textEdit->hasSelectedText() << endl;
}

void KPrNoteBar::slotCopyAvailable( bool yes )
{
    kdDebug(33001) << "slotCopyAvailable( " << yes << " )" << endl;
}

void KPrNoteBar::slotUndoAvailable( bool /*yes*/ )
{
    //kdDebug(33001) << "slotUndoAvailable( " << yes << " )" << endl;
}

void KPrNoteBar::slotRedoAvailable( bool /*yes*/ )
{
    //kdDebug(33001) << "slotRedoAvailable( " << yes << " )" << endl;
}

void KPrNoteBar::printNotes( TQPainter *_painter, KPrinter *_printer, TQValueList<int> _list )
{
    // base code from $TQTDIR/example/textedit/textedit.cpp
    _painter->save();

    TQPaintDeviceMetrics metrics( _painter->device() );
    int dpix = metrics.logicalDpiX();
    int dpiy = metrics.logicalDpiY();

    const int margin = 72; // pt
    TQRect body( margin * dpix / 72, margin * dpiy / 72,
                metrics.width() - margin * dpix / 72 * 2,
                metrics.height() - margin * dpiy / 72 * 2 );

    TQFont font = KoGlobal::defaultFont();
    TQString allText = getNotesTextForPrinting(_list);
    TQString str = TQStyleSheet::convertFromPlainText( allText );

    TQSimpleRichText richText( str, font, TQString(), TQStyleSheet::defaultSheet(),
                              TQMimeSourceFactory::defaultFactory(), body.height() );

    richText.setWidth( _painter, body.width() );

    TQRect viewRect( body );
    do {
        richText.draw( _painter, body.left(), body.top(), viewRect, colorGroup() );
        viewRect.moveBy( 0, body.height() );
        _painter->translate( 0, -body.height() );
        _painter->setFont( font );

        if ( viewRect.top() >= richText.height() )
            break;

        _printer->newPage();
    } while ( true );

    _painter->restore();
}

TQString KPrNoteBar::getNotesTextForPrinting(TQValueList<int> _list) const
{
    TQString allText = TQString();
    bool firstText = true;
    bool noteIsEmpty = true;
    int pageCount = 1;
    KPrDocument *doc=view->kPresenterDoc();
    for ( int i = 0; i < static_cast<int>( doc->pageList().count() ); i++, ++pageCount )
    {
        if (_list.contains(i+1)==0) // that slide isn't printed, don't print its note either
            continue;

        if ( !firstText )
            allText += TQString("\n\n");

        allText += i18n( "Slide Note %1:\n" ).arg( pageCount );
        if(noteIsEmpty && !doc->pageList().at(i)->noteText().isEmpty())
            noteIsEmpty = false;
        allText += doc->pageList().at(i)->noteText();

        firstText = false;
    }
    //code for master page
    if ( !firstText )
        allText += TQString("\n\n");
    allText += i18n( "Master Page Note:\n" );
    if ( !doc->masterPage()->noteText().isEmpty() )
         noteIsEmpty = false;
    allText += doc->masterPage()->noteText();

    if( noteIsEmpty )
        return TQString();
    return allText;
}

#include "KPrNoteBar.moc"
