/* This file is part of the KDE project
   Copyright (C) 2003 Lucijan Busch <lucijan@kde.org>
   Copyright (C) 2004 Cedric Pasteur <cedric.pasteur@free.fr>
   Copyright (C) 2006-2007 Jaroslaw Staniek <js@iidea.pl>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public License
   along with this library; see the file COPYING.LIB.  If not, write to
   the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
*/

#ifndef CONTAINERFACTORY_H
#define CONTAINERFACTORY_H

#include <kcommand.h>

#include "widgetfactory.h"
#include "../utils.h"

namespace KFormDesigner
{
	class Form;
	class FormManager;
	class Container;
}

class InsertPageCommand : public KCommand
{
	public:
		InsertPageCommand(KFormDesigner::Container *container, TQWidget *widget);

		virtual void execute();
		virtual void unexecute();
		virtual TQString name() const;

	protected:
		KFormDesigner::Form *m_form;
		TQString  m_containername;
		TQString  m_name;
		TQString  m_parentname;
		int      m_pageid;
};

//! Helper widget (used when using 'Lay out horizontally')
class KFORMEDITOR_EXPORT HBox : public TQFrame
{
	Q_OBJECT
  TQ_OBJECT

	public:
		HBox(TQWidget *parent, const char *name);
		virtual ~HBox(){;}
		void setPreviewMode() {m_preview = true;}
		virtual void paintEvent(TQPaintEvent *ev);

	protected:
		bool  m_preview;
};

//! Helper widget (used when using 'Lay out vertically')
class KFORMEDITOR_EXPORT VBox : public TQFrame
{
	Q_OBJECT
  TQ_OBJECT

	public:
		VBox(TQWidget *parent, const char *name);
		virtual ~VBox(){;}
		void setPreviewMode() {m_preview = true;}
		virtual void paintEvent(TQPaintEvent *ev);

	protected:
		bool  m_preview;
};

//! Helper widget (used when using 'Lay out in a grid')
class KFORMEDITOR_EXPORT Grid : public TQFrame
{
	Q_OBJECT
  TQ_OBJECT

	public:
		Grid(TQWidget *parent, const char *name);
		virtual ~Grid(){;}
		void setPreviewMode() {m_preview = true;}
		virtual void paintEvent(TQPaintEvent *ev);

	protected:
		bool  m_preview;
};

//! Helper widget (used when using 'Lay out with horizontal flow')
class KFORMEDITOR_EXPORT HFlow : public TQFrame
{
	Q_OBJECT
  TQ_OBJECT

	public:
		HFlow(TQWidget *parent, const char *name);
		virtual ~HFlow(){;}
		void setPreviewMode() {m_preview = true;}
		virtual void paintEvent(TQPaintEvent *ev);

	protected:
		bool  m_preview;
};

//! Helper widget (used when using 'Lay out with horizontal flow')
class KFORMEDITOR_EXPORT VFlow : public TQFrame
{
	Q_OBJECT
  TQ_OBJECT

	public:
		VFlow(TQWidget *parent, const char *name);
		virtual ~VFlow(){;}
		void setPreviewMode() {m_preview = true;}
		virtual void paintEvent(TQPaintEvent *ev);
		virtual TQSize sizeHint() const;

	protected:
		bool  m_preview;
};

//! A simple container widget
class KFORMEDITOR_EXPORT ContainerWidget : public TQWidget
{
	Q_OBJECT
  TQ_OBJECT

	friend class KFDTabWidget;

	public:
		ContainerWidget(TQWidget *parent, const char *name);
		virtual ~ContainerWidget();

		virtual TQSize sizeHint() const;

		//! Used to emit handleDragMoveEvent() signal needed to control dragging over the container's surface
		virtual void dragMoveEvent( TQDragMoveEvent *e );

		//! Used to emit handleDropEvent() signal needed to control dropping on the container's surface
		virtual void dropEvent( TQDropEvent *e );

	signals:
		//! Needed to control dragging over the container's surface
		void handleDragMoveEvent(TQDragMoveEvent *e);

		//! Needed to control dropping on the container's surface
		void handleDropEvent(TQDropEvent *e);
};

//! A tab widget
class KFORMEDITOR_EXPORT KFDTabWidget : public KFormDesigner::TabWidget
{
	Q_OBJECT
  TQ_OBJECT

	public:
		KFDTabWidget(TQWidget *parent, const char *name);
		virtual ~KFDTabWidget();

		virtual TQSize sizeHint() const;

		//! Used to emit handleDragMoveEvent() signal needed to control dragging over the container's surface
		virtual void dragMoveEvent( TQDragMoveEvent *e );

		//! Used to emit handleDropEvent() signal needed to control dropping on the container's surface
		virtual void dropEvent( TQDropEvent *e );

	signals:
		//! Needed to control dragging over the container's surface
		void handleDragMoveEvent(TQDragMoveEvent *e);

		//! Needed to control dropping on the container's surface
		void handleDropEvent(TQDropEvent *e);
};

//! A group box widget
class KFORMEDITOR_EXPORT GroupBox : public TQGroupBox
{
	Q_OBJECT
  TQ_OBJECT

	public:
		GroupBox(const TQString & title, TQWidget *parent, const char *name);
		virtual ~GroupBox();

		//! Used to emit handleDragMoveEvent() signal needed to control dragging over the container's surface
		virtual void dragMoveEvent( TQDragMoveEvent *e );

		//! Used to emit handleDropEvent() signal needed to control dropping on the container's surface
		virtual void dropEvent( TQDropEvent *e );

	signals:
		//! Needed to control dragging over the container's surface
		void handleDragMoveEvent(TQDragMoveEvent *e);

		//! Needed to control dropping on the container's surface
		void handleDropEvent(TQDropEvent *e);
};

//! A form embedded as a widget inside other form
class KFORMEDITOR_EXPORT SubForm : public TQScrollView
{
	Q_OBJECT
  TQ_OBJECT
	TQ_PROPERTY(TQString formName READ formName WRITE setFormName DESIGNABLE true)

	public:
		SubForm(TQWidget *parent, const char *name);
		~SubForm() {}

		//! \return the name of the subform inside the db
		TQString   formName() const { return m_formName; }
		void      setFormName(const TQString &name);

	private:
//		KFormDesigner::FormManager *m_manager;
		KFormDesigner::Form   *m_form;
		TQWidget  *m_widget;
		TQString   m_formName;
};

//! Standard Factory for all container widgets
class ContainerFactory : public KFormDesigner::WidgetFactory
{
	Q_OBJECT
  TQ_OBJECT

	public:
		ContainerFactory(TQObject *parent, const char *name, const TQStringList &args);
		virtual ~ContainerFactory();

		virtual TQWidget *createWidget(const TQCString & classname, TQWidget *parent, const char *name, KFormDesigner::Container *container,
			int options = DefaultOptions);
		virtual bool createMenuActions(const TQCString& classname, TQWidget *w, TQPopupMenu *menu,
			KFormDesigner::Container *container);
		virtual bool startEditing(const TQCString &classname, TQWidget *w,
			KFormDesigner::Container *container);
		virtual bool previewWidget(const TQCString &classname, TQWidget *widget,
			KFormDesigner::Container *container);
		virtual bool saveSpecialProperty(const TQCString &classname, const TQString &name,
			const TQVariant &value, TQWidget *w, TQDomElement &parentNode, TQDomDocument &parent);
		virtual bool readSpecialProperty(const TQCString &classname, TQDomElement &node, TQWidget *w,
			KFormDesigner::ObjectTreeItem *item);
		virtual TQValueList<TQCString> autoSaveProperties(const TQCString &classname);

	protected:
		virtual bool isPropertyVisibleInternal(const TQCString &classname, TQWidget *w, 
			const TQCString &property, bool isTopLevel);
		virtual bool changeText(const TQString &newText);
		virtual void resizeEditor(TQWidget *editor, TQWidget *widget, const TQCString &classname);

	public slots:
		void addTabPage();
		void addStackPage();
		void renameTabPage();
		void removeTabPage();
		void removeStackPage();
		void prevStackPage();
		void nextStackPage();
		void reorderTabs(int oldpos, int newpos);

	private:
//		TQWidget *m_widget;
//		KFormDesigner::Container *m_container;
//		KFormDesigner::FormManager  *m_manager;
};

#endif
