/* This file is part of the KDE project
   Copyright (C) 2007 Jaroslaw Staniek <js@iidea.pl>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public License
   along with this library; see the file COPYING.LIB.  If not, write to
   the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
*/

#include "kexitemplateloader.h"

#include <kstandarddirs.h>
#include <kglobal.h>
#include <klocale.h>
#include <kconfig.h>
#include <kdebug.h>
#include <kiconloader.h>
#include <kapplication.h>

#include <tqdir.h>

//static
KexiTemplateInfo::List KexiTemplateLoader::loadListInfo()
{
	KexiTemplateInfo::List list;
	const TQString subdir = TQString(kapp->instanceName()) + "/templates";
	TQString lang( KGlobal::locale()->language() );
	TQStringList dirs( kapp->dirs()->findDirs("data", subdir) );
	while (true) {
		foreach( TQStringList::ConstIterator, it, dirs) {
			TQDir dir((*it)+lang);
			if (!dir.exists())
				continue;
			if (!dir.isReadable()) {
				kdWarning() << "KexiTemplateLoader::loadListInfo() \"" << dir.absPath() << "\" not readable!" << endl;
				continue;
			}
			const TQStringList templateDirs( dir.entryList(TQDir::Dirs, TQDir::Name) );
			const TQString absDirPath( dir.absPath() + '/' );
			foreach(TQStringList::ConstIterator, templateIt, templateDirs) {
				if ((*templateIt)=="." || (*templateIt==".."))
					continue;
				KexiTemplateInfo info = KexiTemplateLoader::loadInfo( absDirPath + *templateIt );
				if (!info.name.isEmpty())
					list.append( info );	
			}
		}
		if (lang != "en" && list.isEmpty()) //not found for current locale, try "en"
			lang = "en";
		else
			break;
	}
	return list;
}

//static
KexiTemplateInfo KexiTemplateLoader::loadInfo(const TQString& directory)
{
	TQDir dir(directory);
	if (!dir.isReadable()) {
		kdWarning() << "KexiTemplateLoader::loadInfo() \"" 
			<< directory << "\" not readable!" << endl;
		return KexiTemplateInfo();
	}
	if (!TQFileInfo(directory+"/info.txt").isReadable())
		return KexiTemplateInfo();
	KConfig infoTxt(directory+"/info.txt", true/*readonly*/, false/*local*/);
	KexiTemplateInfo info;
	info.name = infoTxt.readEntry("Name");
	if (info.name.isEmpty()) {
		kdWarning() << "KexiTemplateLoader::loadInfo() \"" << (directory+"/info.txt") << "\" contains no \"name\" field" << endl;
		return KexiTemplateInfo();
	}
	const TQStringList templateFiles( dir.entryList("*.kexi", TQDir::Files|TQDir::Readable, TQDir::Name) );
	if (templateFiles.isEmpty()) {
		kdWarning() << "KexiTemplateLoader::loadInfo() no readable .kexi template file found in \"" << directory << "\"" << endl;
		return KexiTemplateInfo();
	}
	info.filename = directory+"/"+templateFiles.first();
	info.description = infoTxt.readEntry("Description");
	const TQString iconFileName( infoTxt.readEntry("Icon") );
	if (!iconFileName.isEmpty())
		info.icon = TQPixmap(directory+'/'+iconFileName);
	if (info.icon.isNull())
		info.icon = DesktopIcon("kexiproject_sqlite"); //default
	TQStringList autoopenObjectsString = infoTxt.readListEntry("AutoOpenObjects");
	foreach( TQStringList::ConstIterator, it, autoopenObjectsString) {
		KexiProjectData::ObjectInfo autoopenObject;
		TQStringList autoopenObjectNameSplitted( TQStringList::split(':', *it) );
		if (autoopenObjectNameSplitted.count()>1) {
			autoopenObject["type"] = autoopenObjectNameSplitted[0];
			autoopenObject["name"] = autoopenObjectNameSplitted[1];
		}
		else {
			autoopenObject["type"] = "table";
			autoopenObject["name"] = autoopenObjectNameSplitted[0];
		}
		autoopenObject["action"] = "open";
		info.autoopenObjects.append( autoopenObject );
	}
	return info;
}
