/* This file is part of the KDE project
   Copyright (C) 2000 Enno Bartels <ebartels@nwn.de>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public License
   along with this library; see the file COPYING.LIB.  If not, write to
   the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
*/

#include <config.h>

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#include <tqmessagebox.h>
#include <tqptrlist.h>
#include <applixwordimport.h>
#include <applixwordimport.moc>
#include <kdebug.h>
#include <KoFilterChain.h>
#include <kgenericfactory.h>

typedef KGenericFactory<APPLIXWORDImport, KoFilter> APPLIXWORDImportFactory;
K_EXPORT_COMPONENT_FACTORY( libapplixwordimport, APPLIXWORDImportFactory( "kofficefilters" ) )


/******************************************************************************
 *  class: APPLIXWORDImport        function: APPLIXWORDImport                 *
 ******************************************************************************
 *                                                                            *
 *  Short description : Constructor                                           *
 *                                                                            *
 *                                                                            *
 ******************************************************************************/
APPLIXWORDImport::APPLIXWORDImport (KoFilter *, const char *, const TQStringList& ) :
                     KoFilter()
{
}

/******************************************************************************
 *  class: APPLIXWORDImport        function: nextLine                         *
 ******************************************************************************
 *                                                                            *
 *  Short description : Readline and update progressbar                       *
 *                                                                            *
 *                                                                            *
 ******************************************************************************/
TQString
APPLIXWORDImport::nextLine (TQTextStream & stream)
{
    TQString s;

    // Read one Line
    s = stream.readLine();

    m_instep += s.length();
    if (m_instep > m_stepsize)
    {
        m_instep    = 0;
        m_progress += 2;
        emit sigProgress (m_progress) ;
    }

    return s;
}



/******************************************************************************
 *  class: APPLIXWORDImport        function: filter                           *
 ******************************************************************************
 *                                                                            *
 *  Short description :                                                       *
 *                                                                            *
 *                                                                            *
 ******************************************************************************/
KoFilter::ConversionStatus APPLIXWORDImport::convert( const TQCString& from, const TQCString& to )
{

    if (to!="application/x-kword" || from!="application/x-applixword")
        return KoFilter::NotImplemented;

    TQFile in(m_chain->inputFile());
    if (!in.open (IO_ReadOnly))
    {
        kdError(30517) << "Unable to open input file!" << endl;
        in.close();
        return KoFilter::FileNotFound;
    }


    TQString str;

    str += "<?xml version=\"1.0\"?>\n";
    str += "<DOC  author=\"Reginald Stadlbauer and Torben Weis\" email=\"reggie@kde.org and weis@kde.org\" editor=\"KWord\" mime=\"application/x-kword\">\n";
    str += " <PAPER format=\"1\" ptWidth=\"595\" ptHeight=\"841\" mmWidth =\"210\" mmHeight=\"297\" inchWidth =\"8.26772\" inchHeight=\"11.6929\" orientation=\"0\" columns=\"1\" ptColumnspc=\"2\" mmColumnspc=\"1\" inchColumnspc=\"0.0393701\" hType=\"0\" fType=\"0\" ptHeadBody=\"9\" ptFootBody=\"9\" mmHeadBody=\"3.5\" mmFootBody=\"3.5\" inchHeadBody=\"0.137795\" inchFootBody=\"0.137795\">\n";
    str += "  <PAPERBORDERS mmLeft=\"10\" mmTop=\"15\" mmRight=\"10\" mmBottom=\"15\" ptLeft=\"28\" ptTop=\"42\" ptRight=\"28\" ptBottom=\"42\" inchLeft=\"0.393701\" inchTop=\"0.590551\" inchRight=\"0.393701\" inchBottom=\"0.590551\"/>\n";
    str += " </PAPER>\n";
    str += " <ATTRIBUTES processing=\"0\" standardpage=\"1\" hasHeader=\"0\" hasFooter=\"0\" />\n";
    str += " <FRAMESETS>\n";
    str += "  <FRAMESET frameType=\"1\" autoCreateNewFrame=\"1\" frameInfo=\"0\" removeable=\"0\">\n";
    str += "   <FRAME left=\"28\" top=\"42\" right=\"566\" bottom=\"798\" runaround=\"1\" runaGapPT=\"2\" runaGapMM=\"1\" runaGapINCH=\"0.0393701\"  lWidth=\"1\" lRed=\"255\" lGreen=\"255\" lBlue=\"255\" lStyle=\"0\"  rWidth=\"1\" rRed=\"255\" rGreen=\"255\" rBlue=\"255\" rStyle=\"0\"  tWidth=\"1\" tRed=\"255\" tGreen=\"255\" tBlue=\"255\" tStyle=\"0\"  bWidth=\"1\" bRed=\"255\" bGreen=\"255\" bBlue=\"255\" bStyle=\"0\" bkRed=\"255\" bkGreen=\"255\" bkBlue=\"255\" bleftpt=\"0\" bleftmm=\"0\" bleftinch=\"0\" brightpt=\"0\" brightmm=\"0\" brightinch=\"0\" btoppt=\"0\" btopmm=\"0\" btopinch=\"0\" bbottompt=\"0\" bbottommm=\"0\" bbottominch=\"0\"/>\n";


    TQTextStream stream (&in);

    m_stepsize = in.size()/50;
    m_instep   = 0;
    m_progress = 0;

    int  rueck;
    int  pos, ok;
    char stylename[100];
    TQString           mystr, textstr;
    TQPtrList<t_mycolor>  mcol;
    TQStringList       mcoltxt;

    /**************************************************************************
     * Read header                                                            *
     **************************************************************************/
    if (! readHeader (stream, in)) return KoFilter::StupidError;


    while (!stream.atEnd())
    {
        // Read one line
        mystr = readTagLine (stream, in);
        ok = true;

        /**********************************************************************
         * jump over start_styles if it exists                                *
         **********************************************************************/
        if (mystr == "<start_styles>")
	{
          printf ("Start styles\n");
          t_mycolor *col = new t_mycolor; // delete is in place
          TQString    coltxt ;
          int zaehler = 0; // Note: "zaehler" means "counter" in English
          do
	  {
            mystr = readTagLine (stream, in);
            if (mystr == "<end_styles>")
	    {
             ok = false;
             kdDebug(30517)<<"End styles\n\n";
	    }
	    else
	    {
              if (mystr.startsWith ("<color "))
	      {
		mystr.remove (0, 8);
                pos = mystr.find ("\"");
		coltxt = mystr.left (pos);
                mystr.remove (0,pos+1);
                rueck = sscanf ((const char *) mystr.latin1() ,
                                ":%d:%d:%d:%d>",
	                         &col->c, &col->m, &col->y, &col->k);
		kdDebug(30517)<<"  Color " <<  zaehler<<"  : "<<col->c << "  " << col->m<< " "<< col->y<<" "<< col->k<<" "<<coltxt<<" "<<endl;
                zaehler ++;

                // Color transformation cmyk -> rgb
                col->r = 255 - (col->c + col->k);
                if (col->r < 0) col->r = 0;

                col->g = 255 - (col->m + col->k);
                if (col->g < 0) col->g = 0;

                col->b = 255 - (col->y + col->k);
                if (col->b < 0) col->b = 0;

	        mcol.append    (col);
	        mcoltxt.append (coltxt);
	      } //end if ...<col...
	    } //end else
	  } // end while
          while (ok == true);
          delete col;
        } // end if ...<start_styles>...
        /***********************************************************************
         * jump over embedded Applix docs                                      *
         ***********************************************************************/
        else if (mystr == "<start_data Applix>")
	{
          kdDebug(30517)<<"\nEmbedded Applix object starts:\n";
          do
	  {
            mystr = readTagLine (stream, in);
            if (mystr == "<end_data>") ok = false;
	    else
	    {
              kdDebug(30517)<<"   "<<mystr<<endl;
	    }
	  }
          while (ok == true);
          kdDebug(30517)<<"Embedded Applix object ends\n\n";

        }
        /**********************************************************************
         * jump over header footer                                            *
         **********************************************************************/
        else if (mystr.startsWith ("<start_hdrftr "))
	{
          kdDebug(30517)<<"\nHeader/Footer starts:\n";
          do
	  {
            mystr = readTagLine (stream, in);
            if (mystr == "<end_hdrftr>") ok = false;
	    else
	    {
              kdDebug(30517)<<"    "<<mystr<<endl;
	    }
	  }
          while (ok == true);
          kdDebug(30517)<<"\nHeader/Footer ends\n";
        }
        /**********************************************************************
         * found a paragraph string                                           *
         **********************************************************************/
        else if (mystr.startsWith ("<P "))
	{
	   sscanf ( (const char *) mystr.latin1(), "<P \"%99s\"", stylename);
           mystr.remove (0, 5+strlen(stylename));
           kdDebug(30517)<<" Para  Name: "<< stylename<<endl;
           kdDebug(30517)<<"       Rest: "<<mystr<<endl;
	}
        /**********************************************************************
         * found a textstring                                                 *
         **********************************************************************/
        else if (mystr.startsWith ("<T "))
	{
          TQString colname;

          // Remove starting tab info
          mystr.remove (0, 4);

          // Remove ending >
          mystr.remove (mystr.length()-1, 1);

          // Separate textstring "
          ok = true;
          int y=0;
          do
	  {
             pos = mystr.find ("\"", y);
	     kdDebug(30517)<<"POS:"<<pos<<" length:"<< mystr.length()<<" y:"<<y <<endl;

             kdDebug(30517)<<"< "<<mystr<<" >\n";
             if(  (pos-1 > -1) && (mystr[pos-1] == '\\'))
             {
               kdDebug(30517)<<" No string end - but G�nsef�sschen\n";
               y=pos+1;
             }
             else
             {
               kdDebug(30517)<<" String end //\n";
               ok = false;
             }
	  }
          while (ok == true);

          textstr = mystr.left (pos);
          mystr.remove (0, pos+1);
          mystr.stripWhiteSpace();
	  kdDebug(30517) <<"Text:<" <<textstr <<" > "<< pos<<"  Rest:<"<< mystr<<"> \n";

          // split format
          TQStringList typeList;
          typeList = TQStringList::split (' ', mystr);

          int fontsize=12, bold=0, italic=0, underline=0, colpos=-1;
          TQString fontname;
          int nn=0;
          for (TQStringList::Iterator it = typeList.begin(); it != typeList.end(); ++it )
          {
            kdDebug(30517) <<"   No: "<< nn<< "   > "<< (*it)<< "< = \n";

            // Looking for bold
            if      ((*it) == "bold")
            {
              bold = 1;
              kdDebug(30517)<<"bold\n";
	    }
            else if ((*it) == "no-bold")
            {
              bold = 0;
              kdDebug(30517)<<"no bold\n";
	    }
            else if ((*it) == "italic")
            {
              italic = 1;
              kdDebug(30517)<<"italic\n";
	    }
            else if ((*it) == "no-italic")
            {
              italic = 0;
              kdDebug(30517) <<"no italic\n";
	    }
            else if ((*it) == "underline")
            {
              underline = 1;
              kdDebug(30517)<<"underline\n";
	    }
            else if ((*it) == "no-underline")
            {
              underline = 0;
              kdDebug(30517) <<"no underline\n";
	    }
            else if ((*it).startsWith ("size"))
            {
              (*it).remove (0, 5);
	      sscanf ( (const char *) (*it).latin1(), "%d", &fontsize);
              kdDebug(30517)<<"fontsize: "<< fontsize<<endl;
	    }
            else if ((*it).startsWith ("face"))
            {
              (*it).remove (0, 6);
	      (*it).remove ((*it).length()-1, 1);
  	      fontname = *it;
              kdDebug(30517)<<"fontname: "<<fontname<<endl;
	    }
            else if ((*it).startsWith ("color:"))
            {
              (*it).remove (0, 7);
	      (*it).remove ((*it).length()-1, 1);
              colname = *it;
              colpos = mcoltxt.findIndex (colname);
	      kdDebug(30517) <<"  Color: "<< colname<<" "<< colpos <<" \n";
	    }
            else
            {
	      kdDebug(30517)<<" "<< (*it)<<endl;
            }


	  }
          kdDebug(30517) <<"\n";

          // Replaces Part for & <>, applixwear special characters and qouts
          replaceSpecial (textstr);


          // add text inside
          str += "    <PARAGRAPH>\n";
          str += "     <TEXT>";
          str += textstr;
          str += "</TEXT>\n";

          if (bold == 1 || underline == 1 || italic == 1 || fontsize != 12 ||
              colpos != -1 || !fontname.isEmpty())
	  {
            str += "     <LAYOUT>\n";
            str += "      <FORMAT>\n";
            if (!fontname.isEmpty())
	    {
              str += "       <FONT name=\"";
              str += fontname;
              str += "\" />\n";
	    }

            if (fontsize != 1)
	    {
              str += "       <SIZE value=\"";
              str += TQString::number (fontsize);
              str += "\" />\n";
	    }

            if (italic == 1)
	    {
              str += "       <ITALIC value=\"1\" />\n";
	    }

            if (bold == 1)
	    {
              str += "       <WEIGHT value=\"75\" />\n";
	    }

            if (underline == 1)
	    {
              str += "       <UNDERLINE value=\"1\" />\n";
	    }

            if (colpos != -1)
	    {
              t_mycolor *mc = new t_mycolor; // delete is in place
              mc = mcol.at(colpos);
              str += "       <COLOR red=\"";

              str += TQString::number (mc->r);
              str += "\" green=\"";
              str += TQString::number (mc->g);
              str += "\" blue=\"";
              str += TQString::number (mc->b);
              str += "\" />\n";
              delete mc;
	    }

            str += "      </FORMAT>\n";
            str += "     </LAYOUT>\n";
	  }
          str += "    </PARAGRAPH>\n";
	}

    }
    emit sigProgress(100);

    str += "  </FRAMESET>\n";
    str += " </FRAMESETS>\n";
    str += "</DOC>\n";
    kdDebug(30517)<<"Text "<< str <<endl;

    KoStoreDevice* out = m_chain->storageFile( "root", KoStore::Write );
    if (!out)
    {
        kdError(30517) << "Unable to open output file!" << endl;
        in.close  ();
        return KoFilter::StorageCreationError;
    }

    TQCString cstring = str.utf8 ();

    out->writeBlock ((const char*) cstring, cstring.length());

    in.close ();
    return KoFilter::OK;
}


/******************************************************************************
 *  function: specCharfind                                                    *
 ******************************************************************************/
TQChar
APPLIXWORDImport::specCharfind (TQChar a, TQChar b)
{
  TQChar chr;

   if      ( (a == 'n') && (b == 'p') ) chr = '�';


   else if ( (a == 'n') && (b == 'c') ) chr = '�';
   else if ( (a == 'p') && (b == 'c') ) chr = '�';

   else if ( (a == 'n') && (b == 'd') ) chr = '�';
   else if ( (a == 'p') && (b == 'd') ) chr = '�';

   else if ( (a == 'n') && (b == 'e') ) chr = '�';
   else if ( (a == 'p') && (b == 'e') ) chr = '�';

   else if ( (a == 'n') && (b == 'f') ) chr = '�';
   else if ( (a == 'p') && (b == 'f') ) chr = '�';

   else if ( (a == 'p') && (b == 'g') ) chr = '�';
   else if ( (a == 'n') && (b == 'g') ) chr = '�';



   else if ( (a == 'n') && (b == 'j') ) chr = '�';
   else if ( (a == 'p') && (b == 'j') ) chr = '�';

   else if ( (a == 'n') && (b == 'k') ) chr = '�';
   else if ( (a == 'p') && (b == 'k') ) chr = '�';

   else if ( (a == 'n') && (b == 'l') ) chr = '�';
   else if ( (a == 'p') && (b == 'l') ) chr = '�';

   else if ( (a == 'p') && (b == 'm') ) chr = '�';
   else if ( (a == 'n') && (b == 'm') ) chr = '�';



   else if ( (a == 'm') && (b == 'a') ) chr = '�';
   else if ( (a == 'o') && (b == 'a') ) chr = '�';

   else if ( (a == 'm') && (b == 'b') ) chr = '�';
   else if ( (a == 'o') && (b == 'b') ) chr = '�';

   else if ( (a == 'm') && (b == 'c') ) chr = '�';
   else if ( (a == 'o') && (b == 'c') ) chr = '�';

   else if ( (a == 'm') && (b == 'd') ) chr = '�';
   else if ( (a == 'o') && (b == 'd') ) chr = '�';

   else if ( (a == 'm') && (b == 'e') ) chr = '�';
   else if ( (a == 'o') && (b == 'e') ) chr = '�';

   else if ( (a == 'm') && (b == 'f') ) chr = '�';
   else if ( (a == 'o') && (b == 'f') ) chr = '�';

   else if ( (a == 'm') && (b == 'g') ) chr = '�';
   else if ( (a == 'o') && (b == 'g') ) chr = '�';



   else if ( (a == 'm') && (b == 'i') ) chr = '�';
   else if ( (a == 'o') && (b == 'i') ) chr = '�';

   else if ( (a == 'm') && (b == 'j') ) chr = '�';
   else if ( (a == 'o') && (b == 'j') ) chr = '�';

   else if ( (a == 'm') && (b == 'k') ) chr = '�';
   else if ( (a == 'o') && (b == 'k') ) chr = '�';

   else if ( (a == 'm') && (b == 'l') ) chr = '�';
   else if ( (a == 'o') && (b == 'l') ) chr = '�';






   else if ( (a == 'm') && (b == 'm') ) chr = '�';
   else if ( (a == 'o') && (b == 'm') ) chr = '�';

   else if ( (a == 'm') && (b == 'n') ) chr = '�';
   else if ( (a == 'o') && (b == 'n') ) chr = '�';

   else if ( (a == 'm') && (b == 'o') ) chr = '�';
   else if ( (a == 'o') && (b == 'o') ) chr = '�';

   else if ( (a == 'm') && (b == 'p') ) chr = '�';
   else if ( (a == 'o') && (b == 'p') ) chr = '�';


   else if ( (a == 'n') && (b == 'b') ) chr = '�';
   else if ( (a == 'p') && (b == 'b') ) chr = '�';


   else if ( (a == 'k') && (b == 'c') ) chr = '�';
   else if ( (a == 'k') && (b == 'j') ) chr = '�';
   else if ( (a == 'l') && (b == 'f') ) chr = '�';
   else if ( (a == 'n') && (b == 'i') ) chr = '�';
   else if ( (a == 'p') && (b == 'i') ) chr = '�';

   else if ( (a == 'l') && (b == 'j') ) chr = '�';
   else if ( (a == 'l') && (b == 'c') ) chr = '�';
   else if ( (a == 'l') && (b == 'd') ) chr = '�';

   else if ( (a == 'l') && (b == 'm') ) chr = '�';
   else if ( (a == 'l') && (b == 'n') ) chr = '�';
   else if ( (a == 'l') && (b == 'o') ) chr = '�';

   else if ( (a == 'l') && (b == 'a') ) chr = '�';

   else if ( (a == 'k') && (b == 'o') ) chr = '�';
   else if ( (a == 'k') && (b == 'h') ) chr = '�';
   else if ( (a == 'k') && (b == 'd') ) chr = '�';

   else if ( (a == 'p') && (b == 'a') ) chr = '�';
   else if ( (a == 'n') && (b == 'a') ) chr = '�';

   else if ( (a == 'l') && (b == 'l') ) chr = '�';
   else if ( (a == 'k') && (b == 'l') ) chr = '�';

   else if ( (a == 'l') && (b == 'k') ) chr = '�';

   else if ( (a == 'l') && (b == 'h') ) chr = '�';

   else if ( (a == 'k') && (b == 'b') ) chr = '�';

   else if ( (a == 'k') && (b == 'e') ) chr = '�';

   else if ( (a == 'l') && (b == 'b') ) chr = '�';

   else if ( (a == 'l') && (b == 'p') ) chr = '�';

   else if ( (a == 'k') && (b == 'f') ) chr = '�';

   else if ( (a == 'p') && (b == 'o') ) chr = '�';
   else if ( (a == 'n') && (b == 'o') ) chr = '�';

   else if ( (a == 'n') && (b == 'n') ) chr = '�';
   else if ( (a == 'p') && (b == 'n') ) chr = '�';
   else if ( (a == 'p') && (b == 'p') ) chr = '�';

   else if ( (a == 'k') && (b == 'k') ) chr = '�';

   else if ( (a == 'k') && (b == 'm') ) chr = '�';
   else if ( (a == 'p') && (b == 'h') ) chr = '�';

   else if ( (a == 'k') && (b == 'g') ) chr = '|';

   else if ( (a == 'l') && (b == 'e') ) chr = '\'';

   else if ( (a == 'k') && (b == 'i') ) chr = '�';

   else if ( (a == 'k') && (b == 'n') ) chr = '�';

   else if ( (a == 'k') && (b == 'p') ) chr = '�';

   else if ( (a == 'l') && (b == 'g') ) chr = '�';

   else if ( (a == 'l') && (b == 'i') ) chr = '�';

   else if ( (a == 'm') && (b == 'h') ) chr = '�';
   else if ( (a == 'o') && (b == 'h') ) chr = '�';

   else if ( (a == 'n') && (b == 'h') ) chr = '�';

   else if ( (a == 'k') && (b == 'a') ) chr = '�';

   else if ( (a == 'a') && (b == 'j') ) chr = '!';

   else  chr = '#';

   return chr;
}



/******************************************************************************
 *  class: APPLIXWORDImport        function: readTagLine                      *
 ******************************************************************************
 *                                                                            *
 *  Short description :                                                       *
 *                                                                            *
 *                                                                            *
 ******************************************************************************/
TQString
APPLIXWORDImport::readTagLine (TQTextStream &stream, TQFile &in)
{
  TQString mystrn, mystr;
  int     ok, pos;

   // Read one line
   mystr = nextLine (stream);

   // Delete whitespaces
   mystr.stripWhiteSpace();

   // Look if the tag continues on the next line
   if ((mystr.length() == 80) && (mystr[mystr.length()-1] == '\\') )
   {
     ok = true;
     do
     {
       // Get the actual position in the file
       pos = in.at ();

       // Read next line
       mystrn = nextLine (stream);

       // Is the new line a new tag line
       if (mystrn[0] == ' ')
       {
	 // remove the whitespace at the start of the new line
         mystrn.remove (0, 1);

	 // remove the '/' at the end of the old line
         mystr.remove (mystr.length()-1, 1);

         // append the new line
         mystr += mystrn;
       }
       else
       {
         // was the newline a newtaglien the jump back
         in.at (pos);
         ok = false;
       }
     }
     while (ok == true);
   }

   return mystr;
}




/******************************************************************************
 *  class: APPLIXWORDImport        function: replaceSpecial                   *
 ******************************************************************************
 *                                                                            *
 *  Short description :                                                       *
 *                                                                            *
 *                                                                            *
 ******************************************************************************/
void
APPLIXWORDImport::replaceSpecial (TQString &textstr)
{
  int ok, pos;

   // 1. Replace Part for this characters: <, >, &
   textstr.replace ('&', "&amp;");
   textstr.replace ('<', "&lt;");
   textstr.replace ('>', "&gt;");


   // 2. Replace part for this characters: applixwear qoutes
   ok = true;
   pos = 0;
   do
   {
      // Searching for an quote
      pos = textstr.find ('\"', pos);

      // Is it an textqoute ?
      if ((pos > -1) && (textstr[pos-1] == '\\'))
      {
        textstr.replace (pos-1, 2,"\"");
      }
      else
      {
        ok = false;
      }
   }
   while (ok == true);



   // 3. Replace part for Applix Characters
   int   foundSpecialCharakter;
   TQChar newchar;

   do
   {
      // initialize
      foundSpecialCharakter = false;

      pos = textstr.find ("^");

      // is there a special character ?
      if (pos > -1 )
      {
         // i have found a special character !
         foundSpecialCharakter = true;

         // translate the applix special character
         newchar = specCharfind (textstr[pos+1], textstr[pos+2]);

         // replace the character
         textstr.replace (pos, 3, newchar);
      }
    }
    while (foundSpecialCharakter == true);
}



/******************************************************************************
 *  class: APPLIXWORDImport       function: readHeader                        *
 ******************************************************************************
 *                                                                            *
 *  Short description :                                                       *
 *                                                                            *
 *                                                                            *
 ******************************************************************************/
int
APPLIXWORDImport::readHeader (TQTextStream &stream, TQFile &in)
{
  TQString mystr;
  int     rueck;
  int     vers[3] = { 0, 0, 0 };

    // Read Headline
    mystr = readTagLine (stream, in);

    // mystr = stream.readLine ();
    rueck = sscanf ((const char *) mystr.latin1() ,
                    "*BEGIN WORDS VERSION=%d/%d ENCODING=%dBIT",
	             &vers[0], &vers[1], &vers[2]);
    printf ("Versions info: %d %d %d\n", vers[0], vers[1], vers[2]);

    // Check the headline
    if (rueck <= 0)
    {
      printf ("Header not correkt - May be it is not an applixword file\n");
      printf ("Headerline: <%s>\n", (const char *) mystr.latin1());

      TQMessageBox::critical (0L, "Applixword header problem",
                                  TQString ("The Applixword header is not correct. "
                                           "May be it is not an applixword file! <BR>"
                                           "This is the header line I did read:<BR><B>%1</B>").arg(mystr.latin1()),
				    "Okay");

      // i18n( "What is the separator used in this file ? First line is \n%1" ).arg(firstLine),
      return false;
    }
    else return true;
}
