/*
 *  Copyright (c) 2004 Bart Coppens <kde@bartcoppens.be>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#include <tqfont.h>
#include <tqrect.h>
#include <tqimage.h>
#include <tqlayout.h>
#include <tqwidget.h>
#include <tqstring.h>
#include <tqpixmap.h>
#include <tqpainter.h>
#include <tqpushbutton.h>
#include <tqfontmetrics.h>
#include <tqhbox.h>

#include <kaction.h>
#include <kinputdialog.h>
#include <klocale.h>
#include <kfontdialog.h>
#include <ksqueezedtextlabel.h>

#include <tqcolor.h>

#include "kis_point.h"
#include "kis_image.h"
#include "kis_layer.h"
#include "kis_group_layer.h"
#include "kis_paint_layer.h"
#include "kis_cursor.h"
#include "kis_tool_text.h"
#include "kis_paint_device.h"
#include "kis_canvas_subject.h"
#include "kis_button_press_event.h"
#include "kis_button_release_event.h"
#include "kis_color.h"
#include "kis_undo_adapter.h"

KisToolText::KisToolText()
    : super(i18n("Text"))
    , m_wasPressed( false )
    , m_windowIsBeingShown( false )
{
    setName("tool_text");
    m_subject = 0;
    setCursor(KisCursor::load("tool_text_cursor.png", 6, 6));
}

KisToolText::~KisToolText()
{
}

void KisToolText::update(KisCanvasSubject *subject)
{
    m_subject = subject;
    super::update(subject);
}

void KisToolText::buttonPress(KisButtonPressEvent *e)
{
    if (m_subject && e->button() == Qt::LeftButton) {
        m_wasPressed = true;
    }
}

void KisToolText::buttonRelease(KisButtonReleaseEvent *e)
{
    if ( m_windowIsBeingShown ) return;

    if (m_subject && e->button() == Qt::LeftButton) {
        if(!m_wasPressed) return;
        m_wasPressed = false;
        KisImageSP img = m_subject->currentImg();

        m_windowIsBeingShown = true;
        bool ok;
        TQString text = KInputDialog::getText(i18n("Font Tool"), i18n("Enter text:"),
             TQString(), &ok);
        if (!ok) {
            m_windowIsBeingShown = false;
            return;
        }

        KisUndoAdapter *undoAdapter = img->undoAdapter();
        if (undoAdapter) {
            undoAdapter->beginMacro(i18n("Text"));
        }

        TQFontMetrics metrics(m_font);
        TQRect boundingRect = TQT_TQRECT_OBJECT(metrics.boundingRect(text)).normalize();
        int xB = - boundingRect.x();
        int yB = - boundingRect.y();

        if (boundingRect.x() < 0 || boundingRect.y() < 0)
            boundingRect.moveBy(- boundingRect.x(), - boundingRect.y());

        TQPixmap pixels(boundingRect.width(), boundingRect.height());
        {
            TQPainter paint(&pixels);
            paint.fillRect(boundingRect, TQt::white);
            paint.setFont(m_font);
            paint.setBrush(TQBrush(TQt::black));
            paint.drawText(xB, yB, text);
        }
        TQImage image = pixels.convertToImage();

        TQ_INT32 height = boundingRect.height();
        TQ_INT32 width = boundingRect.width();
        KisPaintLayer *layer = new KisPaintLayer(img, '"' + text + '"', OPACITY_OPAQUE);
        KisGroupLayerSP parent = img->rootLayer();
        if (img->activeLayer())
            parent = img->activeLayer()->parent();
        img->addLayer(layer, parent, img->activeLayer());
        for (int y = 0; y < height; y++) {
            for (int x = 0; x < width; x++) {
                TQRgb pixel = image.pixel(x, y);
                 // use the 'blackness' as alpha :)
                TQ_UINT8 alpha = 255 - tqRed(pixel) * OPACITY_OPAQUE / 255;
                TQColor c = m_subject->fgColor().toTQColor();
                layer->paintDevice()->setPixel(x, y, c, alpha);
            }
        }

        layer->setOpacity(m_opacity);
        layer->setCompositeOp(m_compositeOp);

        layer->setVisible(false);
        TQ_INT32 x = TQMAX(0, static_cast<int>(e->x() - width/2));
        TQ_INT32 y = TQMAX(0, static_cast<int>(e->y() - height/2));
        layer->setX(x);
        layer->setY(y);
        layer->setVisible(true);
        layer->setDirty();

        if (undoAdapter) {
            undoAdapter->endMacro();
        }

        m_windowIsBeingShown = false;
    }
}

void KisToolText::setFont() {
    KFontDialog::getFont( m_font, false/*, TQWidget* parent! */ );
    m_lbFontName->setText(TQString(m_font.family() + ", %1").arg(m_font.pointSize()));
}

TQWidget* KisToolText::createOptionWidget(TQWidget* parent)
{
    TQWidget *widget = super::createOptionWidget(parent);

    m_lbFont = new TQLabel(i18n("Font: "), widget);

    TQHBox *fontBox = new TQHBox(widget);
    m_lbFontName = new KSqueezedTextLabel(TQString(m_font.family() + ", %1")
        .arg(m_font.pointSize()), fontBox);
    m_btnMoreFonts = new TQPushButton("...", fontBox);

    connect(m_btnMoreFonts, TQT_SIGNAL(released()), this, TQT_SLOT(setFont()));

    addOptionWidgetOption(fontBox, m_lbFont);

    return widget;
}

void KisToolText::setup(KActionCollection *collection)
{
    m_action = static_cast<KRadioAction *>(collection->action(name()));

    if (m_action == 0) {
        m_action = new KRadioAction(i18n("T&ext"),
                        "tool_text",
                        TQt::SHIFT+TQt::Key_T,
                        this,
                        TQT_SLOT(activate()),
                        collection,
                        name());
        m_action->setExclusiveGroup("tools");
        m_action->setToolTip(i18n("Text"));
        m_ownAction = true;
    }
}

#include "kis_tool_text.moc"
