/***************************************************************************
 *
 * knetworkmanager-devicestore_dbus.cpp - A NetworkManager frontend for KDE
 *
 * Copyright (C) 2005, 2006 Novell, Inc.
 *
 * Author: Helmut Schaa <hschaa@suse.de>, <helmut.schaa@gmx.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 **************************************************************************/

/* qt headers */
#include <tqhostaddress.h>
#include <tqvariant.h>

/* kde headers */
#include <kdebug.h>
#include <klocale.h>

/* TQT_DBus headers*/
#include <tqdbusdata.h>
#include <tqdbusdatamap.h>

/* knetworkmanager headers */
#include "knetworkmanager.h"
#include "knetworkmanager-connection_setting_ipv4.h"


using namespace ConnectionSettings;

// reverse order the bytes
TQ_UINT32 swap32(TQ_UINT32 x)
{
	TQ_UINT32 ret = 0;

	TQ_UINT8* from = (TQ_UINT8*) &x;
	TQ_UINT8* to   = (TQ_UINT8*) &ret;

	for (int i = 0; i < 4; ++i)
		to[3-i] = from[i];
	return ret;
}

/*
	class IPv4
*/
IPv4::IPv4(Connection* conn)
	: ConnectionSetting(conn, NM_SETTING_IP4_CONFIG_SETTING_NAME)
{
	_method = METHOD_DHCP;
	_ignore_auto_dns = false;
	_ignore_auto_routes = false;
}

TQValueList<IPv4Address> IPv4::getAddresses() const
{
	return _addresses;
}

void IPv4::setAddresses(const TQValueList<IPv4Address> & adr) 
{
	_addresses = adr;
	emitValidityChanged();
}

TQValueList<TQHostAddress> IPv4::getDNS() const
{
	return _dns;
}

void IPv4::setDNS(const TQValueList<TQHostAddress>& dns)
{
	_dns = dns;
	emitValidityChanged();
}

TQStringList IPv4::getDNSSearch() const
{
	return _dns_search;
}

void IPv4::setDNSSearch(const TQStringList & dnsSearch)
{
	_dns_search = dnsSearch;
	emitValidityChanged();
}

void
IPv4::setMethod(IPV4METHOD method)
{
	_method = method;
	emitValidityChanged();
}

IPv4::IPV4METHOD
IPv4::getMethod() const
{
	return _method;
}

void
IPv4::setIgnoreAutoDNS(bool ignore)
{
	_ignore_auto_dns = ignore;
	emitValidityChanged();
}

bool
IPv4::getIgnoreAutoDNS() const
{
	return _ignore_auto_dns;
}

void
IPv4::setIgnoreAutoRoutes(bool ignore)
{
	_ignore_auto_routes = ignore;
	emitValidityChanged();
}

bool
IPv4::getIgnoreAutoRoutes() const
{
	return _ignore_auto_routes;
}

bool
IPv4::isValid() const
{
	if (_method == METHOD_MANUAL)
	{
		// only check the manual settings

		// at least one address has to be specified
		if (_addresses.empty())
			return false;

		// check every address
		for (TQValueList<IPv4Address>::ConstIterator it = _addresses.begin(); it != _addresses.end(); ++it)
		{
			if ((*it).address.isNull())
				return false;
			if ((*it).netmask.isNull())
				return false;
			// no need to check gateway as it is optional
		}

		// check DNS addresses but may be empty
		for (TQValueList<TQHostAddress>::ConstIterator it = _dns.begin(); it != _dns.end(); ++it)
		{
			if ((*it).isNull())
				return false;
		}

		// don't check DNS Search because it is optional ...
	}

	return true;
}

SettingsMap
IPv4::toMap() const
{
	SettingsMap map;

	if (_method == METHOD_DHCP)
		map.insert(NM_SETTING_IP4_CONFIG_METHOD, TQT_DBusData::fromString(NM_SETTING_IP4_CONFIG_METHOD_AUTO));
	else if (_method == METHOD_AUTOIP)
		map.insert(NM_SETTING_IP4_CONFIG_METHOD, TQT_DBusData::fromString(NM_SETTING_IP4_CONFIG_METHOD_LINK_LOCAL));
	else if (_method == METHOD_SHARED)
		map.insert(NM_SETTING_IP4_CONFIG_METHOD, TQT_DBusData::fromString(NM_SETTING_IP4_CONFIG_METHOD_SHARED));
	else if (_method == METHOD_MANUAL)
	{
		map.insert(NM_SETTING_IP4_CONFIG_METHOD, TQT_DBusData::fromString(NM_SETTING_IP4_CONFIG_METHOD_MANUAL));

		// DNS search
		if (_dns_search.size() > 0)
		{
			TQValueList<TQT_DBusData> dns_search;
			TQStringList::ConstIterator it = _dns_search.begin();
			for(;it != _dns_search.end(); ++it)
				dns_search.append(TQT_DBusData::fromString(*it));
	
			map.insert(NM_SETTING_IP4_CONFIG_DNS_SEARCH, TQT_DBusData::fromTQValueList(dns_search));
		}

		// DNS addresses
		if (_dns.size() > 0)
		{
			TQValueList<TQT_DBusData> dns;
			TQValueList<TQHostAddress>::ConstIterator it_dns = _dns.begin();
			// the strange swap32 is needed as NM reads the address exactly the other way round as TQt
			for(;it_dns != _dns.end(); ++it_dns)
				dns.append(TQT_DBusData::fromUInt32(swap32((*it_dns).toIPv4Address())));
		
			map.insert(NM_SETTING_IP4_CONFIG_DNS, TQT_DBusData::fromTQValueList(dns));
		}

		// IP's
		if (_addresses.size() > 0)
		{
			TQValueList<TQT_DBusData> ips;
			for (TQValueList<IPv4Address>::ConstIterator it = _addresses.begin(); it != _addresses.end(); ++it)
			{
				TQValueList<TQT_DBusData> cur_ip;
				cur_ip.append(TQT_DBusData::fromUInt32(swap32((*it).address.toIPv4Address())));
				cur_ip.append(TQT_DBusData::fromUInt32(toCIDRSuffix((*it).netmask)));
				if (!(*it).gateway.isNull())
					cur_ip.append(TQT_DBusData::fromUInt32(swap32((*it).gateway.toIPv4Address())));
				ips.append(TQT_DBusData::fromTQValueList(cur_ip));
			}
			map.insert(NM_SETTING_IP4_CONFIG_ADDRESSES, TQT_DBusData::fromTQValueList(ips));
		}
	}
	map.insert(NM_SETTING_IP4_CONFIG_IGNORE_AUTO_ROUTES, TQT_DBusData::fromBool(_ignore_auto_routes));
	map.insert(NM_SETTING_IP4_CONFIG_IGNORE_AUTO_DNS, TQT_DBusData::fromBool(_ignore_auto_dns));
	return map;
}

void
IPv4::fromMap(const SettingsMap& map)
{
	SettingsMap::ConstIterator it;

	if ((it = map.find(NM_SETTING_IP4_CONFIG_METHOD)) != map.end())
	{
		if (it.data().toString() == NM_SETTING_IP4_CONFIG_METHOD_AUTO || it.data().toString() == "dhcp")
			_method = METHOD_DHCP;
		else if (it.data().toString() == NM_SETTING_IP4_CONFIG_METHOD_LINK_LOCAL || it.data().toString() == "autoip")
			_method = METHOD_AUTOIP;
		else if (it.data().toString() == NM_SETTING_IP4_CONFIG_METHOD_SHARED)
			_method = METHOD_SHARED;
		else if (it.data().toString() == NM_SETTING_IP4_CONFIG_METHOD_MANUAL)
			_method = METHOD_MANUAL;
	}

	// DNS search
	if ((it = map.find(NM_SETTING_IP4_CONFIG_DNS_SEARCH)) != map.end())
	{
		TQValueList<TQT_DBusData> dns_search = it.data().toTQValueList();
		for (TQValueList<TQT_DBusData>::Iterator it = dns_search.begin(); it != dns_search.end(); ++it)
		{
			_dns_search.append( (*it).toString());
		}
	}

	// DNS addresses
	if ((it = map.find(NM_SETTING_IP4_CONFIG_DNS)) != map.end())
	{
		TQValueList<TQT_DBusData> dns = it.data().toTQValueList();
		for (TQValueList<TQT_DBusData>::Iterator it = dns.begin(); it != dns.end(); ++it)
		{
			_dns.append( TQHostAddress(swap32((*it).toUInt32())) );
		}
	}

	// IP's
	if ((it = map.find(NM_SETTING_IP4_CONFIG_ADDRESSES)) != map.end())
	{
		TQValueList<TQT_DBusData> ips = it.data().toTQValueList();
		for (TQValueList<TQT_DBusData>::Iterator it2 = ips.begin(); it2 != ips.end(); ++it2)
		{
			TQValueList<TQT_DBusData> cur_ip = (*it2).toTQValueList();
			IPv4Address address;

			address.address = swap32(cur_ip[0].toUInt32());

			if (cur_ip[1].toUInt32() >= 0 && cur_ip[1].toUInt32() <= 32)
				address.netmask = fromCIDRSuffix(cur_ip[1].toUInt32());
			else
				address.netmask = swap32(cur_ip[1].toUInt32());

			if (cur_ip.size() > 2)
				address.gateway = swap32(cur_ip[2].toUInt32());
		
			_addresses.append(address);
		}
	}

	if ((it = map.find(NM_SETTING_IP4_CONFIG_IGNORE_AUTO_ROUTES)) != map.end())
		_ignore_auto_routes = it.data().toBool();

	if ((it = map.find(NM_SETTING_IP4_CONFIG_IGNORE_AUTO_DNS)) != map.end())
		_ignore_auto_dns = it.data().toBool();
}

TQ_UINT32 IPv4::toCIDRSuffix(const TQHostAddress& adr) const
{
	TQ_UINT32 netmask = adr.toIPv4Address();
	TQ_UINT32 suffix = 0;
	while (netmask > 0)
	{
		suffix++;
		netmask = netmask << 1;
	}
	return suffix;
}

TQHostAddress IPv4::fromCIDRSuffix(TQ_UINT32 suffix)
{
	TQ_UINT32 netmask = 0xFFFFFFFF;
	netmask = netmask << (32 - suffix);
	return TQHostAddress(netmask);
}

