/***************************************************************************
 *	 Copyright (C) 2007 by Ken Werner									   *
 *	 ken.werner@web.de													   *
 *																		   *
 *	 This program is free software; you can redistribute it and/or modify  *
 *	 it under the terms of the GNU General Public License as published by  *
 *	 the Free Software Foundation; either version 2 of the License, or	   *
 *	 (at your option) any later version.								   *
 *																		   *
 *	 This program is distributed in the hope that it will be useful,	   *
 *	 but WITHOUT ANY WARRANTY; without even the implied warranty of		   *
 *	 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the		   *
 *	 GNU General Public License for more details.						   *
 *																		   *
 *	 You should have received a copy of the GNU General Public License	   *
 *	 along with this program; if not, write to the						   *
 *	 Free Software Foundation, Inc.,									   *
 *	 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.			   *
 ***************************************************************************/

#include "batterysrc.h"
#include <klocale.h>
#include <kdebug.h>

#define DBUS_DEBUG_MESSAGE(func, _dbus_err_) \
	kdDebug() << "error in " << func << ": " << _dbus_err_.name << ", " << _dbus_err_.message << endl

#define FREE_DBUS_ERROR(func, _dbus_err_) \
	do { \
		if(dbus_error_is_set(&_dbus_err_)) { \
			DBUS_DEBUG_MESSAGE(func, _dbus_err_); \
			dbus_error_free(&_dbus_err_); \
		} \
	} while(0)

BatterySrc::BatterySrc(TQWidget* inParent, TQString inUDI):
		LabelSource(inParent),
		mTrigger(this, 5000),
		mUDI(inUDI),
		mLibHalContext(NULL){
	mID = mUDI.section('/', -1);
	mName = mID;
	mDescription = i18n("This source displays the current state of charge of your battery.");

	dbus_error_init(&mDBusError);
	if((mLibHalContext = libhal_ctx_new()) == NULL){
		kdDebug() << "error: libmLibHalContext_new" << endl;
		return;
	}
	if(!libhal_ctx_set_dbus_connection(mLibHalContext, dbus_bus_get (DBUS_BUS_SYSTEM, &mDBusError))){
		FREE_DBUS_ERROR("libhal_ctx_set_dbus_connection", mDBusError);
		libhal_ctx_free(mLibHalContext);
		mLibHalContext = NULL;
		return;
	}
	if(!libhal_ctx_init(mLibHalContext, &mDBusError)){
		FREE_DBUS_ERROR("libhal_ctx_init", mDBusError);
		libhal_ctx_free(mLibHalContext);
		mLibHalContext = NULL;
		return;
	}
}

BatterySrc::~BatterySrc(){
	if(mLibHalContext != NULL){
		if(!libhal_ctx_shutdown(mLibHalContext, &mDBusError)){
			FREE_DBUS_ERROR("libhal_ctx_shutdown", mDBusError);
		}
		libhal_ctx_free(mLibHalContext);
	}
}

std::list<Source*>BatterySrc::createInstances(TQWidget* inParent){
	std::list<Source*> list;
	
	LibHalContext* libHalContext;
	DBusError dBusError;
	dbus_error_init(&dBusError);
	if((libHalContext = libhal_ctx_new()) == NULL){
		kdDebug() << "error: libmLibHalContext_new" << endl;
		return list;
	}
	if(!libhal_ctx_set_dbus_connection(libHalContext, dbus_bus_get (DBUS_BUS_SYSTEM, &dBusError))){
		FREE_DBUS_ERROR("libhal_ctx_set_dbus_connection", dBusError);
		libhal_ctx_free(libHalContext);
		return list;
	}
	if(!libhal_ctx_init(libHalContext, &dBusError)){
		FREE_DBUS_ERROR("libhal_ctx_init", dBusError);
		libhal_ctx_free(libHalContext);
		return list;
	}

	int num = 0;
	char** udis = libhal_manager_find_device_string_match(libHalContext, "info.category", "battery", &num, &dBusError);
	if(!dbus_error_is_set(&dBusError)){
		for(int i = 0; i < num; ++i){
			if(libhal_device_get_property_type(libHalContext, udis[i], "battery.charge_level.last_full", &dBusError) == LIBHAL_PROPERTY_TYPE_INT32 && libhal_device_get_property_type(libHalContext, udis[i], "battery.charge_level.current", &dBusError) == LIBHAL_PROPERTY_TYPE_INT32){
				if(!dbus_error_is_set(&dBusError)) // todo: check whether this check is redundant
					list.push_back(new BatterySrc(inParent, udis[i]));
			}
			if(dbus_error_is_set(&dBusError)) {
				DBUS_DEBUG_MESSAGE("libhal_device_get_property_type", dBusError);
				dbus_error_free(&dBusError);
			}
		}
	} else {
		DBUS_DEBUG_MESSAGE("libhal_manager_find_device_string_match", dBusError);
		dbus_error_free(&dBusError);
	}
	if(!libhal_ctx_shutdown(libHalContext, &dBusError)){
		FREE_DBUS_ERROR("libhal_ctx_shutdown", dBusError);
	}
	// clean up
	libhal_ctx_free(libHalContext);
	libhal_free_string_array(udis);
	return list;
}

TQString BatterySrc::fetchValue() {
	TQString s = "n/a";
	if(mLibHalContext != NULL){
		int last_full = 0;
		int current = 0;
	
		last_full = libhal_device_get_property_int(mLibHalContext, mUDI.ascii(), "battery.charge_level.last_full", &mDBusError);
		if(dbus_error_is_set(&mDBusError)){
			DBUS_DEBUG_MESSAGE("libhal_device_get_property_int", mDBusError);
			dbus_error_free(&mDBusError);
			return s;
		}
		current = libhal_device_get_property_int(mLibHalContext, mUDI.ascii(), "battery.charge_level.current", &mDBusError);
		if(dbus_error_is_set(&mDBusError)){
			DBUS_DEBUG_MESSAGE("libhal_device_get_property_int", mDBusError);
			dbus_error_free(&mDBusError);
			return s;
		}
		s = TQString::number(tqRound((current * 100.0) / last_full)) + " %";
	}
	return s;
}
