/***************************************************************************
    begin                : Fri Jun 4 2004
    copyright            : (C) 2004 by Jeroen Wijnout
    email                : Jeroen.Wijnhout@kdemail.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "configtester.h"

#include <tqfile.h>

#include <kio/netaccess.h>
#include <klocale.h>
#include <kprocess.h>
#include <kstandarddirs.h>
#include <ktempdir.h>
#include <ksimpleconfig.h>
#include <kglobal.h>
#include "kiledebug.h"

#include "kiletoolmanager.h"
#include "kileinfo.h"
#include "kileconfig.h"
#include "kiletool.h"

ConfigTest::ConfigTest() :
	m_name(TQString()),
	m_arg(TQString()),
	m_altArg(TQString()),
	m_mustPass(false)
{
}

ConfigTest::ConfigTest(const TQString &name, bool mustpass, const TQString &arg, const TQString &altarg /*= TQString()*/) :
	m_name(name),
	m_arg(arg),
	m_altArg(altarg),
	m_mustPass(mustpass)
{
}

int ConfigTest::status() const
{
	bool passed = false;
	if ( m_name == "binary" )
		passed = m_arg.endsWith(m_altArg);
	else if ( m_name == "version" )
		passed = true;
	else
		passed = (m_arg == "0");

	if ( passed ) return Success;
	else if ( m_mustPass ) return Critical;
	else return Failure;
}

TQString ConfigTest::name() const
{
	return prettyName(m_name);
}

TQString ConfigTest::resultText() const
{
	TQString str = successMessage(m_name);
	if ( status() == Failure ) str = failureMessage(m_name);
	else if ( status() == Critical ) str = criticalMessage(m_name);

	if ( m_name == "binary" )
	{
		str += " (" + m_altArg + " => " + m_arg + ')';
		if ( status()==Failure && s_msgFailure.contains(m_altArg) ) 
			str += TQString("<br>(%1)").arg( s_msgFailure[m_altArg] );
		return str;
	}
	else if ( m_name == "version" )
		return m_arg;
	else
		return str;
}

TQMap<TQString,TQString> ConfigTest::s_prettyName;
TQMap<TQString,TQString> ConfigTest::s_msgSuccess;
TQMap<TQString,TQString> ConfigTest::s_msgFailure;
TQMap<TQString,TQString> ConfigTest::s_msgCritical;

void ConfigTest::addPrettyName(const TQString &test, const TQString &prettyName) { s_prettyName [test] = prettyName;}
void ConfigTest::addSuccessMessage(const TQString &test, const TQString &msg) { s_msgSuccess [test]  = msg; }
void ConfigTest::addFailureMessage(const TQString &test, const TQString &msg) { s_msgFailure [test] = msg; }
void ConfigTest::addCriticalMessage(const TQString &test, const TQString &msg) { s_msgCritical [test] = msg; }

TQString ConfigTest::prettyName(const TQString &test)
{
	if ( s_prettyName.contains(test) ) return s_prettyName[test];
	else return test;
}

TQString ConfigTest::successMessage(const TQString &test)
{
	if ( s_msgSuccess.contains(test) ) return s_msgSuccess[test];
	else return i18n("Passed");
}

TQString ConfigTest::failureMessage(const TQString &test)
{
	if ( s_msgFailure.contains(test) ) return s_msgFailure[test];
	else return i18n("Failed");
}

TQString ConfigTest::criticalMessage(const TQString &test)
{
	if ( s_msgCritical.contains(test) ) return s_msgCritical[test];
	else return i18n("Critical failure");
}

Tester::Tester(TQObject *parent, const char *name) : TQObject(parent, name), m_process(0L)
{
	ConfigTest::addPrettyName("binary", i18n("Binary"));
	ConfigTest::addCriticalMessage("binary", i18n("Could not find the binary for this essential tool."));

	ConfigTest::addPrettyName("basic", i18n("Simple Test"));
	ConfigTest::addCriticalMessage("basic", i18n("This essential tool does not work at all, check your installation."));

	ConfigTest::addPrettyName("version", i18n("Version"));

	ConfigTest::addPrettyName("kile", i18n("Running in Kile"));
	TQString str = i18n("Kile is not configured correctly. Go to Settings->Configure Kile->Tools and either fix the problem or change to the default settings.");
	ConfigTest::addCriticalMessage("kile", str);
	ConfigTest::addFailureMessage("kile", str);

	ConfigTest::addPrettyName("src", i18n("Source Specials Switch"));
	ConfigTest::addSuccessMessage("src", i18n("Supported, use the 'Modern' configuration for (La)TeX and PDF(La)TeX to auto-enable inverse and forward search capabilities."));
	ConfigTest::addFailureMessage("src", i18n("Not supported, use the srcltx package to enable the inverse and forward search capabilities."));

	// add some special messages, when programs are not installed, 
	// which are not needed, but probably useful for the work with kile
	ConfigTest::addFailureMessage("dvipng", i18n("You can't use the png preview for mathgroups in the bottom bar."));
	ConfigTest::addFailureMessage("convert", i18n("You can't use the png previews with conversions 'dvi->ps->png' and 'pdf->png'."));
	ConfigTest::addFailureMessage("acroread", i18n("You can't open pdf documents with Acrobat Reader. But you could use KPDF or KGhostView."));
}


Tester::~Tester()
{
	if (m_tempDir) m_tempDir->unlink();
	delete m_tempDir;
	delete m_process;
}

void Tester::saveResults(const KURL & dest)
{
	KIO::NetAccess::file_copy(KURL::fromPathOrURL(m_resultsFile), dest, -1, true);
}

void Tester::runTests()
{
	TQString srcdir = KGlobal::dirs()->findResourceDir("appdata","test/runTests.sh") + "test";
	KILE_DEBUG() << "Tester::runTests: srcdir = " << srcdir << endl;
	m_tempDir = new KTempDir();
	TQString destdir = m_tempDir->name();
	KILE_DEBUG() << "Tester::runTests: destdir = " << destdir << endl;
	m_resultsFile = destdir + "results.rc";

	TQString shellname = KGlobal::dirs()->findExe("sh");
	KILE_DEBUG() << "Tester::runTests: shellname = " << shellname << endl;
	m_process = new KShellProcess(TQFile::encodeName( shellname ));
	if (! KileConfig::teXPaths().isEmpty())
	{
		m_process->setEnvironment("TEXINPUTS", KileInfo::expandEnvironmentVars( KileConfig::teXPaths() + ":$TEXINPUTS"));
	}
	*m_process << "cd " + KShellProcess::quote(destdir) + " && ";
	*m_process << "cp " + KShellProcess::quote(srcdir) +"/* " + KShellProcess::quote(destdir) + " && ";
	*m_process << "source runTests.sh " + KShellProcess::quote(m_resultsFile) + " " +  KShellProcess::quote(destdir);
	connect(m_process, TQT_SIGNAL(receivedStdout(KProcess *, char *, int)), this, TQT_SLOT(determineProgress(KProcess *, char *, int)));
	connect(m_process, TQT_SIGNAL(processExited(KProcess *)), this, TQT_SLOT(processTestResults(KProcess *)));
	if (m_process->start(KProcess::NotifyOnExit, KProcess::AllOutput)) emit(started());
}

void Tester::stop()
{
	if (m_process) m_process->kill();
}

void Tester::determineProgress(KProcess */*proc*/, char *buf, int len)
{
	static TQString s = TQString();

	s += TQString::fromLocal8Bit(buf, len);
	if ( s.endsWith("\n") )
	{
		bool ok = false;
		int number = s.toInt(&ok);
		if (ok) emit(percentageDone(number));
		s = TQString();
	}
}

void Tester::processTestResults (KProcess *proc)
{
	if (proc->normalExit())
	{
		emit(percentageDone(100));

		KSimpleConfig config(m_resultsFile, true);
		TQStringList groups = config.groupList();
		TQStringList::Iterator itend = groups.end();
		for ( TQStringList::Iterator it = groups.begin(); it != itend; ++it )
			processTool(&config, *it);

		emit(finished(true));
	}
	else
	{
		emit(percentageDone(0));
		emit(finished(false));
	}
}

void Tester::processTool(KConfig *config, const TQString &tool)
{
	config->setGroup(tool);

	TQStringList criticaltests = TQStringList::split(",", config->readEntry("mustpass", ""));

	//Did we find the executable?
	TQValueList<ConfigTest> tests;
	tests << ConfigTest("binary", criticaltests.contains("where"), config->readEntry("where"), config->readEntry("executable"));
	if (config->hasKey("version") ) tests << ConfigTest("version", criticaltests.contains("version"), config->readEntry("version"));
	if (config->hasKey("basic") ) tests << ConfigTest("basic", criticaltests.contains("basic"), config->readEntry("basic"));
	if (config->hasKey("src") ) tests << ConfigTest("src", criticaltests.contains("src"), config->readEntry("src"));
	if (config->hasKey("kile") ) tests << ConfigTest("kile", criticaltests.contains("kile"), config->readEntry("kile"));

	addResult(tool, tests);
}

void Tester::addResult(const TQString &tool, const TQValueList<ConfigTest> &tests)
{
	m_results [tool] = tests;
}

TQStringList Tester::testedTools()
{
	return m_results.keys();
}

TQValueList<ConfigTest> Tester::resultForTool(const TQString & tool)
{
	return m_results[tool];
}

int Tester::statusForTool(const TQString & tool)
{
	TQValueList<ConfigTest> tests = m_results[tool];
	int status = ConfigTest::Success;
	for ( uint i = 0; i < tests.count(); ++i)
	{
		if ( (tests[i].status() == ConfigTest::Failure) && (status == ConfigTest::Success)) status = ConfigTest::Failure;
		if (tests[i].status() == ConfigTest::Critical) status = ConfigTest::Critical;
	}
	return status;
}

#include "configtester.moc"
