f/***************************************************************************
                          barcode.cpp  -  description
                             -------------------
    begin                : Die Apr 23 2002
    copyright            : (C) 2002 by Dominik Seichter
    email                : domseichter@web.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "sqltables.h"
#include "barcodecache.h"

#include "mybarcode.h"
#include <stdio.h>

// TQt includes
#include <tqdir.h>
#include <tqpainter.h>
#include <tqpaintdevicemetrics.h>
#include <tqsqlquery.h>

// KDE includes
#include <kapplication.h>
#include <kconfig.h>
#include <klocale.h>
#include <kmessagebox.h>
#include <kprocess.h>
#include <kstandarddirs.h>
#include <ktempfile.h>

#define BARCODE_MARGIN 10     /* Margin added by GNU Barcode to the barcodes */

TQValueList<barcodeFormat> codes;

void init() {
    BarCode::setHaveBarcode();
}

bool barcodeData::operator==( const struct barcodeData d ) const {
    bool b = ( value == d.value && type == d.type &&
               scale == d.scale && text == d.text );

    if( BarCode::hasFeature( type, TBARCODEADV ) )
        b = ( b && tbarcode == d.tbarcode );

    if( BarCode::hasFeature( type, DATAMATRIX ) )
        b = ( b && datamatrix == d.datamatrix );

    if( BarCode::hasFeature( type, PDF417BARCODE ) )
        b = ( b && pdf417 == d.pdf417 );

    return b;
};

BarCode::BarCode( const barcodeData* data )
{
    barcode = *data;
    m_index = 0;
}

BarCode::BarCode()
{
    fillDefault( &barcode );
    m_index = 0;    
}

BarCode::~BarCode()
{
}

const TQPixmap BarCode::pixmap()
{
    if( p.isNull() )
        createBarcode( &p, KApplication::desktop() );

    if( p.isNull() ) {
        KMessageBox::error( 0, "<qt>" + i18n("Barcode not valid!") + "<br>" + barcode.value + "</qt>" );
        barcode.valid = false;
    }
        
    return p;
}

bool BarCode::createPixmap( TQPixmap* target, int resx, int resy )
{
    KTempFile* output = new KTempFile( TQString(), ".ps" );;
    output->file()->close();
    
    KTempFile* input = new KTempFile( TQString(), ".pbm" );
    input->file()->close();

    m_value = createSequence( barcode.value );

    if( BarCode::hasFeature( barcode.type, PDF417 ) ) {
        if(!createPdf417( output )) {
            cleanUp( output, input, target );
            return false;
        }
    } else if( BarCode::hasFeature( barcode.type, TBARCODE ) ) {
        if(!createTBarcode( output )) {
            cleanUp( output, input, target );
            return false;
        }
    } else { // if( BarCode::hasFeature( barcode.type, GNU_BARCODE ) ) {
        TQString flag = barcode.text ? "" : "-n";

        KShellProcess proc;
        proc << "barcode" << "-E"
             << "-b" << KShellProcess::quote( m_value ) << flag
             << "-e" << barcode.type << "-o" << output->name();

        proc.start( KProcess::Block, KProcess::NoCommunication );
        proc.resume();

        if( proc.exitStatus() ) {
            cleanUp( output, input, target );
            return false;
        }
    }

    TQFileInfo fi( output->name() );
    // if file size = 0, error in generation
    if( !fi.size() ) {
        cleanUp( output, input, target );
        return false;
    }

    TQSize s = getBoundingBox( output->name() );
    double sw = (double)s.width()/72 * resx;
    double sh = (double)s.height()/72 * resy;
    
    KShellProcess proc2;
    proc2 << "gs" << TQString("-g%1x%2").arg(int(sw*(double)barcode.scale)).arg(int(sh*(double)barcode.scale))
          << "-r" + TQString::number( resx*(double)barcode.scale ) + "x" + TQString::number( resy*(double)barcode.scale )
          << "-sDEVICE=pbmraw" << "-sOutputFile=" + input->name() << "-sNOPAUSE"
          << "-q " + output->name() << "-c showpage" << "-c quit";
    
    proc2.start( KProcess::Block, KProcess::NoCommunication );
    proc2.resume();

    if( proc2.exitStatus() ) {
        cleanUp( output, input, target );
        return false;
    }

    target->load( input->name(), "PBM" );
//    BarcodeCache::instance()->write( barcode, resx, resy, target, m_value );

    input->unlink();
    output->unlink();

    delete output;
    delete input;

    return true;
}

void BarCode::createBarcode( TQPixmap* target, TQPaintDevice* device )
{
    TQPaintDeviceMetrics pdm( device );
    int resx = pdm.logicalDpiX();
    int resy = pdm.logicalDpiY();

    TQString value = createSequence( barcode.value );
    TQPixmap* cached = 0;//BarcodeCache::instance()->read( barcode, resx, resy, value );

    // no matching barcode found in cache
    if( !cached ) {
        if( !createPixmap( target, resx, resy ) )
            return;
    } else {
        *target = *cached;
        delete cached;
    }
    
    if( BarCode::hasFeature( barcode.type, PDF417 ) ) {
        // we have to scale to the correct resolution.
        // we scale already here and not at the end,
        // so that the addMargin function does not get a scaled margin.
        TQPaintDeviceMetrics pdm( KApplication::desktop() );
        int screenresx = pdm.logicalDpiX();
        int screenresy = pdm.logicalDpiY();
    
        TQWMatrix m;
        double scalex = (resx/screenresx)*barcode.scale;
        double scaley = (resy/screenresy)*barcode.scale;
        m.scale( scalex, scaley );
        *target = target->xForm( m );
    }
    *target = cut( target, barcode.cut );
    *target = addMargin( target, barcode.margin );

    // Rotate
    TQWMatrix m;
    m.rotate( (double)barcode.rotation );
    *target = target->xForm( m );

    barcode.valid = true;
}

bool BarCode::createPdf417( KTempFile* output )
{
    KTempFile text( TQString(), ".txt" );
    TQTextStream t( text.file() );
    t << m_value;
    text.file()->close();

    // ps does not work use pbm!
    KShellProcess proc;
    proc << "pdf417_enc" << "-tps" << text.name() << output->name()
         << barcode.pdf417.row
         << barcode.pdf417.col
         << barcode.pdf417.err;
         
    proc.start( KProcess::Block, KProcess::NoCommunication );
    proc.resume();

    if( proc.exitStatus() ) {
        text.unlink();
        return false;
    }

    text.unlink();    
    return true;    
}

bool BarCode::createTBarcode( KTempFile* output )
{
    // print text
    TQString flag = barcode.text ? "" : "n";
    // escape text
    flag.append( barcode.tbarcode.escape ? " son" : " soff" );
    // autocorrection
    flag.append( barcode.tbarcode.autocorrect ? " Aon" : " Aoff" );
    // text above
    flag.append( barcode.tbarcode.above ? " a" : "" );
    
    KShellProcess proc;
    proc << "tbarcodeclient" << "-f" + output->name();
    if( !BarCode::hasFeature( barcode.type, BARCODE2D ) )
        proc << TQString( "m%1" ).arg( barcode.tbarcode.modulewidth * 1000 );

    if( BarCode::hasFeature( barcode.type, DATAMATRIX ) ) 
        proc << TQString( "Ds%1" ).arg( barcode.datamatrix.size );

    if( BarCode::hasFeature( barcode.type, PDF417BARCODE ) )
        proc << TQString( "Pr%1 Pc%2 Pe%3" ).arg( barcode.pdf417.row )
                                           .arg( barcode.pdf417.col )
                                           .arg( barcode.pdf417.err );
        
    proc << barcode.type << "tPS" << TQString("c%1").arg( barcode.tbarcode.checksum ) << flag << "d" + KShellProcess::quote(  m_value );
    proc.start( KProcess::Block, KProcess::NoCommunication );
    proc.resume();

    if( proc.exitStatus() )
        return false;

    return true;
}

const TQPixmap BarCode::printerPixmap( TQPaintDevice* device )
{
    if( pp.isNull() )
        createBarcode( &pp, device );
    
    return pp;
}

void BarCode::fillDefault( barcodeData* data )
{
    data->margin = 10;
    data->text = true;
    data->value = "1234567890";
    data->type = "code39";
    data->scale = 1.0;
    data->cut = 1.0;
    data->rotation = 0;
    data->valid = false;

    data->pdf417.row = 24;
    data->pdf417.col = 8;
    data->pdf417.err = 5;

    data->datamatrix.size = 0;

    data->tbarcode.modulewidth = 0.353;
    data->tbarcode.escape = false;
    data->tbarcode.above = false;
    data->tbarcode.autocorrect = false;
    data->tbarcode.checksum = 0;
        
    data->xml.caption = "Static";
    data->xml.x = 0;
    data->xml.y = 0;

    data->sequence.enabled = false;
    data->sequence.mode = NUM;
    data->sequence.step = 1;
    data->sequence.start = 1;
}

void BarCode::redrawBarcode()
{
    p.resize( 0, 0 );
    pp.resize( 0, 0 );
}

TQPixmap BarCode::cut( TQPixmap* pic, double cut)
{
    if( cut == 1.0 )
        return (*pic);

    TQPixmap pcut( pic->width(), int((double)pic->height() * cut) );
    pcut.fill( TQt::white ); // barcode.bg

    TQWMatrix m;
    /*
     * if text is above the barcode cut from
     * below the barcode.
     */

    // TODO: put this into one if, I am to stupid today.....
    if( BarCode::hasFeature( barcode.type, TBARCODEADV ) ) {
        if( !barcode.tbarcode.above )
            m.rotate( 180 );
    } else
        m.rotate( 180 );

    TQPainter painter( &pcut );
    painter.drawPixmap( 0, 0, pic->xForm( m ) );

    return pcut.xForm( m );
}

TQPixmap BarCode::addMargin( TQPixmap* pic, int margin )
{
    TQPixmap p;

    /* We have to handle UPC special because of the checksum character
     * which is printed on the right margin.
     * The samve goes for ISBN codes.
     * Any other formats??
     */

    bool gnubarcode = BarCode::hasFeature( barcode.type, GNU_BARCODE );
    double barm = gnubarcode ? BARCODE_MARGIN * barcode.scale : 0;

    // Add margin
    double sx = barm;
    double sy = barm;
    double sw = pic->width() - barm * 2;
    double sh = pic->height() - barm * 2;

    if( gnubarcode && (barcode.type == "upc" || barcode.type == "isbn") ) {
        sw = pic->width() - barm;

        p.resize( pic->width() + int(margin*2 - barm), pic->height() + int(margin * 2 - barm * 2) );
    } else
        p.resize( pic->width() + int(margin*2 - barm * 2), pic->height() + int(margin * 2 - barm * 2) );

    p.fill( TQt::white ); // barcode.bg
    TQPainter painter( &p );
    painter.drawPixmap( margin, margin, *pic, (int)sx, (int)sy, (int)sw, (int)sh );
    painter.end();

    return p;
}

const TQString BarCode::getMaxLength( const TQString & name )
{
    TQSqlQuery query("select uid, (length(barcode_no)) as LEN from " TABLE_BASIC
        " where encoding_type = '" + name +"' ORDER by LEN DESC LIMIT 1" );

    while( query.next() ) {
        TQSqlQuery queryuid("select barcode_no from barcode_basic where uid = '"
            + query.value( 0 ).toString() + "'" );
        while( queryuid.next() )
            if(!queryuid.value( 0 ).toString().isEmpty())
                return queryuid.value( 0 ).toString();
    }

    TQSqlQuery query1("select uid, (length(barcode_no)) as LEN from " TABLE_CUSTOMER_TEXT 
                    " where encoding_type = '" + name +"' ORDER by LEN DESC LIMIT 1" );

    while( query1.next() ) {
        TQSqlQuery queryuid("select barcode_no from customer_text where uid = '"
            + query1.value( 0 ).toString() + "'" );
        while( queryuid.next() )
            if(!queryuid.value( 0 ).toString().isEmpty())
                return queryuid.value( 0 ).toString();
    }

    return "1234567";        
}

void BarCode::cleanUp( KTempFile* file, KTempFile* file2, TQPixmap* target )
{
    target->resize( 0, 0 );

    file->unlink();
    file2->unlink();
    delete file;
    delete file2;
}

TQString BarCode::createSequence( const TQString & value )
{
    if( !barcode.sequence.enabled )
        return value;
        
    if( value.contains( '#' ) <= 0 )
        return value;

    TQString text = value;
    int pos = 0, counter = 1;

    pos = text.find("#", pos);
    pos++;
    while( text[pos] == '#' ) {
        text.remove(pos, 1);
        counter++;
    }

    pos = text.find("#", 0);
    TQString temp;

    if( barcode.sequence.mode == NUM ) {
        int v = barcode.sequence.start + m_index*barcode.sequence.step;
        temp.sprintf("%0*i", counter, v );
    } else {
        for( int i = 0; i < counter; i++ )
            temp.append( "A" );

        unsigned int z = 0;
        for( int p = temp.length(); p >= 0; p--, z++ ) {
            if( barcode.sequence.mode == ALPHA ) {
                int v = 'A' + m_index*barcode.sequence.step;
                v -= z*('Z'-'A');
            
                if( v <= 'Z' ) {
                    temp[p] = TQChar(v);                        
                    break;
                } else if( v > 'Z' )
                    v = 'Z';
                temp[p] = TQChar(v);            
            } else if( barcode.sequence.mode == ALPHANUM ) {
                qDebug("NOT IMPLEMENTED");
/*                char array[36];
                for( unsigned int i = 'A'; i <= 'Z'; i++ )
                    array[i-'A'] = i;
                for( unsigned int i = '0'; i <= '9'; i++ )
                    array['Z'-'A'+i-'0'] = i;

                int z = m_index*barcode.sequence.step;
                if( z < sizeof(array) )
                    temp[]
                int v = array[ ]*/
            }
        }
    }
    
    text.replace( pos, 1, temp);

    return text;
}

TQString BarCode::sequenceValue()
{
    return createSequence( barcode.value );
}

bool BarCode::hasFeature( const TQString & type, unsigned int feature )
{
    for( unsigned int i = 0; i < codes.count(); i++ )
        if( codes[i].name == type )
            return (codes[i].features & feature) == feature;
            
    return false;
}

TQSize BarCode::getBoundingBox( const TQString & filename )
{ 
    TQSize s(0,0);
    TQFile f( filename );
    if( !f.open( IO_ReadOnly ) )
        return s;
        
    TQString t;
    while( f.readLine( t, 1000 ) != -1 )
    {
        if( t.startsWith( "%%BoundingBox:") )
        {
            int x = 0;
            int y = 0;
            int w = 0;
            int h = 0;
            t = t.right( t.length() - 14 );
            sscanf( (const char*)t, "%d %d %d %d", &x, &y, &w, &h );
            s = TQSize( w, h );
            break;
        }
    }

    f.close();
    return s;
}

bool BarCode::m_haveGnuBarcode = false;
bool BarCode::m_havePdfEncode = false;
bool BarCode::m_haveTBarcode = false;

barcodeFormat BarCode::fillStruct( const TQString & name, const TQString & text, const int feature )
{
    barcodeFormat t;
    t.name = name;
    t.text = text;
    t.features = feature;
    return t;
}

void BarCode::setHaveBarcode()
{
    m_haveGnuBarcode = !KStandardDirs::findExe( "barcode" ).isNull();
    m_havePdfEncode = !KStandardDirs::findExe( "pdf417_enc" ).isNull();
    m_haveTBarcode = !KStandardDirs::findExe( "tbarcodeclient" ).isNull();

    if( codes.count() == 0 ) {
        if( m_haveGnuBarcode ) {
            codes.append( fillStruct( "ean", i18n("EAN (EAN 8 or EAN 13)"), GNU_BARCODE ) );
            codes.append( fillStruct( "upc", i18n("UPC (12-digit EAN; UPCA and UPCB)"), GNU_BARCODE ) );
            codes.append( fillStruct( "isbn", i18n("ISBN (still EAN13)"), GNU_BARCODE | NOCUT  ) );
            codes.append( fillStruct( "code39", i18n("Code 39"), GNU_BARCODE ) );
            codes.append( fillStruct( "code39 -c", i18n("Code 39 (no checksum)"), GNU_BARCODE ) );
            codes.append( fillStruct( "code128", i18n("Code 128 (a,b,c: autoselection)"), GNU_BARCODE ) );
            codes.append( fillStruct( "code128c", i18n("Code 128C (compact form digits)"), GNU_BARCODE ) );
            codes.append( fillStruct( "code128b", i18n("Code 128B, full printable ascii"), GNU_BARCODE ) );
            codes.append( fillStruct( "i25", i18n("interleaved 2 of 5 (only digits)"), GNU_BARCODE ) );
            codes.append( fillStruct( "i25 -c", i18n("interleaved 2 of 5 (only digits, no checksum)"), GNU_BARCODE ) );
            codes.append( fillStruct( "128raw", i18n("Raw code 128"), GNU_BARCODE ) );
            codes.append( fillStruct( "cbr", i18n("Codabar"), GNU_BARCODE ) );
            codes.append( fillStruct( "cbr -c", i18n("Codabar (no checksum)"), GNU_BARCODE ) );            
            codes.append( fillStruct( "msi", i18n("MSI"), GNU_BARCODE ) );
            codes.append( fillStruct( "pls", i18n("Plessey"), GNU_BARCODE ) );
            codes.append( fillStruct( "code93", i18n("Code 93"), GNU_BARCODE ) );
        }

        if( m_havePdfEncode ) {
            codes.append( fillStruct( "pdf417", i18n("pdf 417 2D Barcode"), BARCODE2D | PDF417 | PDF417BARCODE ) );
        }
 
        if( m_haveTBarcode ) {
            codes.append( fillStruct( "b1", "Code 11", TBARCODE | TBARCODEADV | MODULO10CHECK ) );
            codes.append( fillStruct( "b2", "Code 2 of 5 (Standard)", TBARCODE | TBARCODEADV | MODULO10CHECK ) );
            codes.append( fillStruct( "b3", "Interleaved 2 of 5 Standard", TBARCODE | TBARCODEADV ) );
            codes.append( fillStruct( "b4", "Code 2 of 5 IATA", TBARCODE | TBARCODEADV | MODULO10CHECK ) );
            codes.append( fillStruct( "b5", "Code 2 of 5 Matrix", TBARCODE | TBARCODEADV | MODULO10CHECK ) );
            codes.append( fillStruct( "b6", "Code 2 of 5 Data Logic", TBARCODE | TBARCODEADV | MODULO10CHECK ) );
            codes.append( fillStruct( "b7", "Code 2 of 5 Industrial", TBARCODE | TBARCODEADV | MODULO10CHECK ) );
            codes.append( fillStruct( "b8", "Code 3 of 9 (Code 39)", TBARCODE | TBARCODEADV | MODULOALLCHECK ) );
            codes.append( fillStruct( "b9", "Code 3 of 9 (Code 39) ASCII", TBARCODE | TBARCODEADV | MODULOALLCHECK ) );
            codes.append( fillStruct( "b10", "EAN8", TBARCODE | TBARCODEADV | EAN8CHECK | MODULO10CHECK ) );
            codes.append( fillStruct( "b11", "EAN8 - 2 digits add on", TBARCODE | NOCUT | TBARCODEADV | EAN8CHECK | MODULO10CHECK ) );
            codes.append( fillStruct( "b12", "EAN8 - 5 digits add on", TBARCODE | NOCUT | TBARCODEADV | EAN8CHECK | MODULO10CHECK ) );
            codes.append( fillStruct( "b13", "EAN13", TBARCODE | TBARCODEADV | EAN13CHECK | MODULO10CHECK ) );
            codes.append( fillStruct( "b14", "EAN13 - 2 digits add on", TBARCODE | NOCUT | TBARCODEADV | EAN13CHECK | MODULO10CHECK ) );
            codes.append( fillStruct( "b15", "EAN13 - 5 digits add on", TBARCODE | NOCUT | TBARCODEADV | EAN13CHECK | MODULO10CHECK ) );
            codes.append( fillStruct( "b16", "EAN128 (supports AIS)", TBARCODE | TBARCODEADV | MODULOALLCHECK ) );
            codes.append( fillStruct( "b17", "UPC 12 Digits", TBARCODE | TBARCODEADV | MODULOALLCHECK ) );
            codes.append( fillStruct( "b18", "CodaBar (2 width)", TBARCODE | TBARCODEADV | MODULO10CHECK ) );
            codes.append( fillStruct( "b19", "CodaBar (18 widths)", TBARCODE | TBARCODEADV ) );
            codes.append( fillStruct( "b20", "Code128", TBARCODE | TBARCODEADV ) );
            codes.append( fillStruct( "b21", "Deutsche Post Leitcode", TBARCODE | TBARCODEADV ) );
            codes.append( fillStruct( "b22", "Deutsche Post Identcode", TBARCODE | TBARCODEADV ) );
            codes.append( fillStruct( "b25", "Code 93", TBARCODE | TBARCODEADV ) );
            codes.append( fillStruct( "b26", "Identical to eBC_UPCA", TBARCODE | TBARCODEADV ) );
            codes.append( fillStruct( "b33", "UCC128 (= EAN128)", TBARCODE | TBARCODEADV ) );
            codes.append( fillStruct( "b34", "UPC A", TBARCODE | TBARCODEADV | TBARCODEADV | UPCACHECK ) );
            codes.append( fillStruct( "b35", "UPC A - 2 digit add on", TBARCODE | TBARCODEADV | UPCACHECK ) );
            codes.append( fillStruct( "b36", "UPC A - 5 digit add on", TBARCODE | TBARCODEADV | UPCACHECK ) );
            codes.append( fillStruct( "b37", "UPC E", TBARCODE | TBARCODEADV | UPCECHECK ) );
            codes.append( fillStruct( "b38", "UPC E - 2 digit add on", TBARCODE | TBARCODEADV | UPCECHECK ) );
            codes.append( fillStruct( "b39", "UPC E - 5 digit add on", TBARCODE | TBARCODEADV | UPCECHECK ) );
            codes.append( fillStruct( "b40", "PostNet ZIP (5d.)", TBARCODE | NOCUT | TBARCODEADV | POSTNETCHECK ) );
            codes.append( fillStruct( "b41", "PostNet ZIP (5d.+CD)", TBARCODE | NOCUT | TBARCODEADV | POSTNETCHECK ) );
            codes.append( fillStruct( "b42", "PostNet ZIP (8d.)", TBARCODE | NOCUT | TBARCODEADV | POSTNETCHECK ) );
            codes.append( fillStruct( "b43", "PostNet ZIP+4 (5d.+4d.+CD)", TBARCODE | NOCUT | TBARCODEADV | POSTNETCHECK ) );
            codes.append( fillStruct( "b44", "PostNet DPBC (5d.+4d.+2d.)", TBARCODE | NOCUT | TBARCODEADV | POSTNETCHECK ) );
            codes.append( fillStruct( "b45", "PostNet DPBC (5d.+4d.+2d.+CD)", TBARCODE | NOCUT | TBARCODEADV | POSTNETCHECK ) );
            codes.append( fillStruct( "b46", "Plessey Code", TBARCODE | TBARCODEADV ) );
            codes.append( fillStruct( "b47", "MSI Code", TBARCODE | TBARCODEADV | MODULO10CHECK ) );
            codes.append( fillStruct( "b50", "LOGMARS", TBARCODE | TBARCODEADV | MODULOALLCHECK ) );
            codes.append( fillStruct( "b55", "PDF417 - 2D bar code", TBARCODE | BARCODE2D | PDF417BARCODE ) );
            codes.append( fillStruct( "b56", "PDF417 Truncated - 2D bar code", TBARCODE | BARCODE2D | PDF417BARCODE ) );
            codes.append( fillStruct( "b57", "MaxiCode - 2D-bar code (Postscript only)", TBARCODE | BARCODE2D ) );
            codes.append( fillStruct( "b58", "QR-Code", TBARCODE | BARCODE2D ) );
            codes.append( fillStruct( "b59", "Code128 (CharSet A)", TBARCODE | TBARCODEADV | CODE128CHECK | MODULOALLCHECK ) );
            codes.append( fillStruct( "b60", "Code128 (CharSet B)", TBARCODE | TBARCODEADV | CODE128CHECK | MODULOALLCHECK ) );
            codes.append( fillStruct( "b61", "Code128 (CharSet C)", TBARCODE | TBARCODEADV | CODE128CHECK | MODULOALLCHECK ) );
            codes.append( fillStruct( "b62", "Code 93 Ascii", TBARCODE | TBARCODEADV | MODULOALLCHECK ) );
            codes.append( fillStruct( "b63", "Australian Post Standard Customer", TBARCODE | NOCUT | TBARCODEADV ) );
            codes.append( fillStruct( "b64", "Australian Post Customer 2", TBARCODE | NOCUT | TBARCODEADV ) );
            codes.append( fillStruct( "b65", "Australian Post Customer 3", TBARCODE | NOCUT | TBARCODEADV ) );
            codes.append( fillStruct( "b66", "Australian Post Reply Paid", TBARCODE | NOCUT | TBARCODEADV ) );
            codes.append( fillStruct( "b67", "Australian Post Routing", TBARCODE | NOCUT | TBARCODEADV ) );
            codes.append( fillStruct( "b68", "Australian Post Redirection", TBARCODE | NOCUT | TBARCODEADV ) );
            codes.append( fillStruct( "b69", "ISBN Code (=EAN13P5)", TBARCODE | TBARCODEADV ) );
            codes.append( fillStruct( "b70", "Royal Mail 4 State (RM4SCC)", TBARCODE | NOCUT | TBARCODEADV ) );
            codes.append( fillStruct( "b71", "Data Matrix", DATAMATRIX | TBARCODE | BARCODE2D | NOSCALE ) );
        }
    }
}

