/* ============================================================
 *
 * This file is a part of digiKam project
 * http://www.digikam.org
 *
 * Date        : 2004-09-30
 * Description : a plugin to add rain drop over an image
 * 
 * Copyright (C) 2004-2008 by Gilles Caulier <caulier dot gilles at gmail dot com>
 * Copyright (C) 2006-2008 by Marcel Wiesweg <marcel dot wiesweg at gmx dot de>
 * 
 * This program is free software; you can redistribute it
 * and/or modify it under the terms of the GNU General
 * Public License as published by the Free Software Foundation;
 * either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * ============================================================ */

// TQt includes. 
 
#include <tqlabel.h>
#include <tqwhatsthis.h>
#include <tqlayout.h>
#include <tqframe.h>
#include <tqimage.h>

// KDE includes.

#include <kconfig.h>
#include <klocale.h>
#include <kaboutdata.h>
#include <kiconloader.h>
#include <kapplication.h>
#include <kstandarddirs.h>
#include <knuminput.h>

// Local includes.

#include "version.h"
#include "ddebug.h"
#include "imageiface.h"
#include "imagewidget.h"
#include "raindrop.h"
#include "imageeffect_raindrop.h"
#include "imageeffect_raindrop.moc"

namespace DigikamRainDropImagesPlugin
{

ImageEffect_RainDrop::ImageEffect_RainDrop(TQWidget* parent)
                    : Digikam::ImageGuideDlg(parent, i18n("Add Raindrops to Photograph"), 
                                             "raindrops", false, true, false,
                                             Digikam::ImageGuideWidget::HVGuideMode)
{
    TQString whatsThis;

    KAboutData* about = new KAboutData("digikam",
                                       I18N_NOOP("Raindrops"), 
                                       digikam_version,
                                       I18N_NOOP("A digiKam image plugin to add raindrops to an image."),
                                       KAboutData::License_GPL,
                                       "(c) 2004-2005, Gilles Caulier\n"
                                       "(c) 2006-2008, Gilles Caulier and Marcel Wiesweg",
                                       0,
                                       "http://www.digikam.org");

    about->addAuthor("Gilles Caulier", I18N_NOOP("Author and maintainer"),
                     "caulier dot gilles at gmail dot com");

    about->addAuthor("Pieter Z. Voloshyn", I18N_NOOP("Raindrops algorithm"), 
                     "pieter dot voloshyn at gmail dot com"); 

    about->addAuthor("Marcel Wiesweg", I18N_NOOP("Developer"),
                     "marcel dot wiesweg at gmx dot de");

    setAboutData(about);

    TQWhatsThis::add( m_imagePreviewWidget, i18n("<p>This is the preview of the Raindrop effect."
                                           "<p>Note: if you have previously selected an area in the editor, "
                                           "this will be unaffected by the filter. You can use this method to "
                                           "disable the Raindrops effect on a human face, for example.") );
    
    // -------------------------------------------------------------
    
    TQWidget *gboxSettings = new TQWidget(plainPage());
    TQGridLayout* gridSettings = new TQGridLayout( gboxSettings, 5, 2, spacingHint());
                                                  
    TQLabel *label1 = new TQLabel(i18n("Drop size:"), gboxSettings);
    
    m_dropInput = new KIntNumInput(gboxSettings);
    m_dropInput->setRange(0, 200, 1, true);
    m_dropInput->setValue(80);
    TQWhatsThis::add( m_dropInput, i18n("<p>Set here the raindrops' size."));
    
    gridSettings->addMultiCellWidget(label1, 0, 0, 0, 2);
    gridSettings->addMultiCellWidget(m_dropInput, 1, 1, 0, 2);
    
    // -------------------------------------------------------------

    TQLabel *label2 = new TQLabel(i18n("Number:"), gboxSettings);
    
    m_amountInput = new KIntNumInput(gboxSettings);
    m_amountInput->setRange(1, 500, 1, true);
    m_amountInput->setValue(150);
    TQWhatsThis::add( m_amountInput, i18n("<p>This value controls the maximum number of raindrops.")); 
    
    gridSettings->addMultiCellWidget(label2, 2, 2, 0, 2);
    gridSettings->addMultiCellWidget(m_amountInput, 3, 3, 0, 2);
    
    // -------------------------------------------------------------

    TQLabel *label3 = new TQLabel(i18n("Fish eyes:"), gboxSettings);
    
    m_coeffInput = new KIntNumInput(gboxSettings);
    m_coeffInput->setRange(1, 100, 1, true);
    m_coeffInput->setValue(30);
    TQWhatsThis::add( m_coeffInput, i18n("<p>This value is the fish-eye-effect optical "
                                        "distortion coefficient."));     
    
    gridSettings->addMultiCellWidget(label3, 4, 4, 0, 2);
    gridSettings->addMultiCellWidget(m_coeffInput, 5, 5, 0, 2);
    
    setUserAreaWidget(gboxSettings);

    // -------------------------------------------------------------
    
    connect(m_dropInput, TQT_SIGNAL(valueChanged(int)),
            this, TQT_SLOT(slotTimer()));  
    
    connect(m_amountInput, TQT_SIGNAL(valueChanged(int)),
            this, TQT_SLOT(slotTimer()));  
    
    connect(m_coeffInput, TQT_SIGNAL(valueChanged(int)),
            this, TQT_SLOT(slotTimer()));  
}

ImageEffect_RainDrop::~ImageEffect_RainDrop()
{
}

void ImageEffect_RainDrop::renderingFinished()
{
    m_dropInput->setEnabled(true);
    m_amountInput->setEnabled(true);
    m_coeffInput->setEnabled(true);
}

void ImageEffect_RainDrop::readUserSettings(void)
{
    KConfig *config = kapp->config();
    config->setGroup("raindrops Tool Dialog");

    m_dropInput->blockSignals(true);
    m_amountInput->blockSignals(true);
    m_coeffInput->blockSignals(true);
    
    m_dropInput->setValue(config->readNumEntry("DropAdjustment", 80));
    m_amountInput->setValue(config->readNumEntry("AmountAdjustment", 150));
    m_coeffInput->setValue(config->readNumEntry("CoeffAdjustment", 30));
    
    m_dropInput->blockSignals(false);
    m_amountInput->blockSignals(false);
    m_coeffInput->blockSignals(false);
    
    slotEffect();
}

void ImageEffect_RainDrop::writeUserSettings(void)
{
    KConfig *config = kapp->config();
    config->setGroup("raindrops Tool Dialog");
    config->writeEntry("DropAdjustment", m_dropInput->value());
    config->writeEntry("AmountAdjustment", m_amountInput->value());
    config->writeEntry("CoeffAdjustment", m_coeffInput->value());
    config->sync();
}

void ImageEffect_RainDrop::resetValues()
{
    m_dropInput->blockSignals(true);
    m_amountInput->blockSignals(true);
    m_coeffInput->blockSignals(true);
    
    m_dropInput->setValue(80);
    m_amountInput->setValue(150);
    m_coeffInput->setValue(30);

    m_dropInput->blockSignals(false);
    m_amountInput->blockSignals(false);
    m_coeffInput->blockSignals(false);
} 

void ImageEffect_RainDrop::prepareEffect()
{
    m_dropInput->setEnabled(false);
    m_amountInput->setEnabled(false);
    m_coeffInput->setEnabled(false);
    
    int d        = m_dropInput->value();
    int a        = m_amountInput->value();
    int c        = m_coeffInput->value();

    Digikam::ImageIface* iface = m_imagePreviewWidget->imageIface();

    // Selected data from the image
    TQRect selection( iface->selectedXOrg(), iface->selectedYOrg(),
                     iface->selectedWidth(), iface->selectedHeight() );

    m_threadedFilter = dynamic_cast<Digikam::DImgThreadedFilter *>(
                       new RainDrop(iface->getOriginalImg(), this, d, a, c, &selection));
}

void ImageEffect_RainDrop::prepareFinal()
{
    m_dropInput->setEnabled(false);
    m_amountInput->setEnabled(false);
    m_coeffInput->setEnabled(false);
    
    int d       = m_dropInput->value();
    int a       = m_amountInput->value();
    int c       = m_coeffInput->value();

    Digikam::ImageIface iface(0, 0);

    // Selected data from the image
    TQRect selection( iface.selectedXOrg(), iface.selectedYOrg(),
                     iface.selectedWidth(), iface.selectedHeight() );

    m_threadedFilter = dynamic_cast<Digikam::DImgThreadedFilter *>(
                       new RainDrop(iface.getOriginalImg(), this, d, a, c, &selection));
}

void ImageEffect_RainDrop::putPreviewData(void)
{
    Digikam::ImageIface* iface = m_imagePreviewWidget->imageIface();

    Digikam::DImg imDest = m_threadedFilter->getTargetImage()
            .smoothScale(iface->previewWidth(), iface->previewHeight());
    iface->putPreviewImage(imDest.bits());

    m_imagePreviewWidget->updatePreview();
}

void ImageEffect_RainDrop::putFinalData(void)
{
    Digikam::ImageIface iface(0, 0);

    iface.putOriginalImage(i18n("RainDrop"),
                           m_threadedFilter->getTargetImage().bits());
}

}  // NameSpace DigikamRainDropImagesPlugin

