/* ============================================================
 *
 * This file is a part of digiKam project
 * http://www.digikam.org
 *
 * Date        : 2005-02-17
 * Description : a plugin to change image perspective .
 * 
 * Copyright (C) 2005-2008 by Gilles Caulier <caulier dot gilles at gmail dot com>
 * Copyright (C) 2006-2008 by Marcel Wiesweg <marcel dot wiesweg at gmx dot de>
 * 
 * This program is free software; you can redistribute it
 * and/or modify it under the terms of the GNU General
 * Public License as published by the Free Software Foundation;
 * either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * ============================================================ */

// TQt includes. 
 
#include <tqvgroupbox.h>
#include <tqlabel.h>
#include <tqspinbox.h>
#include <tqpushbutton.h>
#include <tqwhatsthis.h>
#include <tqlayout.h>
#include <tqframe.h>
#include <tqcheckbox.h>

// KDE includes.

#include <kcolorbutton.h>
#include <kcursor.h>
#include <kconfig.h>
#include <klocale.h>
#include <kaboutdata.h>
#include <khelpmenu.h>
#include <kiconloader.h>
#include <kapplication.h>
#include <kpopupmenu.h>
#include <kstandarddirs.h>
#include <kseparator.h>

// Local includes.

#include "version.h"
#include "ddebug.h"
#include "dimg.h"
#include "imageiface.h"
#include "perspectivewidget.h"
#include "imageeffect_perspective.h"
#include "imageeffect_perspective.moc"

namespace DigikamPerspectiveImagesPlugin
{

ImageEffect_Perspective::ImageEffect_Perspective(TQWidget* parent)
                       : Digikam::ImageDlgBase(parent, i18n("Adjust Photograph Perspective"), 
                                               "perspective", false, false)
{
    TQString whatsThis;
    
    // About data and help button.
    
    KAboutData* about = new KAboutData("digikam",
                                       I18N_NOOP("Perspective"), 
                                       digikam_version,
                                       I18N_NOOP("A digiKam image plugin to process image perspective adjustment."),
                                       KAboutData::License_GPL,
                                       "(c) 2005-2006, Gilles Caulier\n"
                                       "(c) 2006-2008, Gilles Caulier and Marcel Wiesweg",
                                       0,
                                       "http://www.digikam.org");

    about->addAuthor("Gilles Caulier", I18N_NOOP("Author and maintainer"),
                     "caulier dot gilles at gmail dot com");

    about->addAuthor("Marcel Wiesweg", I18N_NOOP("Developer"),
                     "marcel dot wiesweg at gmx dot de");

    setAboutData(about);
    
    // -------------------------------------------------------------
    
    TQFrame *frame = new TQFrame(plainPage());
    frame->setFrameStyle(TQFrame::Panel|TQFrame::Sunken);
    TQVBoxLayout* l  = new TQVBoxLayout(frame, 5, 0);
    m_previewWidget = new PerspectiveWidget(525, 350, frame);
    l->addWidget(m_previewWidget);
    TQWhatsThis::add( m_previewWidget, i18n("<p>This is the perspective transformation operation preview. "
                                           "You can use the mouse for dragging the corner to adjust the "
                                           "perspective transformation area."));
    setPreviewAreaWidget(frame); 
    
    // -------------------------------------------------------------

    TQString temp;
    Digikam::ImageIface iface(0, 0);

    TQWidget *gbox2          = new TQWidget(plainPage());
    TQGridLayout *gridLayout = new TQGridLayout( gbox2, 13, 2, spacingHint());

    TQLabel *label1  = new TQLabel(i18n("New width:"), gbox2);
    m_newWidthLabel = new TQLabel(temp.setNum( iface.originalWidth()) + i18n(" px"), gbox2);
    m_newWidthLabel->setAlignment( AlignBottom | AlignRight );

    TQLabel *label2   = new TQLabel(i18n("New height:"), gbox2);
    m_newHeightLabel = new TQLabel(temp.setNum( iface.originalHeight()) + i18n(" px"), gbox2);
    m_newHeightLabel->setAlignment( AlignBottom | AlignRight );
    
    gridLayout->addMultiCellWidget(label1, 0, 0, 0, 0);
    gridLayout->addMultiCellWidget(m_newWidthLabel, 0, 0, 1, 2);
    gridLayout->addMultiCellWidget(label2, 1, 1, 0, 0);
    gridLayout->addMultiCellWidget(m_newHeightLabel, 1, 1, 1, 2);
    
    // -------------------------------------------------------------
    
    KSeparator *line = new KSeparator(Horizontal, gbox2);
    
    TQLabel *angleLabel = new TQLabel(i18n("Angles (in degrees):"), gbox2);
    TQLabel *label3 = new TQLabel(i18n("  Top left:"), gbox2);
    m_topLeftAngleLabel = new TQLabel(gbox2);
    TQLabel *label4 = new TQLabel(i18n("  Top right:"), gbox2);
    m_topRightAngleLabel = new TQLabel(gbox2);
    TQLabel *label5 = new TQLabel(i18n("  Bottom left:"), gbox2);
    m_bottomLeftAngleLabel = new TQLabel(gbox2);
    TQLabel *label6 = new TQLabel(i18n("  Bottom right:"), gbox2);
    m_bottomRightAngleLabel = new TQLabel(gbox2);
    
    gridLayout->addMultiCellWidget(line, 2, 2, 0, 2);
    gridLayout->addMultiCellWidget(angleLabel, 3, 3, 0, 2);
    gridLayout->addMultiCellWidget(label3, 4, 4, 0, 0);
    gridLayout->addMultiCellWidget(m_topLeftAngleLabel, 4, 4, 1, 2);
    gridLayout->addMultiCellWidget(label4, 5, 5, 0, 0);
    gridLayout->addMultiCellWidget(m_topRightAngleLabel, 5, 5, 1, 2);
    gridLayout->addMultiCellWidget(label5, 6, 6, 0, 0);
    gridLayout->addMultiCellWidget(m_bottomLeftAngleLabel, 6, 6, 1, 2);
    gridLayout->addMultiCellWidget(label6, 7, 7, 0, 0);
    gridLayout->addMultiCellWidget(m_bottomRightAngleLabel, 7, 7, 1, 2);

    // -------------------------------------------------------------

    KSeparator *line2 = new KSeparator(Horizontal, gbox2);

    m_drawWhileMovingCheckBox = new TQCheckBox(i18n("Draw preview while moving"), gbox2);
    gridLayout->addMultiCellWidget(line2, 8, 8, 0, 2);
    gridLayout->addMultiCellWidget(m_drawWhileMovingCheckBox, 9, 9, 0, 2);

    m_drawGridCheckBox = new TQCheckBox(i18n("Draw grid"), gbox2);
    gridLayout->addMultiCellWidget(m_drawGridCheckBox, 10, 10, 0, 2);

    // -------------------------------------------------------------

    TQLabel *label7 = new TQLabel(i18n("Guide color:"), gbox2);
    m_guideColorBt = new KColorButton( TQColor( TQt::red ), gbox2 );
    TQWhatsThis::add( m_guideColorBt, i18n("<p>Set here the color used to draw guides dashed-lines."));
    gridLayout->addMultiCellWidget(label7, 11, 11, 0, 0);
    gridLayout->addMultiCellWidget(m_guideColorBt, 11, 11, 2, 2);

    TQLabel *label8 = new TQLabel(i18n("Guide width:"), gbox2);
    m_guideSize    = new TQSpinBox( 1, 5, 1, gbox2);
    TQWhatsThis::add( m_guideSize, i18n("<p>Set here the width in pixels used to draw guides dashed-lines."));
    gridLayout->addMultiCellWidget(label8, 12, 12, 0, 0);
    gridLayout->addMultiCellWidget(m_guideSize, 12, 12, 2, 2);

    gridLayout->setColStretch(1, 10);
    gridLayout->setRowStretch(13, 10);

    setUserAreaWidget(gbox2);

    // -------------------------------------------------------------

    connect(m_previewWidget, TQT_SIGNAL(signalPerspectiveChanged(TQRect, float, float, float, float)),
            this, TQT_SLOT(slotUpdateInfo(TQRect, float, float, float, float)));  

    connect(m_drawWhileMovingCheckBox, TQT_SIGNAL(toggled(bool)),
            m_previewWidget, TQT_SLOT(slotToggleDrawWhileMoving(bool)));

    connect(m_drawGridCheckBox, TQT_SIGNAL(toggled(bool)),
            m_previewWidget, TQT_SLOT(slotToggleDrawGrid(bool)));

    connect(m_guideColorBt, TQT_SIGNAL(changed(const TQColor &)),
            m_previewWidget, TQT_SLOT(slotChangeGuideColor(const TQColor &)));

    connect(m_guideSize, TQT_SIGNAL(valueChanged(int)),
            m_previewWidget, TQT_SLOT(slotChangeGuideSize(int)));
}

ImageEffect_Perspective::~ImageEffect_Perspective()
{
}

void ImageEffect_Perspective::readUserSettings(void)
{
    TQColor defaultGuideColor(TQt::red);
    KConfig *config = kapp->config();
    config->setGroup("perspective Tool Dialog");
    m_drawWhileMovingCheckBox->setChecked(config->readBoolEntry("Draw While Moving", true));
    m_drawGridCheckBox->setChecked(config->readBoolEntry("Draw Grid", false));
    m_guideColorBt->setColor(config->readColorEntry("Guide Color", &defaultGuideColor));
    m_guideSize->setValue(config->readNumEntry("Guide Width", 1));
    m_previewWidget->slotToggleDrawWhileMoving(m_drawWhileMovingCheckBox->isChecked());
    m_previewWidget->slotToggleDrawGrid(m_drawGridCheckBox->isChecked());
    m_previewWidget->slotChangeGuideColor(m_guideColorBt->color());
    m_previewWidget->slotChangeGuideSize(m_guideSize->value());
}

void ImageEffect_Perspective::writeUserSettings(void)
{
    KConfig *config = kapp->config();
    config->setGroup("perspective Tool Dialog");
    config->writeEntry("Draw While Moving", m_drawWhileMovingCheckBox->isChecked());
    config->writeEntry("Draw Grid", m_drawGridCheckBox->isChecked());
    config->writeEntry("Guide Color", m_guideColorBt->color());
    config->writeEntry("Guide Width", m_guideSize->value());
    config->sync();
}

void ImageEffect_Perspective::resetValues()
{
    m_previewWidget->reset();
}

void ImageEffect_Perspective::finalRendering()
{
    kapp->setOverrideCursor( KCursor::waitCursor() );
    m_previewWidget->applyPerspectiveAdjustment();
    accept();   
    kapp->restoreOverrideCursor();       
}

void ImageEffect_Perspective::slotUpdateInfo(TQRect newSize, float topLeftAngle, float topRightAngle,
                                             float bottomLeftAngle, float bottomRightAngle)
{
    TQString temp;
    m_newWidthLabel->setText(temp.setNum( newSize.width())   + i18n(" px") );
    m_newHeightLabel->setText(temp.setNum( newSize.height()) + i18n(" px") );
    
    m_topLeftAngleLabel->setText(temp.setNum( topLeftAngle, 'f', 1 ));
    m_topRightAngleLabel->setText(temp.setNum( topRightAngle, 'f', 1 ));
    m_bottomLeftAngleLabel->setText(temp.setNum( bottomLeftAngle, 'f', 1 ));
    m_bottomRightAngleLabel->setText(temp.setNum( bottomRightAngle, 'f', 1 ));
}

}  // NameSpace DigikamPerspectiveImagesPlugin

