/* ============================================================
 *
 * This file is a part of digiKam project
 * http://www.digikam.org
 *
 * Date        : 2004-06-26
 * Description : Drag object info container.
 * 
 * Copyright (C) 2004-2005 by Renchi Raju <renchi@pooh.tam.uiuc.edu>
 * Copyright (C) 2004-2005 by Joern Ahrens <joern.ahrens@kdemail.net>
 * Copyright (C) 2006-2008 by Gilles Caulier <caulier dot gilles at gmail dot com>
 *
 * This program is free software; you can redistribute it
 * and/or modify it under the terms of the GNU General
 * Public License as published by the Free Software Foundation;
 * either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * ============================================================ */

#ifndef DRAGOBJECTS_H
#define DRAGOBJECTS_H

// TQt includes.

#include <tqdragobject.h>
#include <tqstringlist.h>
#include <tqvaluelist.h>

// KDE includes.

#include <kurldrag.h>

// Local includes.

#include "cameratype.h"

class TQWidget;

namespace Digikam
{

/**
 * Provides a drag object for a list of KURLs with its related database IDs.
 *
 * Images can be moved through ItemDrag. It is possible to move them on
 * another application which understands KURLDrag, like konqueror, to
 * copy the images. Digikam can use the IDs, if ItemDrag is dropped
 * on digikam itself.
 * The kioURLs are internally used with the digikamalbums kioslave.
 * The "normal" KURLDrag urls are used for external drops (k3b, gimp, ...)
 */
class ItemDrag : public KURLDrag
{
public:

    ItemDrag(const KURL::List& urls, const KURL::List& kioURLs,
             const TQValueList<int>& albumIDs,
             const TQValueList<int>& imageIDs,
             TQWidget* dragSource=0, const char* name=0);

    static bool canDecode(const TQMimeSource* e);
    static bool decode(const TQMimeSource* e,
                       KURL::List &urls,
                       KURL::List &kioURLs,
                       TQValueList<int>& albumIDs,
                       TQValueList<int>& imageIDs);

protected:

    virtual const char* format(int i) const;
    virtual TQByteArray encodedData(const char* mime) const;

private:

    KURL::List      m_kioURLs;
    TQValueList<int> m_albumIDs;
    TQValueList<int> m_imageIDs;
};


/**
 * Provides a drag object for a tag
 *
 * When a tag is moved through drag'n'drop an object of this class 
 * is created.
 */
class TagDrag : public TQDragObject
{
public:

    TagDrag(int albumid, TQWidget *dragSource = 0, const char *name = 0);
    static bool     canDecode(const TQMimeSource* e);

protected:

    TQByteArray      encodedData( const char* ) const;
    const char*     format(int i) const;

private:
    int     mAlbumID;
};

/**
 * Provides a drag object for an album
 *
 * When an album is moved through drag'n'drop an object of this class 
 * is created.
 */
class AlbumDrag : public KURLDrag
{
public:
    AlbumDrag(const KURL &url, int albumid, 
              TQWidget *dragSource = 0, const char *name = 0);
    static bool     canDecode(const TQMimeSource* e);
    static bool     decode(const TQMimeSource* e, KURL::List &urls, 
                           int &albumID);
protected:

    TQByteArray      encodedData(const char*) const;
    const char*     format(int i) const;

private:
    int     mAlbumID;
};

/**
 * Provides a drag object for a list of tags
 *
 * When a tag is moved through drag'n'drop an object of this class 
 * is created.
 */
class TagListDrag : public TQDragObject
{
public:

    TagListDrag(const TQValueList<int>& tagIDs, TQWidget *dragSource = 0,
                const char *name = 0);
    static bool canDecode(const TQMimeSource* e);

protected:

    TQByteArray  encodedData( const char* ) const;
    const char* format(int i) const;

private:

    TQValueList<int> m_tagIDs;
};

/**
 * Provides a drag object for a list of camera items
 *
 * When a camera item is moved through drag'n'drop an object of this class
 * is created.
 */
class CameraItemListDrag : public TQDragObject
{
public:

    CameraItemListDrag(const TQStringList& cameraItemPaths, TQWidget *dragSource = 0,
                       const char *name = 0);
    static bool canDecode(const TQMimeSource* e);

protected:

    TQByteArray  encodedData( const char* ) const;
    const char* format(int i) const;

private:

    TQStringList m_cameraItemPaths;
};

/**
 * Provides a drag object for a camera object
 *
 * When a camera object is moved through drag'n'drop an object of this class
 * is created.
 */
class CameraDragObject : public TQStoredDrag
{

public:

    CameraDragObject(const CameraType& ctype, TQWidget* dragSource=0);
    ~CameraDragObject();

    static bool canDecode(const TQMimeSource* e);
    static bool decode(const TQMimeSource* e, CameraType& ctype);

private:

    void setCameraType(const CameraType& ctype);

};

}  // namespace Digikam

#endif /* DRAGOBJECTS_H */
