/***************************************************************************
 *   Copyright (C) 2003 by S�bastien Lao�t                                 *
 *   slaout@linux62.org                                                    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.             *
 ***************************************************************************/

#include <tqlayout.h>
#include <tqlineedit.h>
#include <tqlabel.h>
#include <tqsizegrip.h>
#include <tqpushbutton.h>
#include <tqstring.h>
#include <tqsizepolicy.h>
#include <kopenwith.h>
#include <klocale.h>
#include <tqwhatsthis.h>
#include <kiconview.h>
#include <kiconloader.h>
#include <tqdragobject.h>
#include <tqfontdatabase.h>

#include "variouswidgets.h"

/** class RunCommandRequester: */

RunCommandRequester::RunCommandRequester(const TQString &runCommand, const TQString &message, TQWidget *parent, const char *name)
 : TQWidget(parent, name)
{
	m_message = message;

	TQHBoxLayout *layout = new TQHBoxLayout(this, /*margin=*/0, KDialogBase::spacingHint());
	m_runCommand        = new TQLineEdit(runCommand, this);
	TQPushButton *pb     = new TQPushButton(/*"C&hoose..."*/i18n("..."), this);

	pb->setSizePolicy(TQSizePolicy::Fixed, TQSizePolicy::Fixed);

	layout->addWidget(m_runCommand);
	layout->addWidget(pb);

	connect( pb, TQT_SIGNAL(clicked()), this, TQT_SLOT(slotSelCommand()) );
}

RunCommandRequester::~RunCommandRequester()
{
}

void RunCommandRequester::slotSelCommand()
{
	KOpenWithDlg *dlg =  new KOpenWithDlg(KURL::List(), m_message, m_runCommand->text(), this);
	dlg->exec();
	if ( ! dlg->text().isEmpty() )
		m_runCommand->setText(dlg->text());
}

TQString RunCommandRequester::runCommand()
{
	return m_runCommand->text();
}

void RunCommandRequester::setRunCommand(const TQString &runCommand)
{
	m_runCommand->setText(runCommand);
}

/** class IconSizeCombo: */

IconSizeCombo::IconSizeCombo(bool rw, TQWidget *parent, const char *name)
 : TQComboBox(rw, parent, name)
{
	insertItem(i18n("16 by 16 pixels"));
	insertItem(i18n("22 by 22 pixels"));
	insertItem(i18n("32 by 32 pixels"));
	insertItem(i18n("48 by 48 pixels"));
	insertItem(i18n("64 by 64 pixels"));
	insertItem(i18n("128 by 128 pixels"));
	setCurrentItem(2);
}

IconSizeCombo::~IconSizeCombo()
{
}

int IconSizeCombo::iconSize()
{
	switch (currentItem()) {
		default:
		case 0: return 16;
		case 1: return 22;
		case 2: return 32;
		case 3: return 48;
		case 4: return 64;
		case 5: return 128;
	}
}

void IconSizeCombo::setSize(int size)
{
	switch (size) {
		default:
		case 16:  setCurrentItem(0); break;
		case 22:  setCurrentItem(1); break;
		case 32:  setCurrentItem(2); break;
		case 48:  setCurrentItem(3); break;
		case 64:  setCurrentItem(4); break;
		case 128: setCurrentItem(5); break;
	}
}

/** class ViewSizeDialog: */

ViewSizeDialog::ViewSizeDialog(TQWidget *parent, int w, int h)
 : TQDialog(parent, "ViewSizeDialog")
{
	TQLabel *label = new TQLabel(i18n(
		"Resize the window to select the image size\n"
		"and close it or press Escape to accept changes."), this);
	label->move(8, 8);
	label->setFixedSize( label->sizeHint() );

	// setSizeGripEnabled(true) doesn't work (the grip stay at the same place), so we emulate it:
	m_sizeGrip = new TQSizeGrip(this);
	m_sizeGrip->setFixedSize( m_sizeGrip->sizeHint() );

	setGeometry(x(), y(), w, h);
}

ViewSizeDialog::~ViewSizeDialog()
{
}

void ViewSizeDialog::resizeEvent(TQResizeEvent *)
{
	setCaption( i18n("%1 by %2 pixels").arg(TQString::number(width())).arg(TQString::number(height())) );
	m_sizeGrip->move( width() - m_sizeGrip->width(), height() - m_sizeGrip->height() );
}

/** class HelpLabel: */

HelpLabel::HelpLabel(const TQString &text, const TQString &message, TQWidget *parent)
 : KURLLabel(parent), m_message(message)
{
	setText(text);
	connect( this, TQT_SIGNAL(leftClickedURL()), this, TQT_SLOT(showMessage()) );
}

HelpLabel::~HelpLabel()
{
}

void HelpLabel::showMessage()
{
	TQWhatsThis::display(m_message, mapToGlobal( TQPoint(width() / 2, height()) ));
}

void HelpLabel::keyPressEvent(TQKeyEvent *event)
{
	if (event->key() == TQt::Key_Space)
		showMessage();
	else
		KURLLabel::keyPressEvent(event);
}

/** class IconSizeDialog: */

class UndraggableKIconView : public KIconView
{
  public:
	UndraggableKIconView(TQWidget * parent = 0, const char * name = 0, WFlags f = 0) : KIconView(parent, name, f) {}
	TQDragObject* dragObject() { return 0; }
};

IconSizeDialog::IconSizeDialog(const TQString &caption, const TQString &message, const TQString &icon, int iconSize, TQWidget *parent)
 : KDialogBase(KDialogBase::Swallow, caption, KDialogBase::Ok | KDialogBase::Cancel,
               KDialogBase::Ok, parent, /*name=*/0, /*modal=*/true, /*separator=*/false)
{
	TQWidget *page = new TQWidget(this);
	TQVBoxLayout *topLayout = new TQVBoxLayout(page, /*margin=*/0, spacingHint());

	TQLabel *label = new TQLabel(message, page);
	topLayout->addWidget(label);

	KIconView *iconView = new UndraggableKIconView(page);
	iconView->setItemsMovable(false);
	iconView->setSelectionMode(KIconView::Single);
	m_size16  = new KIconViewItem(iconView, 0,        i18n("16 by 16 pixels"),   DesktopIcon(icon, 16));
	m_size22  = new KIconViewItem(iconView, m_size16, i18n("22 by 22 pixels"),   DesktopIcon(icon, 22));
	m_size32  = new KIconViewItem(iconView, m_size22, i18n("32 by 32 pixels"),   DesktopIcon(icon, 32));
	m_size48  = new KIconViewItem(iconView, m_size32, i18n("48 by 48 pixels"),   DesktopIcon(icon, 48));
	m_size64  = new KIconViewItem(iconView, m_size48, i18n("64 by 64 pixels"),   DesktopIcon(icon, 64));
	m_size128 = new KIconViewItem(iconView, m_size64, i18n("128 by 128 pixels"), DesktopIcon(icon, 128));
	iconView->setMinimumWidth(m_size16->width() + m_size22->width() + m_size32->width() + m_size48->width() + m_size64->width() + m_size128->width() +
	                          (6+2) * iconView->spacing() + 20);
	iconView->setMinimumHeight(m_size128->height() + 2 * iconView->spacing() + 20);
	topLayout->addWidget(iconView);
	switch (iconSize) {
		case 16:  iconView->setSelected(m_size16,  true); m_iconSize = 16;  break;
		case 22:  iconView->setSelected(m_size22,  true); m_iconSize = 22;  break;
		default:
		case 32:  iconView->setSelected(m_size32,  true); m_iconSize = 32;  break;
		case 48:  iconView->setSelected(m_size48,  true); m_iconSize = 48;  break;
		case 64:  iconView->setSelected(m_size64,  true); m_iconSize = 64;  break;
		case 128: iconView->setSelected(m_size128, true); m_iconSize = 128; break;
	}

	connect( iconView, TQT_SIGNAL(executed(TQIconViewItem*)),      this, TQT_SLOT(choose(TQIconViewItem*)) );
	connect( iconView, TQT_SIGNAL(returnPressed(TQIconViewItem*)), this, TQT_SLOT(choose(TQIconViewItem*)) );
	connect( iconView, TQT_SIGNAL(selectionChanged()),            this, TQT_SLOT(slotSelectionChanged()) );

	setMainWidget(page);
}

IconSizeDialog::~IconSizeDialog()
{
}

void IconSizeDialog::slotSelectionChanged()
{
	// Change m_iconSize to the new selected one:
	if (m_size16->isSelected())  { m_iconSize = 16;  return; }
	if (m_size22->isSelected())  { m_iconSize = 22;  return; }
	if (m_size32->isSelected())  { m_iconSize = 32;  return; }
	if (m_size48->isSelected())  { m_iconSize = 48;  return; }
	if (m_size64->isSelected())  { m_iconSize = 64;  return; }
	if (m_size128->isSelected()) { m_iconSize = 128; return; }

	// But if user unselected the item (by eg. right clicking a free space), reselect the last one:
	switch (m_iconSize) {
		case 16:  m_size16->setSelected(true);  m_iconSize = 16;  break;
		case 22:  m_size22->setSelected(true);  m_iconSize = 22;  break;
		default:
		case 32:  m_size32->setSelected(true);  m_iconSize = 32;  break;
		case 48:  m_size48->setSelected(true);  m_iconSize = 48;  break;
		case 64:  m_size64->setSelected(true);  m_iconSize = 64;  break;
		case 128: m_size128->setSelected(true); m_iconSize = 128; break;
	}
}

void IconSizeDialog::choose(TQIconViewItem*)
{
	actionButton(KDialogBase::Ok)->animateClick();
}

void IconSizeDialog::slotCancel()
{
	m_iconSize = -1;
	KDialogBase::slotCancel();
}

/** class FontSizeCombo: */

FontSizeCombo::FontSizeCombo(bool rw, bool withDefault, TQWidget *parent, const char *name)
 : KComboBox(rw, parent, name), m_withDefault(withDefault)
{
	if (m_withDefault)
		insertItem(i18n("(Default)"));

	TQFontDatabase fontDB;
	TQValueList<int> sizes = fontDB.standardSizes();
	for (TQValueList<int>::Iterator it = sizes.begin(); it != sizes.end(); ++it)
		insertItem(TQString::number(*it));

//	connect( this, TQT_SIGNAL(acivated(const TQString&)), this, TQT_SLOT(textChangedInCombo(const TQString&)) );
	connect( this, TQT_SIGNAL(textChanged(const TQString&)), this, TQT_SLOT(textChangedInCombo(const TQString&)) );

	// TODO: 01617 void KFontSizeAction::setFontSize( int size )
}

FontSizeCombo::~FontSizeCombo()
{
}

void FontSizeCombo::textChangedInCombo(const TQString &text)
{
	bool ok = false;
	int size = text.toInt(&ok);
	if (ok)
		emit sizeChanged(size);
}

void FontSizeCombo::keyPressEvent(TQKeyEvent *event)
{
	if (event->key() == TQt::Key_Escape)
		emit escapePressed();
	else if (event->key() == TQt::Key_Return)
		emit returnPressed2();
	else
		KComboBox::keyPressEvent(event);
}

void FontSizeCombo::setFontSize(int size)
{
	setCurrentText(TQString::number(size));

	// TODO: SEE KFontSizeAction::setFontSize( int size ) !!! for a more complete method!
}

int FontSizeCombo::fontSize()
{
	bool ok = false;
	int size = currentText().toInt(&ok);
	if (ok)
		return size;

	size = text(currentItem()).toInt(&ok);
	if (ok)
		return size;

	return font().pointSize();
}

#include "variouswidgets.moc"
