/***************************************************************************
 *   Copyright (C) 2003 by S�astien Laot                                 *
 *   slaout@linux62.org                                                    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.             *
 ***************************************************************************/

#ifndef NOTEEDIT_H
#define NOTEEDIT_H

#include <kdialogbase.h>
#include <tqtextedit.h>
#include <tqlineedit.h>

class TQWidget;
//class TQLineEdit;
class TQPushButton;
class KIconButton;
class KURLRequester;
class KTextEdit;
class KMainWindow;
class KTooolBar;
class KToggleAction;

class FontSizeCombo;

class Note;
class RunCommandRequester;
class FocusedFontCombo;
class FocusedColorCombo;

#include "notecontent.h"

/** The base class for every note editors.
  * Scenario:
  *  The Basket class calls NoteEditor::editNoteContent() with the NoteContent to edit.
  *  This method create the good child NoteEditor depending
  *  on the note content type and return it to the Basket.
  *  This custom NoteEditor have two choices regarding what to do in its constructor:
  *   - Display a dialog and then call cancel() if the user canceled the dialog;
  *   - Create an inline editor and call setInlineEditor() with that editor as parameter.
  *     When the user exit the edition, validate() is called by the Basket.
  *     You should then call setEmpty() is the user cleared the content.
  *  The custom editor SHOULD call the NoteEditor constructor.
  *  If the user cleared the content OR if the user canceled the dialog whereas he/she
  *  JUST ADDED the note, then the note will be deleted by the Basket.
  */
class NoteEditor : public TQObject
{
  Q_OBJECT
  TQ_OBJECT
  public:
	NoteEditor(NoteContent *noteContent);
	bool        isEmpty()  { return m_isEmpty;      }
	bool        canceled() { return m_canceled;     }
	bool        isInline() { return m_widget != 0;  }
	TQWidget*    widget()   { return m_widget;       }
	KTextEdit*  textEdit() { return m_textEdit;     }
	TQLineEdit*  lineEdit() { return m_lineEdit;     }

  private:
	bool         m_isEmpty;
	bool         m_canceled;
	TQWidget     *m_widget;
	KTextEdit   *m_textEdit;
	TQLineEdit   *m_lineEdit;
	NoteContent *m_noteContent;

  public:
	NoteContent* noteContent() { return m_noteContent; }
	Note*        note();
  protected:
	void setEmpty() { m_isEmpty  = true; }
	void cancel()   { m_canceled = true; }
	void setInlineEditor(TQWidget *inlineEditor);
  public:
	virtual void validate() {}
	virtual void autoSave(bool /*toFileToo*/) {} // Same as validate(), but does not precede editor close and is triggered either while the editor widget changed size or after 3 seconds of inactivity.

  signals:
	void askValidation();
	void mouseEnteredEditorWidget();

  public:
	static NoteEditor* editNoteContent(NoteContent *noteContent, TQWidget *parent);
};

class TextEditor : public NoteEditor
{
  Q_OBJECT
  TQ_OBJECT
  public:
	TextEditor(TextContent *textContent, TQWidget *parent);
	~TextEditor();
	void validate();
	void autoSave(bool toFileToo);
  protected:
	TextContent *m_textContent;
};

class HtmlEditor : public NoteEditor
{
  Q_OBJECT
  TQ_OBJECT
  public:
	HtmlEditor(HtmlContent *htmlContent, TQWidget *parent);
	~HtmlEditor();
	void validate();
	void autoSave(bool toFileToo);
  protected:
	HtmlContent *m_htmlContent;
  public slots:
	void cursorPositionChanged();
	void textChanged();
	void fontChanged(const TQFont &font);
  protected slots:
//	void slotVerticalAlignmentChanged(TQTextEdit::VerticalAlignment align);
	//	void setBold();
	//	void setItalic();
	//	void setUnderline();
	void setLeft();
	void setCentered();
	void setRight();
	void setBlock();
};

class ImageEditor : public NoteEditor
{
  Q_OBJECT
  TQ_OBJECT
  public:
	ImageEditor(ImageContent *imageContent, TQWidget *parent);
};

class AnimationEditor : public NoteEditor
{
  Q_OBJECT
  TQ_OBJECT
  public:
	AnimationEditor(AnimationContent *animationContent, TQWidget *parent);
};

class FileEditor : public NoteEditor
{
  Q_OBJECT
  TQ_OBJECT
  public:
	FileEditor(FileContent *fileContent, TQWidget *parent);
	~FileEditor();
	void validate();
	void autoSave(bool toFileToo);
  protected:
	FileContent *m_fileContent;
};

class LinkEditor : public NoteEditor
{
  Q_OBJECT
  TQ_OBJECT
  public:
	LinkEditor(LinkContent *linkContent, TQWidget *parent);
};

class LauncherEditor : public NoteEditor
{
  Q_OBJECT
  TQ_OBJECT
  public:
	LauncherEditor(LauncherContent *launcherContent, TQWidget *parent);
};

class ColorEditor : public NoteEditor
{
  Q_OBJECT
  TQ_OBJECT
  public:
	ColorEditor(ColorContent *colorContent, TQWidget *parent);
};

class UnknownEditor : public NoteEditor
{
  Q_OBJECT
  TQ_OBJECT
  public:
	  UnknownEditor(UnknownContent *unknownContent, TQWidget *parent);
};

/** TQLineEdit behavior:
  * Create a new TQLineEdit with a text, then the user select a part of it and press ONE letter key.
  * The signal textChanged() is not emitted!
  * This class correct that!
  */
class DebuggedLineEdit : public TQLineEdit
{
  Q_OBJECT
  TQ_OBJECT
  public:
	DebuggedLineEdit(const TQString &text, TQWidget *parent = 0);
	~DebuggedLineEdit();
  protected:
	void keyPressEvent(TQKeyEvent *event);
};

/** The dialog to edit Link Note content.
  * @author S�astien Laot
  */
class LinkEditDialog : public KDialogBase
{
  Q_OBJECT
  TQ_OBJECT
  public:
	LinkEditDialog(LinkContent *contentNote, TQWidget *parent = 0);
	~LinkEditDialog();
	void polish();
  protected slots:
	void slotOk();
	void urlChanged(const TQString&);
	void doNotAutoTitle(const TQString&);
	void doNotAutoIcon(TQString);
	void guessTitle();
	void guessIcon();
  private:
	LinkContent   *m_noteContent;
	bool           m_isAutoModified;
	KURLRequester *m_url;
	TQLineEdit     *m_title;
	KIconButton   *m_icon;
	TQPushButton   *m_autoTitle;
	TQPushButton   *m_autoIcon;
};


/** The dialog to edit Launcher Note content.
  * @author S�astien Laot
  */
class LauncherEditDialog : public KDialogBase
{
  Q_OBJECT
  TQ_OBJECT
  public:
	LauncherEditDialog(LauncherContent *contentNote, TQWidget *parent = 0);
	~LauncherEditDialog();
	void polish();
  protected slots:
	void slotOk();
	void guessIcon();
  private:
	LauncherContent     *m_noteContent;
	RunCommandRequester *m_command;
	TQLineEdit           *m_name;
	KIconButton         *m_icon;
};

/** This class manage toolbars for the inline editors.
  * The toolbars should be created once at the application startup,
  * then KXMLGUI can manage them and save theire state and position...
  * @author S�astien Laot
  */
class InlineEditors : public TQObject
{
  Q_OBJECT
  TQ_OBJECT
  public:
	InlineEditors();
	~InlineEditors();
	void initToolBars(KActionCollection *actionCollection);
	static InlineEditors* instance();

  public:
	// Rich Text ToolBar:
	KToolBar* richTextToolBar();
	void enableRichTextToolBar();
	void disableRichTextToolBar();
	FocusedFontCombo  *richTextFont;
	FontSizeCombo     *richTextFontSize;
	FocusedColorCombo *richTextColor;
	KToggleAction     *richTextBold;
	KToggleAction     *richTextItalic;
	KToggleAction     *richTextUnderline;
//	KToggleAction     *richTextSuper;
//	KToggleAction     *richTextSub;
	KToggleAction     *richTextLeft;
	KToggleAction     *richTextCenter;
	KToggleAction     *richTextRight;
	KToggleAction     *richTextJustified;
	KAction	  *richTextUndo;
	KAction	  *richTextRedo;
};

#endif // NOTEEDIT_H
