/***************************************************************************
 *   Copyright (C) 2003 by Sbastien Laot                                 *
 *   slaout@linux62.org                                                    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.             *
 ***************************************************************************/

#ifndef NOTEDRAG_H
#define NOTEDRAG_H

#include <tqstring.h>
#include <tqdragobject.h>
#include <tqdatastream.h>
#include <tqpixmap.h>
#include <tqvaluelist.h>
#include <kmultipledrag.h>

class TQDataStream;

class Basket;
class Note;
class NoteSelection;

/** Dragging/Copying/Cutting Scenario:
  * - User select some notes and cut them;
  * - NoteDrag::toMultipleDrag() is called with a tree of the selected notes (see Basket::toSelectionTree()):
  *   - This method create a new KMultipleDrag object, create a stream,
  *   - And then browse all notes and call the virtual Note::serialize() with the stream as parameter for them to serialize theire content in the "native format".
  *   - This give the MIME type "application/x-basket-note" that will be used by the application to paste the notes exactly as they were.
  *   - Then the method try to set alterante formats for the dragged objects:
  *   - It call successively toText() for each notes and stack up the result so theire is ONE big text flavour to add to the KMultipleDrag object
  *   - It do the same with toHtml(), toImage() and toLink() to have those flavours as well, if possible...
  *   - If there is only ONE copied note, addAlternateDragObjects() is called on it, so that Unknown objects can be dragged "as is".
  *   - It's OK for the flavours. The method finally set the drag feedback pixmap by asking every selected notes to draw the content to a small pixmap.
  *   - The pixmaps are joined to one big pixmap (but it should not exceed a defined size) and a border is drawn on this image.
  *
  * Pasting/Dropping Scenario:
  *
  * @author Sbastien Laot
  */
class NoteDrag
{
  protected:
	static void serializeNotes(     NoteSelection *noteList, TQDataStream &stream,         bool cutting );
	static void serializeText(      NoteSelection *noteList, KMultipleDrag *multipleDrag               );
	static void serializeHtml(      NoteSelection *noteList, KMultipleDrag *multipleDrag               );
	static void serializeImage(     NoteSelection *noteList, KMultipleDrag *multipleDrag               );
	static void serializeLinks(     NoteSelection *noteList, KMultipleDrag *multipleDrag, bool cutting );
	static void setFeedbackPixmap(  NoteSelection *noteList, KMultipleDrag *multipleDrag               );
	static Note* decodeHierarchy(TQDataStream &stream, Basket *parent, bool moveFiles, bool moveNotes, Basket *originalBasket);
  public:
	static TQPixmap feedbackPixmap(NoteSelection *noteList);
	static TQDragObject* dragObject(NoteSelection *noteList, bool cutting, TQWidget *source = 0);
	static bool canDecode(TQMimeSource *source);
	static Note* decode(TQMimeSource *source, Basket *parent, bool moveFiles, bool moveNotes);
	static Basket* basketOf(TQMimeSource *source);
	static TQValueList<Note*> notesOf(TQMimeSource *source);
	static void createAndEmptyCuttingTmpFolder();

	static const char *NOTE_MIME_STRING;
};

/** TQTextDrag with capabilities to drop GNOME and Mozilla texts
  * as well as UTF-16 texts even if it was supposed to be encoded
  * with local encoding!
  * @author Sbastien Laot
  */
class ExtendedTextDrag : public TQTextDrag
{
  Q_OBJECT
  TQ_OBJECT
  public:
	static bool decode(const TQMimeSource *e, TQString &str);
	static bool decode(const TQMimeSource *e, TQString &str, TQCString &subtype);
};

// Support KDE 3.3 and older PROTECTED KURLDrag::encodedData()!

#include <kurldrag.h>
class KURLDrag2 : public KURLDrag
{
  Q_OBJECT
  TQ_OBJECT
  public:
	KURLDrag2(const KURL::List &urls) : KURLDrag(urls) {}
	TQByteArray encodedData2(const char *mime) const
	{
		return encodedData(mime);
	}
};

#endif // NOTEDRAG_H
