/***************************************************************************
 *   Copyright (C) 2003 by Sébastien Laoût                                 *
 *   slaout@linux62.org                                                    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.             *
 ***************************************************************************/

#ifndef NOTECONTENT_H
#define NOTECONTENT_H

#include <tqobject.h>
#include <tqstring.h>
#include <tqsimplerichtext.h>
#include <tqpixmap.h>
#include <tqmovie.h>
#include <tqcolor.h>
#include <kurl.h>

#include "linklabel.h"

class TQDomDocument;
class TQDomElement;
class TQPainter;
class TQWidget;
class TQPoint;
class TQRect;
class TQStringList;
class KMultipleDrag;

class KFileItem;
namespace KIO { class PreviewJob; }

class Note;
class Basket;
class FilterData;
class HtmlExporter;

/** A list of numeric identifier for each note type.
  * Declare a varible with the type NoteType::Id and assign a value like NoteType::Text...
  * @author S�astien Laot
  */
namespace NoteType
{
	enum Id { Group = 255, Text = 1, Html, Image, Animation, Sound, File, Link, Launcher, Color, Unknown }; // Always positive
}

/** Abstract base class for every content type of basket note.
 * It's a base class to represent those types: Text, Html, Image, Animation, Sound, File, Link, Launcher, Color, Unknown.
 * @author S�astien Laot
 */
class NoteContent // TODO: Mark some methods as const!             and some (like typeName() as static!
{
  public:
	// Constructor and destructor:
	NoteContent(Note *parent, const TQString &fileName = "");              /// << Constructor. Inherited notes should call it to initialize the parent note.
	virtual ~NoteContent()                                             {} /// << Virtual destructor. Reimplement it if you should destroy some data your custom types.
	// Simple Abstract Generic Methods:
	virtual NoteType::Id type()                                      = 0; /// << @return the internal number that identify that note type.
	virtual TQString typeName()                                       = 0; /// << @return the translated type name to display in the user interface.
	virtual TQString lowerTypeName()                                  = 0; /// << @return the type name in lowercase without space, for eg. saving.
	virtual TQString toText(const TQString &cuttedFullPath);                /// << @return a plain text equivalent of the content.
	virtual TQString toHtml(const TQString &imageName, const TQString &cuttedFullPath) = 0; /// << @return an HTML text equivalent of the content. @param imageName Save image in this TQt ressource.
	virtual TQPixmap toPixmap()                      { return TQPixmap(); } /// << @return an image equivalent of the content.
	virtual void    toLink(KURL *url, TQString *title, const TQString &cuttedFullPath); /// << Set the link to the content. By default, it set them to fullPath() if useFile().
	virtual bool    useFile()                                        = 0; /// << @return true if it use a file to store the content.
	virtual bool    canBeSavedAs()                                   = 0; /// << @return true if the content can be saved as a file by the user.
	virtual TQString saveAsFilters()                                  = 0; /// << @return the filters for the user to choose a file destination to save the note as.
	virtual bool    match(const FilterData &data)                    = 0; /// << @return true if the content match the filter criterias.
	// Complexe Abstract Generic Methods:
	virtual void exportToHTML(HTMLExporter *exporter, int indent)    = 0; /// << Export the note in an HTML file.
	virtual TQString cssClass()                                       = 0; /// << @return the CSS class of the note when exported to HTML
	virtual int     setWidthAndGetHeight(int width)                  = 0; /// << Relayout content with @p width (never less than minWidth()). @return its new height.
	virtual void    paint(TQPainter *painter, int width, int height, const TQColorGroup &colorGroup, bool isDefaultColor, bool isSelected, bool isHovered) = 0; /// << Paint the content on @p painter, at coordinate (0, 0) and with the size (@p width, @p height).
	virtual bool    loadFromFile(bool /*lazyLoad*/)     { return false; } /// << Load the content from the file. The default implementation does nothing. @see fileName().
	virtual bool    finishLazyLoad()                    { return false; } /// << Load what was not loaded by loadFromFile() if it was lazy-loaded
	virtual bool    saveToFile()                        { return false; } /// << Save the content to the file. The default implementation does nothing. @see fileName().
	virtual TQString linkAt(const TQPoint &/*pos*/)          { return ""; } /// << @return the link anchor at position @p pos or "" if there is no link.
	virtual void    saveToNode(TQDomDocument &doc, TQDomElement &content);  /// << Save the note in the basket XML file. By default it store the filename if a file is used.
	virtual void    fontChanged()                                    = 0; /// << If your content display textual data, called when the font have changed (from tags or basket font)
	virtual void    linkLookChanged()                                  {} /// << If your content use LinkDisplay with preview enabled, reload the preview (can have changed size)
	virtual TQString editToolTipText()                                = 0; /// << @return "Edit this [text|image|...]" to put in the tooltip for the note's content zone.
	virtual void    toolTipInfos(TQStringList */*keys*/, TQStringList */*values*/) {} /// << Get "key: value" couples to put in the tooltip for the note's content zone.
	// Custom Zones:                                                      ///    Implement this if you want to store custom data.
	virtual int     zoneAt(const TQPoint &/*pos*/)           { return 0; } /// << If your note-type have custom zones, @return the zone at @p pos or 0 if it's not a custom zone!
	virtual TQRect   zoneRect(int zone, const TQPoint &/*pos*/);            /// << Idem, @return the rect of the custom zone
	virtual TQString zoneTip(int /*zone*/)                  { return ""; } /// << Idem, @return the toolTip of the custom zone
	virtual void    setCursor(TQWidget */*widget*/, int /*zone*/)       {} /// << Idem, set the mouse cursor for widget @p widget when it is over zone @p zone!
	virtual void    setHoveredZone(int /*oldZone*/, int /*newZone*/)   {} /// << If your note type need some feedback, you get notified of hovering changes here.
	virtual TQString statusBarMessage(int /*zone*/)         { return ""; } /// << @return the statusBar message to show for zone @p zone, or "" if nothing special have to be said.
	// Drag and Drop Content:
	virtual void    serialize(TQDataStream &/*stream*/)                 {} /// << Serialize the content in a TQDragObject. If it consists of a file, it can be serialized for you.
	virtual bool    shouldSerializeFile()           { return useFile(); } /// << @return true if the dragging process should serialize the filename (and move the file if cutting).
	virtual void    addAlternateDragObjects(KMultipleDrag*/*dragObj*/) {} /// << If you offer more than toText/Html/Image/Link(), this will be called if this is the only selected.
	virtual TQPixmap feedbackPixmap(int width, int height)            = 0; /// << @return the pixmap to put under the cursor while dragging this object.
	virtual bool    needSpaceForFeedbackPixmap()        { return false; } /// << @return true if a space must be inserted before and after the DND feedback pixmap.
	// Content Edition:
	virtual int      xEditorIndent()                        { return 0; } /// << If the editor should be indented (eg. to not cover an icon), return the number of pixels.
	// Open Content or File:
	virtual KURL urlToOpen(bool /*with*/); /// << @return the URL to open the note, or an invalid KURL if it's not openable. If @p with if false, it's a normal "Open". If it's true, it's for an "Open with..." action. The default implementation return the fullPath() if the note useFile() and nothing if not.
	enum OpenMessage {
		OpenOne,              /// << Message to send to the statusbar when opening this note.
		OpenSeveral,          /// << Message to send to the statusbar when opening several notes of this type.
		OpenOneWith,          /// << Message to send to the statusbar when doing "Open With..." on this note.
		OpenSeveralWith,      /// << Message to send to the statusbar when doing "Open With..." several notes of this type.
		OpenOneWithDialog,    /// << Prompt-message of the "Open With..." dialog for this note.
		OpenSeveralWithDialog /// << Prompt-message of the "Open With..." dialog for several notes of this type.
	};
	virtual TQString messageWhenOpenning(OpenMessage /*where*/) { return TQString(); } /// << @return the message to display according to @p where or nothing if it can't be done. @see OpenMessage describing the nature of the message that should be returned... The default implementation return an empty string. NOTE: If urlToOpen() is invalid and messageWhenOpenning() is not empty, then the user will be prompted to edit the note (with the message returned by messageWhenOpenning()) for eg. being able to edit URL of a link if it's empty when opening it...
	virtual TQString customOpenCommand() { return TQString(); }  /// << Reimplement this if your urlToOpen() should be opened with another application instead of the default KDE one. This choice should be left to the users in the setting (choice to use a custom app or not, and which app).
	// Common File Management:                                            ///    (and do save changes) and optionnaly hide the toolbar.
	virtual void setFileName(const TQString &fileName); /// << Set the filename. Reimplement it if you eg. want to update the view when the filename is changed.
	bool trySetFileName(const TQString &fileName);      /// << Set the new filename and return true. Can fail and return false if a file with this fileName already exists.
	TQString  fullPath();                               /// << Get the absolute path of the file where this content is stored on disk.
	TQString  fileName() { return m_fileName; }         /// << Get the file name where this content is stored (relative to the basket folder). @see fullPath().
	int      minWidth() { return m_minWidth; }         /// << Get the minimum width for this content.
	Note    *note()     { return m_note;     }         /// << Get the note managing this content.
	Basket  *basket();                                 /// << Get the basket containing the note managing this content.
  public:
	void setEdited(); /// << Mark the note as edited NOW: change the "last modification time and time" AND save the basket to XML file.
  protected:
	void contentChanged(int newMinWidth); /// << When the content has changed, inherited classes should call this to specify its new minimum size and trigger a basket relayout.
  private:
	Note    *m_note;
	TQString  m_fileName;
	int      m_minWidth;
  public:
	static const int FEEDBACK_DARKING;
};

/** Real implementation of plain text notes:
 * @author S�astien Laot
 */
class TextContent : public NoteContent
{
  public:
	// Constructor and destructor:
	TextContent(Note *parent, const TQString &fileName, bool lazyLoad = false);
	~TextContent();
	// Simple Generic Methods:
	NoteType::Id type();
	TQString typeName();
	TQString lowerTypeName();
	TQString toText(const TQString &/*cuttedFullPath*/);
	TQString toHtml(const TQString &imageName, const TQString &cuttedFullPath);
	bool    useFile();
	bool    canBeSavedAs();
	TQString saveAsFilters();
	bool    match(const FilterData &data);
	// Complexe Generic Methods:
	void    exportToHTML(HTMLExporter *exporter, int indent);
	TQString cssClass();
	int     setWidthAndGetHeight(int width);
	void    paint(TQPainter *painter, int width, int height, const TQColorGroup &colorGroup, bool isDefaultColor, bool isSelected, bool isHovered);
	bool    loadFromFile(bool lazyLoad);
	bool    finishLazyLoad();
	bool    saveToFile();
	TQString linkAt(const TQPoint &pos);
	void    fontChanged();
	TQString editToolTipText();
	// Drag and Drop Content:
	TQPixmap feedbackPixmap(int width, int height);
	// Open Content or File:
	TQString messageWhenOpenning(OpenMessage where);
//	TQString customOpenCommand();
	// Content-Specific Methods:
	void    setText(const TQString &text, bool lazyLoad = false); /// << Change the text note-content and relayout the note.
	TQString text() { return m_text; }     /// << @return the text note-content.
  protected:
	TQString          m_text;
	TQSimpleRichText *m_simpleRichText;
};

/** Real implementation of rich text (HTML) notes:
 * @author S�astien Laot
 */
class HtmlContent : public NoteContent
{
  public:
	// Constructor and destructor:
	HtmlContent(Note *parent, const TQString &fileName, bool lazyLoad = false);
	~HtmlContent();
	// Simple Generic Methods:
	NoteType::Id type();
	TQString typeName();
	TQString lowerTypeName();
	TQString toText(const TQString &/*cuttedFullPath*/);
	TQString toHtml(const TQString &imageName, const TQString &cuttedFullPath);
	bool    useFile();
	bool    canBeSavedAs();
	TQString saveAsFilters();
	bool    match(const FilterData &data);
	// Complexe Generic Methods:
	void    exportToHTML(HTMLExporter *exporter, int indent);
	TQString cssClass();
	int     setWidthAndGetHeight(int width);
	void    paint(TQPainter *painter, int width, int height, const TQColorGroup &colorGroup, bool isDefaultColor, bool isSelected, bool isHovered);
	bool    loadFromFile(bool lazyLoad);
	bool    finishLazyLoad();
	bool    saveToFile();
	TQString linkAt(const TQPoint &pos);
	void    fontChanged();
	TQString editToolTipText();
	// Drag and Drop Content:
	TQPixmap feedbackPixmap(int width, int height);
	// Open Content or File:
	TQString messageWhenOpenning(OpenMessage where);
	TQString customOpenCommand();
	// Content-Specific Methods:
	void    setHtml(const TQString &html, bool lazyLoad = false); /// << Change the HTML note-content and relayout the note.
	TQString html() { return m_html; }     /// << @return the HTML note-content.
  protected:
	TQString          m_html;
	TQString          m_textEquivalent; //OPTIM_FILTER
	TQSimpleRichText *m_simpleRichText;
};

/** Real implementation of image notes:
 * @author S�astien Laot
 */
class ImageContent : public NoteContent
{
  public:
	// Constructor and destructor:
	ImageContent(Note *parent, const TQString &fileName, bool lazyLoad = false);
	// Simple Generic Methods:
	NoteType::Id type();
	TQString typeName();
	TQString lowerTypeName();
	TQString toHtml(const TQString &imageName, const TQString &cuttedFullPath);
	TQPixmap toPixmap();
	bool    useFile();
	bool    canBeSavedAs();
	TQString saveAsFilters();
	bool    match(const FilterData &data);
	// Complexe Generic Methods:
	void    exportToHTML(HTMLExporter *exporter, int indent);
	TQString cssClass();
	int     setWidthAndGetHeight(int width);
	void    paint(TQPainter *painter, int width, int height, const TQColorGroup &colorGroup, bool isDefaultColor, bool isSelected, bool isHovered);
	bool    loadFromFile(bool lazyLoad);
	bool    finishLazyLoad();
	bool    saveToFile();
	void    fontChanged();
	TQString editToolTipText();
	void    toolTipInfos(TQStringList *keys, TQStringList *values);
	// Drag and Drop Content:
	TQPixmap feedbackPixmap(int width, int height);
	bool    needSpaceForFeedbackPixmap() { return true; }
	// Open Content or File:
	TQString messageWhenOpenning(OpenMessage where);
	TQString customOpenCommand();
	// Content-Specific Methods:
	void    setPixmap(const TQPixmap &pixmap); /// << Change the pixmap note-content and relayout the note.
	TQPixmap pixmap() { return m_pixmap; }     /// << @return the pixmap note-content.
  protected:
	TQPixmap  m_pixmap;
	char    *m_format;
};

/** Real implementation of animated image (GIF, MNG) notes:
 * @author S�astien Laot
 */
class AnimationContent : public TQObject, public NoteContent // TQObject to be able to receive TQMovie signals
{
  Q_OBJECT
  TQ_OBJECT
  public:
	// Constructor and destructor:
	AnimationContent(Note *parent, const TQString &fileName, bool lazyLoad = false);
	// Simple Generic Methods:
	NoteType::Id type();
	TQString typeName();
	TQString lowerTypeName();
	TQString toHtml(const TQString &imageName, const TQString &cuttedFullPath);
	TQPixmap toPixmap();
	bool    useFile();
	bool    canBeSavedAs();
	TQString saveAsFilters();
	bool    match(const FilterData &data);
	void    fontChanged();
	TQString editToolTipText();
	// Drag and Drop Content:
	TQPixmap feedbackPixmap(int width, int height);
	bool    needSpaceForFeedbackPixmap() { return true; }
	// Complexe Generic Methods:
	void    exportToHTML(HTMLExporter *exporter, int indent);
	TQString cssClass();
	int     setWidthAndGetHeight(int width);
	void    paint(TQPainter *painter, int width, int height, const TQColorGroup &colorGroup, bool isDefaultColor, bool isSelected, bool isHovered);
	bool    loadFromFile(bool lazyLoad);
	bool    finishLazyLoad();
	bool    saveToFile();
	// Open Content or File:
	TQString messageWhenOpenning(OpenMessage where);
	TQString customOpenCommand();
	// Content-Specific Methods:
	bool    setMovie(const TQMovie &movie); /// << Change the movie note-content and relayout the note.
	TQMovie  movie() { return m_movie; }    /// << @return the movie note-content.
  protected slots:
	void movieUpdated(const TQRect&);
	void movieResized(const TQSize&);
	void movieStatus(int status);
  protected:
	TQMovie  m_movie;
	int     m_oldStatus;
	static int INVALID_STATUS;
};

/** Real implementation of file notes:
 * @author S�astien Laot
 */
class FileContent : public TQObject, public NoteContent
{
  Q_OBJECT
  TQ_OBJECT
  public:
	// Constructor and destructor:
	FileContent(Note *parent, const TQString &fileName);
	// Simple Generic Methods:
	NoteType::Id type();
	TQString typeName();
	TQString lowerTypeName();
	TQString toHtml(const TQString &imageName, const TQString &cuttedFullPath);
	bool    useFile();
	bool    canBeSavedAs();
	TQString saveAsFilters();
	bool    match(const FilterData &data);
	// Complexe Generic Methods:
	void    exportToHTML(HTMLExporter *exporter, int indent);
	TQString cssClass();
	int     setWidthAndGetHeight(int width);
	void    paint(TQPainter *painter, int width, int height, const TQColorGroup &colorGroup, bool isDefaultColor, bool isSelected, bool isHovered);
	bool    loadFromFile(bool /*lazyLoad*/);
	void    fontChanged();
	void    linkLookChanged();
	TQString editToolTipText();
	void    toolTipInfos(TQStringList *keys, TQStringList *values);
	// Drag and Drop Content:
	TQPixmap feedbackPixmap(int width, int height);
	// Custom Zones:
	int     zoneAt(const TQPoint &pos);
	TQRect   zoneRect(int zone, const TQPoint &/*pos*/);
	TQString zoneTip(int zone);
	void    setCursor(TQWidget *widget, int zone);
	// Content Edition:
	int      xEditorIndent();
	// Open Content or File:
	TQString messageWhenOpenning(OpenMessage where);
	// Content-Specific Methods:
	void    setFileName(const TQString &fileName); /// << Reimplemented to be able to relayout the note.
	virtual LinkLook* linkLook() { return LinkLook::fileLook; }
  protected:
	LinkDisplay m_linkDisplay;
	// File Preview Management:
  protected slots:
	void newPreview(const KFileItem*, const TQPixmap &preview);
	void removePreview(const KFileItem*);
	void startFetchingUrlPreview();
  protected:
	KIO::PreviewJob *m_previewJob;
};

/** Real implementation of sound notes:
 * @author S�astien Laot
 */
class SoundContent : public FileContent // A sound is a file with just a bit different user interaction
{
  Q_OBJECT
  TQ_OBJECT
  public:
	// Constructor and destructor:
	SoundContent(Note *parent, const TQString &fileName);
	// Simple Generic Methods:
	NoteType::Id type();
	TQString typeName();
	TQString lowerTypeName();
	TQString toHtml(const TQString &imageName, const TQString &cuttedFullPath);
	bool    useFile();
	bool    canBeSavedAs();
	TQString saveAsFilters();
	bool    match(const FilterData &data);
	TQString editToolTipText();
	// Complexe Generic Methods:
	TQString cssClass();
	// Custom Zones:
	TQString zoneTip(int zone);
	void    setHoveredZone(int oldZone, int newZone);
	// Open Content or File:
	TQString messageWhenOpenning(OpenMessage where);
	TQString customOpenCommand();
	// Content-Specific Methods:
	LinkLook* linkLook() { return LinkLook::soundLook; }
};

/** Real implementation of link notes:
 * @author S�astien Laot
 */
class LinkContent : public TQObject, public NoteContent
{
  Q_OBJECT
  TQ_OBJECT
  public:
	// Constructor and destructor:
	LinkContent(Note *parent, const KURL &url, const TQString &title, const TQString &icon, bool autoTitle, bool autoIcon);
	// Simple Generic Methods:
	NoteType::Id type();
	TQString typeName();
	TQString lowerTypeName();
	TQString toText(const TQString &/*cuttedFullPath*/);
	TQString toHtml(const TQString &imageName, const TQString &cuttedFullPath);
	void    toLink(KURL *url, TQString *title, const TQString &cuttedFullPath);
	bool    useFile();
	bool    canBeSavedAs();
	TQString saveAsFilters();
	bool    match(const FilterData &data);
	// Complexe Generic Methods:
	void    exportToHTML(HTMLExporter *exporter, int indent);
	TQString cssClass();
	int     setWidthAndGetHeight(int width);
	void    paint(TQPainter *painter, int width, int height, const TQColorGroup &colorGroup, bool isDefaultColor, bool isSelected, bool isHovered);
	void    saveToNode(TQDomDocument &doc, TQDomElement &content);
	void    fontChanged();
	void    linkLookChanged();
	TQString editToolTipText();
	void    toolTipInfos(TQStringList *keys, TQStringList *values);
	// Drag and Drop Content:
	void    serialize(TQDataStream &stream);
	TQPixmap feedbackPixmap(int width, int height);
	// Custom Zones:
	int     zoneAt(const TQPoint &pos);
	TQRect   zoneRect(int zone, const TQPoint &/*pos*/);
	TQString zoneTip(int zone);
	void    setCursor(TQWidget *widget, int zone);
	TQString statusBarMessage(int zone);
	// Open Content or File:
	KURL urlToOpen(bool /*with*/);
	TQString messageWhenOpenning(OpenMessage where);
	// Content-Specific Methods:
	void    setLink(const KURL &url, const TQString &title, const TQString &icon, bool autoTitle, bool autoIcon); /// << Change the link and relayout the note.
	KURL    url()       { return m_url;       } /// << @return the URL of the link note-content.
	TQString title()     { return m_title;     } /// << @return the displayed title of the link note-content.
	TQString icon()      { return m_icon;      } /// << @return the displayed icon of the link note-content.
	bool    autoTitle() { return m_autoTitle; } /// << @return if the title is auto-computed from the URL.
	bool    autoIcon()  { return m_autoIcon;  } /// << @return if the icon is auto-computed from the URL.
  protected:
	KURL        m_url;
	TQString     m_title;
	TQString     m_icon;
	bool        m_autoTitle;
	bool        m_autoIcon;
	LinkDisplay m_linkDisplay;
	// File Preview Management:
  protected slots:
	void newPreview(const KFileItem*, const TQPixmap &preview);
	void removePreview(const KFileItem*);
	void startFetchingUrlPreview();
  protected:
	KIO::PreviewJob *m_previewJob;
};

/** Real implementation of launcher notes:
 * @author S�astien Laot
 */
class LauncherContent : public NoteContent
{
  public:
	// Constructor and destructor:
	LauncherContent(Note *parent, const TQString &fileName);
	// Simple Generic Methods:
	NoteType::Id type();
	TQString typeName();
	TQString lowerTypeName();
	TQString toHtml(const TQString &imageName, const TQString &cuttedFullPath);
	void    toLink(KURL *url, TQString *title, const TQString &cuttedFullPath);
	bool    useFile();
	bool    canBeSavedAs();
	TQString saveAsFilters();
	bool    match(const FilterData &data);
	// Complexe Generic Methods:
	void    exportToHTML(HTMLExporter *exporter, int indent);
	TQString cssClass();
	int     setWidthAndGetHeight(int width);
	void    paint(TQPainter *painter, int width, int height, const TQColorGroup &colorGroup, bool isDefaultColor, bool isSelected, bool isHovered);
	bool    loadFromFile(bool /*lazyLoad*/);
	void    fontChanged();
	TQString editToolTipText();
	void    toolTipInfos(TQStringList *keys, TQStringList *values);
	// Drag and Drop Content:
	TQPixmap feedbackPixmap(int width, int height);
	// Custom Zones:
	int     zoneAt(const TQPoint &pos);
	TQRect   zoneRect(int zone, const TQPoint &/*pos*/);
	TQString zoneTip(int zone);
	void    setCursor(TQWidget *widget, int zone);
	// Open Content or File:
	KURL urlToOpen(bool with);
	TQString messageWhenOpenning(OpenMessage where);
	// Content-Specific Methods:
	void    setLauncher(const TQString &name, const TQString &icon, const TQString &exec); /// << Change the launcher note-content and relayout the note. Normally called by loadFromFile (no save done).
	TQString name() { return m_name; }                              /// << @return the URL of the launcher note-content.
	TQString icon() { return m_icon; }                              /// << @return the displayed icon of the launcher note-content.
	TQString exec() { return m_exec; }                              /// << @return the execute command line of the launcher note-content.
	// TODO: KService *service() ??? And store everything in thta service ?
  protected:
	TQString     m_name; // TODO: Store them in linkDisplay to gain place (idem for Link notes)
	TQString     m_icon;
	TQString     m_exec;
	LinkDisplay m_linkDisplay;
};

/** Real implementation of color notes:
 * @author S�astien Laot
 */
class ColorContent : public NoteContent
{
  public:
	// Constructor and destructor:
	ColorContent(Note *parent, const TQColor &color);
	// Simple Generic Methods:
	NoteType::Id type();
	TQString typeName();
	TQString lowerTypeName();
	TQString toText(const TQString &/*cuttedFullPath*/);
	TQString toHtml(const TQString &imageName, const TQString &cuttedFullPath);
	bool    useFile();
	bool    canBeSavedAs();
	TQString saveAsFilters();
	bool    match(const FilterData &data);
	// Complexe Generic Methods:
	void    exportToHTML(HTMLExporter *exporter, int indent);
	TQString cssClass();
	int     setWidthAndGetHeight(int width);
	void    paint(TQPainter *painter, int width, int height, const TQColorGroup &colorGroup, bool isDefaultColor, bool isSelected, bool isHovered);
	void    saveToNode(TQDomDocument &doc, TQDomElement &content);
	void    fontChanged();
	TQString editToolTipText();
	void    toolTipInfos(TQStringList *keys, TQStringList *values);
	// Drag and Drop Content:
	void    serialize(TQDataStream &stream);
	TQPixmap feedbackPixmap(int width, int height);
	bool    needSpaceForFeedbackPixmap() { return true; }
	void    addAlternateDragObjects(KMultipleDrag *dragObject);
	// Content-Specific Methods:
	void    setColor(const TQColor &color); /// << Change the color note-content and relayout the note.
	TQColor  color() { return m_color; }    /// << @return the color note-content.
  protected:
	TQColor  m_color;
	static const int RECT_MARGIN;
};

/** Real implementation of unknown MIME-types dropped notes:
 * @author S�astien Laot
 */
class UnknownContent : public NoteContent
{
  public:
	// Constructor and destructor:
	UnknownContent(Note *parent, const TQString &fileName);
	// Simple Generic Methods:
	NoteType::Id type();
	TQString typeName();
	TQString lowerTypeName();
	TQString toText(const TQString &/*cuttedFullPath*/);
	TQString toHtml(const TQString &imageName, const TQString &cuttedFullPath);
	void    toLink(KURL *url, TQString *title, const TQString &cuttedFullPath);
	bool    useFile();
	bool    canBeSavedAs();
	TQString saveAsFilters();
	bool    match(const FilterData &data);
	// Complexe Generic Methods:
	void    exportToHTML(HTMLExporter *exporter, int indent);
	TQString cssClass();
	int     setWidthAndGetHeight(int width);
	void    paint(TQPainter *painter, int width, int height, const TQColorGroup &colorGroup, bool isDefaultColor, bool isSelected, bool isHovered);
	bool    loadFromFile(bool /*lazyLoad*/);
	void    fontChanged();
	TQString editToolTipText();
	// Drag and Drop Content:
	bool    shouldSerializeFile() { return false; }
	void    addAlternateDragObjects(KMultipleDrag *dragObject);
	TQPixmap feedbackPixmap(int width, int height);
	bool    needSpaceForFeedbackPixmap() { return true; }
	// Open Content or File:
	KURL urlToOpen(bool /*with*/) { return KURL(); }
	// Content-Specific Methods:
	TQString mimeTypes() { return m_mimeTypes; } /// << @return the list of MIME types this note-content contains.
  protected:
	TQString m_mimeTypes;
	static const int DECORATION_MARGIN;
};

void NoteFactory__loadNode(const TQDomElement &content, const TQString &lowerTypeName, Note *parent, bool lazyLoad);

#endif // NOTECONTENT_H
