/***************************************************************************
 *   Copyright (C) 2003 by S�astien Laot                                 *
 *   slaout@linux62.org                                                    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.             *
 ***************************************************************************/

#ifndef LINKLABEL_H
#define LINKLABEL_H

#include <tqframe.h>

class TQString;
class KURL;
class TQColor;
class TQLabel;
class TQBoxLayout;
class TQSpacerItem;
class TQPushButton;
class TQCheckBox;
class TQComboBox;

class KColorCombo2;
class IconSizeCombo;

class HTMLExporter;
class HelpLabel;
class KCModule;

/** Store the style of links
  * @author S�astien Laot
  */
class LinkLook
{
  public:
	enum Underlining { Always = 0, Never, OnMouseHover, OnMouseOutside };
	enum Preview { None = 0, IconSize, TwiceIconSize, ThreeIconSize };
	LinkLook(bool useLinkColor = true, bool canPreview = true);
	LinkLook(const LinkLook &other);
	void setLook( bool italic, bool bold, int underlining,
	              TQColor color, TQColor hoverColor,
	              int iconSize, int preview /*= None*/ );
	inline bool   italic()       const { return m_italic;       }
	inline bool   bold()         const { return m_bold;         }
	inline int    underlining()  const { return m_underlining;  }
	inline TQColor color()        const { return m_color;        }
	inline TQColor hoverColor()   const { return m_hoverColor;   }
	inline int    iconSize()     const { return m_iconSize;     }
	inline int    preview()      const { return m_preview;      }
	inline bool   useLinkColor() const { return m_useLinkColor; }
	inline bool   canPreview()   const { return m_canPreview;   }
	/* Helpping Functions */
	bool underlineOutside() const { return underlining() == Always || underlining() == OnMouseOutside; }
	bool underlineInside()  const { return underlining() == Always || underlining() == OnMouseHover;   }
	bool previewEnabled()   const { return canPreview() && preview() > None;                           }
	int  previewSize() const;
	TQColor effectiveColor() const;
	TQColor effectiveHoverColor() const;
	TQColor defaultColor() const;
	TQColor defaultHoverColor() const;
	TQString toCSS(const TQString &cssClass, const TQColor &defaultTextColor) const;
  private:
	bool   m_italic;
	bool   m_bold;
	int    m_underlining;
	TQColor m_color;
	TQColor m_hoverColor;
	int    m_iconSize;
	int    m_preview;
	bool   m_useLinkColor;
	bool   m_canPreview;
  public:
	/* Global Looks */
	static LinkLook *soundLook;
	static LinkLook *fileLook;
	static LinkLook *localLinkLook;
	static LinkLook *networkLinkLook;
	static LinkLook *launcherLook;
	/* Static method to get a LinkLook from an URL */
	static LinkLook* lookForURL(const KURL &url);
};

/** Used to represent links with icon and specific look
  * Note : This label will appear blank while LinkLook willn't be set
  * @author S�astien Laot
  */
class LinkLabel : public TQFrame
{
  Q_OBJECT
  TQ_OBJECT
  public:
	LinkLabel(int hAlign, int vAlign, TQWidget *parent = 0, const char *name = 0, WFlags f = 0);
	LinkLabel(const TQString &title, const TQString &icon, LinkLook *look, int hAlign, int vAlign,
	          TQWidget *parent = 0, const char *name = 0, WFlags f = 0);
	~LinkLabel();
  public:
	void setLink(const TQString &title, const TQString &icon, LinkLook *look = 0);
	void setLook(LinkLook *look);
	void setAlign(int hAlign, int vAlign);
	void setSelected(bool selected);
	void setPaletteBackgroundColor(const TQColor &color);
	int  heightForWidth(int w = -1) const;
	TQString toHtml(const TQString &imageName);
  protected:
	void initLabel(int hAlign, int vAlign);
	void enterEvent(TQEvent*);
	void leaveEvent(TQEvent*);
  private:
	TQBoxLayout  *m_layout;
	TQLabel      *m_icon;
	TQLabel      *m_title;
	TQSpacerItem *m_spacer1;
	TQSpacerItem *m_spacer2;

	bool         m_isSelected;
	bool         m_isHovered;

	LinkLook    *m_look;
	int          m_hAlign;
	int          m_vAlign;
};

/** THE NEW CLASS TO DISPLAY Links FOR THE NEW BASKET ENGINE.
 * We should get ride of class LinkLabel soon.
 * And LinkLabel will be entirely rewritten to use this LinkDisplay as the drawing primitives.
 * @author S�astien Laot
 */
class LinkDisplay
{
  public:
	LinkDisplay();                                                               /// << Create a new empty unselected LinkDisplay. Please then call setLink() to make sense.
	// Configure the link displayer:
	void    setLink(const TQString &title, const TQString &icon, LinkLook *look, const TQFont &font);  /// << Change the content and disposition. minWidth(), width() & height() can have changed. Keep the old preview (if any)
	void    setLink(const TQString &title, const TQString &icon, const TQPixmap &preview, LinkLook *look, const TQFont &font);  /// << Idem but change the preview too (or remove it if it is invalid)
	void    setWidth(int width);                                                 /// << Set a new width. @see height() that will be computed.
	// Get its properties:
	int     minWidth() const { return m_minWidth; }                              /// << @return the minimum width to display this link.
	int     maxWidth() const { return m_maxWidth; }                              /// << @return the maximum width to display this link.
	int     width()    const { return m_width;    }                              /// << @return the width of the link. It is never less than minWidth()!
	int     height()   const { return m_height;   }                              /// << @return the height if the link after having set it a width.
	// And finaly, use it:
	void    paint(TQPainter *painter, int x, int y, int width, int height, const TQColorGroup &colorGroup, bool isDefaultColor, bool isSelected, bool isHovered, bool isIconButtonHovered) const; /// << Draw the link on a painter. Set textColor to be !isValid() to use the LinkLook color. Otherwise it will use this color!
	TQPixmap feedbackPixmap(int width, int height, const TQColorGroup &colorGroup, bool isDefaultColor); /// << @return the pixmap to put under the cursor while dragging this object.
	// Eventually get some information about the link display:
	bool    iconButtonAt(const TQPoint &pos) const;                               /// << @return true if the icon button is under point @p pos.
	TQRect   iconButtonRect() const;                                              /// << @return the rectangle of the icon button.
	// Utility function:
	TQFont   labelFont(TQFont font, bool isIconButtonHovered) const;               /// << @return the font for this link, according to parent font AND LinkLook!
	int     heightForWidth(int width) const;                                     /// << @return the needed height to display the link in function of a width.
	TQString toHtml(const TQString &imageName) const;                              /// << Convert the link to HTML code, using the LinkLook to style it.
	TQString toHtml(HTMLExporter *exporter, const KURL &url, const TQString &title = "");
  private:
	TQString   m_title;
	TQString   m_icon;
	TQPixmap   m_preview;
	LinkLook *m_look;
	TQFont     m_font;
	int       m_minWidth;
	int       m_maxWidth;
	int       m_width;
	int       m_height;
};

/** A widget to edit a LinkLook, showing a live example to the user.
  * @author S�astien Laot
  */
class LinkLookEditWidget : public TQWidget
{
  Q_OBJECT
  TQ_OBJECT
  public:
	LinkLookEditWidget(KCModule* module, const TQString exTitle, const TQString exIcon,
					   TQWidget *parent = 0, const char *name = 0, WFlags fl = 0);
	~LinkLookEditWidget();
	void saveChanges();
	void saveToLook(LinkLook *look);
	void set(LinkLook *look);
  private slots:
	void slotChangeLook();
  protected:
	LinkLook      *m_look;
	TQCheckBox     *m_italic;
	TQCheckBox     *m_bold;
	TQComboBox     *m_underlining;
	KColorCombo2  *m_color;
	KColorCombo2  *m_hoverColor;
	IconSizeCombo *m_iconSize;
	TQComboBox     *m_preview;
	LinkLook      *m_exLook;
	LinkLabel     *m_example;
	TQString        m_exTitle;
	TQString        m_exIcon;
	HelpLabel     *m_hLabel;
	TQLabel        *m_label;
};

#endif // LINKLABEL_H
