/***************************************************************************
 *   Copyright (C) 2003 by S�astien Laot                                 *
 *   slaout@linux62.org                                                    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.             *
 ***************************************************************************/

#include <tqlabel.h>
#include <kurl.h>
#include <tqlayout.h>
#include <kiconloader.h>
#include <tqcursor.h>
#include <klocale.h>
#include <tqpushbutton.h>
#include <tqcheckbox.h>
#include <tqcombobox.h>
#include <tqhgroupbox.h>
#include <tqpainter.h>
#include <kglobalsettings.h>
#include <tqstyle.h>
#include <kapplication.h>
#include <kaboutdata.h>
#include <kdialogbase.h>
#include <kcmodule.h>
#include <kdebug.h>

#include "linklabel.h"
#include "variouswidgets.h"
#include "tools.h"
#include "global.h"
#include "kcolorcombo2.h"
#include "htmlexporter.h"

/** LinkLook */

LinkLook *LinkLook::soundLook       = new LinkLook(/*useLinkColor=*/false, /*canPreview=*/false);
LinkLook *LinkLook::fileLook        = new LinkLook(/*useLinkColor=*/false, /*canPreview=*/true);
LinkLook *LinkLook::localLinkLook   = new LinkLook(/*useLinkColor=*/true,  /*canPreview=*/true);
LinkLook *LinkLook::networkLinkLook = new LinkLook(/*useLinkColor=*/true,  /*canPreview=*/false);
LinkLook *LinkLook::launcherLook    = new LinkLook(/*useLinkColor=*/true,  /*canPreview=*/false);

LinkLook::LinkLook(bool useLinkColor, bool canPreview)
{
	m_useLinkColor = useLinkColor;
	m_canPreview   = canPreview;
	m_iconSize     = 0;
}

LinkLook::LinkLook(const LinkLook &other)
{
	m_useLinkColor = other.useLinkColor();
	m_canPreview   = other.canPreview();
	setLook( other.italic(), other.bold(), other.underlining(),
	         other.color(), other.hoverColor(),
	         other.iconSize(), other.preview() );
}

void LinkLook::setLook(bool italic, bool bold, int underlining,
                       TQColor color, TQColor hoverColor,
                       int iconSize, int preview)
{
	m_italic      = italic;
	m_bold        = bold;
	m_underlining = underlining;
	m_color       = color;
	m_hoverColor  = hoverColor;
	m_iconSize    = iconSize;
	m_preview     = (canPreview() ? preview : None);
}

int LinkLook::previewSize() const
{
	if (previewEnabled()) {
		switch (preview()) {
			default:
			case None:          return 0;
			case IconSize:      return iconSize();
			case TwiceIconSize: return iconSize() * 2;
			case ThreeIconSize: return iconSize() * 3;
		}
	} else
		return 0;
}

TQColor LinkLook::effectiveColor() const
{
	if (m_color.isValid())
		return m_color;
	else
		return defaultColor();
}

TQColor LinkLook::effectiveHoverColor() const
{
	if (m_hoverColor.isValid())
		return m_hoverColor;
	else
		return defaultHoverColor();
}

TQColor LinkLook::defaultColor() const
{
	if (m_useLinkColor)
		return KGlobalSettings::linkColor();
	else
		return KGlobalSettings::textColor();
}

TQColor LinkLook::defaultHoverColor() const
{
	return TQt::red;
}

LinkLook* LinkLook::lookForURL(const KURL &url)
{
	return url.isLocalFile() ? localLinkLook : networkLinkLook;
}

TQString LinkLook::toCSS(const TQString &cssClass, const TQColor &defaultTextColor) const
{
	// Set the link class:
	TQString css = TQString("   .%1 a { display: block; width: 100%;").arg(cssClass);
	if (underlineOutside())
		css += " text-decoration: underline;";
	else
		css += " text-decoration: none;";
	if (m_italic == true)
		css += " font-style: italic;";
	if (m_bold == true)
		css += " font-weight: bold;";
	TQColor textColor = (color().isValid() || m_useLinkColor ? effectiveColor() : defaultTextColor);
	css += TQString(" color: %1; }\n").arg(textColor.name());

	// Set the hover state class:
	TQString hover;
	if (m_underlining == OnMouseHover)
		hover = "text-decoration: underline;";
	else if (m_underlining == OnMouseOutside)
		hover = "text-decoration: none;";
	if (effectiveHoverColor() != effectiveColor()) {
		if (!hover.isEmpty())
			hover += " ";
		hover += TQString("color: %4;").arg(effectiveHoverColor().name());
	}

	// But include it only if it contain a different style than non-hover state:
	if (!hover.isEmpty())
		css += TQString("   .%1 a:hover { %2 }\n").arg(cssClass, hover);

	return css;
}

/** LinkLabel */

LinkLabel::LinkLabel(int hAlign, int vAlign, TQWidget *parent, const char *name, WFlags f)
 : TQFrame(parent, name, f), m_isSelected(false), m_isHovered(false), m_look(0)
{
	initLabel(hAlign, vAlign);
}

LinkLabel::LinkLabel(const TQString &title, const TQString &icon, LinkLook *look, int hAlign, int vAlign,
                     TQWidget *parent, const char *name, WFlags f)
 : TQFrame(parent, name, f), m_isSelected(false), m_isHovered(false), m_look(0)
{
	initLabel(hAlign, vAlign);
	setLink(title, icon, look);
}

void LinkLabel::initLabel(int hAlign, int vAlign)
{
	m_layout  = new TQBoxLayout(this, TQBoxLayout::LeftToRight);
	m_icon    = new TQLabel(this);
	m_title   = new TQLabel(this);
	m_spacer1 = new TQSpacerItem(0, 0, TQSizePolicy::Preferred/*Expanding*/, TQSizePolicy::Preferred/*Expanding*/);
	m_spacer2 = new TQSpacerItem(0, 0, TQSizePolicy::Preferred/*Expanding*/, TQSizePolicy::Preferred/*Expanding*/);

	m_hAlign = hAlign;
	m_vAlign = vAlign;

	m_title->setTextFormat(TQt::PlainText);

	// DEGUB:
	//m_icon->setPaletteBackgroundColor("lightblue");
	//m_title->setPaletteBackgroundColor("lightyellow");
}

LinkLabel::~LinkLabel()
{
}

void LinkLabel::setLink(const TQString &title, const TQString &icon, LinkLook *look)
{
	if (look)
		m_look = look; // Needed for icon size

	m_title->setText(title);
	m_title->setShown( ! title.isEmpty() );

	if (icon.isEmpty())
		m_icon->clear();
	else {
		TQPixmap pixmap = DesktopIcon(icon, m_look->iconSize(), m_look->iconSize(), kapp);
		if (!pixmap.isNull())
			m_icon->setPixmap(pixmap);
	}
	m_icon->setShown( ! icon.isEmpty() );

	if (look)
		setLook(look);
}

void LinkLabel::setLook(LinkLook *look) // FIXME: called externaly (so, without setLink()) it's buggy (icon not
{
	m_look = look;

	TQFont font;
	font.setBold(look->bold());
	font.setUnderline(look->underlineOutside());
	font.setItalic(look->italic());
	m_title->setFont(font);
	m_title->setPaletteForegroundColor( m_isSelected ? KApplication::palette().active().highlightedText() : look->effectiveColor() );

	m_icon->setShown( m_icon->pixmap() && ! m_icon->pixmap()->isNull() );

	setAlign(m_hAlign, m_vAlign);
}

void LinkLabel::setAlign(int hAlign, int vAlign)
{
	m_hAlign = hAlign;
	m_vAlign = vAlign;

	if (!m_look)
		return;

	// Define alignment flags :
	//FIXME TODO: Use directly flags !
	int hFlag, vFlag, wBreak;
	switch (hAlign) {
		default:
		case 0: hFlag = TQt::AlignLeft;    break;
		case 1: hFlag = TQt::AlignHCenter; break;
		case 2: hFlag = TQt::AlignRight;   break;
	}
	switch (vAlign) {
		case 0: vFlag = TQt::AlignTop;     break;
		default:
		case 1: vFlag = TQt::AlignVCenter; break;
		case 2: vFlag = TQt::AlignBottom;  break;
	}
	wBreak = TQt::WordBreak * (hAlign != 1);

	// Clear the widget :
	m_layout->removeItem(m_spacer1);
	m_layout->remove(m_icon);
	m_layout->remove(m_title);
	m_layout->removeItem(m_spacer2);

	// Otherwise, minimumSize will be incoherent (last size ? )
	m_layout->setResizeMode(TQLayout::Minimum);

	// And re-populate the widget with the appropriates things and order
	bool addSpacers = hAlign == 1;
	m_layout->setDirection(TQBoxLayout::LeftToRight);
	//m_title->setSizePolicy( TQSizePolicy(TQSizePolicy::Fixed, TQSizePolicy::Maximum/*Expanding*/, 0, 0, false) );
	m_icon->setSizePolicy( TQSizePolicy(TQSizePolicy::Fixed, TQSizePolicy::Preferred/*Expanding*/, 0, 0, false) );
	m_spacer1->changeSize( 0, 0, TQSizePolicy::Expanding, TQSizePolicy::Preferred/*Expanding*/ );
	m_spacer2->changeSize( 0, 0, TQSizePolicy::Expanding, TQSizePolicy::Preferred/*Expanding*/ );

	m_icon->setAlignment( hFlag | vFlag );
	m_title->setAlignment( hFlag | vFlag | wBreak );
	if ( addSpacers && (vAlign != 0) ||
	   (m_title->text().isEmpty() && hAlign == 2) )
		m_layout->addItem(m_spacer1);
	if (hAlign == 2) { // If align at right, icon is at right
		m_layout->addWidget(m_title);
		m_layout->addWidget(m_icon);
	} else {
		m_layout->addWidget(m_icon);
		m_layout->addWidget(m_title);
	}
	if ( addSpacers && (vAlign != 2) ||
	   (m_title->text().isEmpty() && hAlign == 0) )
		m_layout->addItem(m_spacer2);
}

void LinkLabel::enterEvent(TQEvent*)
{
	m_isHovered = true;
	if ( ! m_isSelected )
		m_title->setPaletteForegroundColor(m_look->effectiveHoverColor());

	TQFont font = m_title->font();
	font.setUnderline(m_look->underlineInside());
	m_title->setFont(font);
}

void LinkLabel::leaveEvent(TQEvent*)
{
	m_isHovered = false;
	if ( ! m_isSelected )
		m_title->setPaletteForegroundColor(m_look->effectiveColor());

	TQFont font = m_title->font();
	font.setUnderline(m_look->underlineOutside());
	m_title->setFont(font);
}

void LinkLabel::setSelected(bool selected)
{
	m_isSelected = selected;
	if (selected)
		m_title->setPaletteForegroundColor(KApplication::palette().active().highlightedText());
	else if (m_isHovered)
		m_title->setPaletteForegroundColor(m_look->effectiveHoverColor());
	else
		m_title->setPaletteForegroundColor(m_look->effectiveColor());
}

void LinkLabel::setPaletteBackgroundColor(const TQColor &color)
{
	TQFrame::setPaletteBackgroundColor(color);
	m_title->setPaletteBackgroundColor(color);
}

int LinkLabel::heightForWidth(int w) const
{
	int iconS  = (m_icon->isShown())   ? m_look->iconSize()                 : 0; // Icon size
	int iconW  = iconS;                                                          // Icon width to remove to w
	int titleH = (m_title->isShown())  ? m_title->heightForWidth(w - iconW) : 0; // Title height

	return (titleH >= iconS) ? titleH : iconS; // No margin for the moment !
}

TQString LinkLabel::toHtml(const TQString &imageName)
{
	TQString begin = "<font color=" + m_look->effectiveColor().name() + ">";
	TQString end   = "</font>";
	if (m_look->italic()) {
		begin += "<i>";
		end.prepend("</i>");
	}
	if (m_look->bold()) {
		begin += "<b>";
		end.prepend("</b>");
	}
	if (m_look->underlineOutside()) {
		begin += "<u>";
		end.prepend("</u>");
	}
	if (m_icon->pixmap()) {
		TQPixmap icon(*m_icon->pixmap());
		begin.prepend("<img src=" + imageName + " style=\"vertical-align: middle\"> ");
		TQMimeSourceFactory::defaultFactory()->setPixmap(imageName, icon);
	} else
		TQMimeSourceFactory::defaultFactory()->setData(imageName, 0L);
	return begin + Tools::textToHTMLWithoutP(m_title->text()) + end;
}

/** class LinkDisplay
 */

LinkDisplay::LinkDisplay()
 : m_title(), m_icon(), m_preview(), m_look(0), m_font(), m_minWidth(0), m_width(0), m_height(0)
{
}

void LinkDisplay::setLink(const TQString &title, const TQString &icon, LinkLook *look, const TQFont &font)
{
	setLink(title, icon, m_preview, look, font);
}

void LinkDisplay::setLink(const TQString &title, const TQString &icon, const TQPixmap &preview, LinkLook *look, const TQFont &font)
{
	m_title   = title;
	m_icon    = icon;
	m_preview = preview;
	m_look    = look;
	m_font    = font;

	// "Constants":
	int BUTTON_MARGIN = kapp->style().pixelMetric(TQStyle::PM_ButtonMargin);
	int LINK_MARGIN   = BUTTON_MARGIN + 2;

	// Recompute m_minWidth:
	TQRect textRect = TQFontMetrics(labelFont(font, false)).boundingRect(0, 0, /*width=*/1, 500000, TQt::AlignAuto | TQt::AlignTop | TQt::WordBreak, m_title);
	int iconPreviewWidth = TQMAX(m_look->iconSize(), (m_look->previewEnabled() ? m_preview.width() : 0));
	m_minWidth = BUTTON_MARGIN - 1 + iconPreviewWidth + LINK_MARGIN + textRect.width();
	// Recompute m_maxWidth:
	textRect = TQFontMetrics(labelFont(font, false)).boundingRect(0, 0, /*width=*/50000000, 500000, TQt::AlignAuto | TQt::AlignTop | TQt::WordBreak, m_title);
	m_maxWidth = BUTTON_MARGIN - 1 + iconPreviewWidth + LINK_MARGIN + textRect.width();
	// Adjust m_width:
	if (m_width < m_minWidth)
		setWidth(m_minWidth);
	// Recompute m_height:
	m_height = heightForWidth(m_width);
}

void LinkDisplay::setWidth(int width)
{
	if (width < m_minWidth)
		width = m_minWidth;

	if (width != m_width) {
		m_width  = width;
		m_height = heightForWidth(m_width);
	}
}

/** Paint on @p painter
  *       in (@p x, @p y, @p width, @p height)
  *       using @p colorGroup for the button drawing (if @p isHovered)
  *       and the LinkLook color() for the text,
  *       unless [the LinkLook !color.isValid() and it does not useLinkColor()] or [@p isDefaultColor is false]: in this case it will use @p colorGroup.text().
  *       It will draw the button if @p isIconButtonHovered.
  */
void LinkDisplay::paint(TQPainter *painter, int x, int y, int width, int height, const TQColorGroup &colorGroup,
                        bool isDefaultColor, bool isSelected, bool isHovered, bool isIconButtonHovered) const
{
	int BUTTON_MARGIN = kapp->style().pixelMetric(TQStyle::PM_ButtonMargin);
	int LINK_MARGIN   = BUTTON_MARGIN + 2;

	TQPixmap pixmap;
	// Load the preview...:
	if (!isHovered && m_look->previewEnabled() && !m_preview.isNull())
		pixmap  = m_preview;
	// ... Or the icon (if no preview or if the "Open" icon should be shown):
	else {
		int           iconSize   = m_look->iconSize();
		TQString       iconName   = (isHovered ? Global::openNoteIcon() : m_icon);
		KIcon::States iconState  = (isIconButtonHovered ? KIcon::ActiveState : KIcon::DefaultState);
		pixmap = kapp->iconLoader()->loadIcon(iconName, KIcon::Desktop, iconSize, iconState, 0L, /*canReturnNull=*/false);
	}
	int iconPreviewWidth  = TQMAX(m_look->iconSize(), (m_look->previewEnabled() ? m_preview.width()  : 0));
	int pixmapX = (iconPreviewWidth - pixmap.width()) / 2;
	int pixmapY = (height - pixmap.height()) / 2;
	// Draw the button (if any) and the icon:
	if (isHovered)
		kapp->style().tqdrawPrimitive(TQStyle::PE_ButtonCommand, painter, TQRect(-1, -1, iconPreviewWidth + 2*BUTTON_MARGIN, height + 2),
		                            colorGroup, TQStyle::Style_Enabled | (isIconButtonHovered ? TQStyle::Style_MouseOver : 0));
	painter->drawPixmap(x + BUTTON_MARGIN - 1 + pixmapX, y + pixmapY, pixmap);

	// Figure out the text color:
	if (isSelected)
		painter->setPen(KGlobalSettings::highlightedTextColor());
	else if (isIconButtonHovered)
		painter->setPen(m_look->effectiveHoverColor());
	else if (!isDefaultColor || (!m_look->color().isValid() && !m_look->useLinkColor())) // If the color is FORCED or if the link color default to the text color:
		painter->setPen(colorGroup.text());
	else
		painter->setPen(m_look->effectiveColor());
	// Draw the text:
	painter->setFont(labelFont(m_font, isIconButtonHovered));
	painter->drawText(x + BUTTON_MARGIN - 1 + iconPreviewWidth + LINK_MARGIN, y, width - BUTTON_MARGIN + 1 - iconPreviewWidth - LINK_MARGIN, height,
	                  TQt::AlignAuto | TQt::AlignVCenter | TQt::WordBreak, m_title);
}

TQPixmap LinkDisplay::feedbackPixmap(int width, int height, const TQColorGroup &colorGroup, bool isDefaultColor)
{
	int theWidth  = TQMIN(width, maxWidth());
	int theHeight = TQMIN(height, heightForWidth(theWidth));
	TQPixmap pixmap(theWidth, theHeight);
	pixmap.fill(colorGroup.background());
	TQPainter painter(&pixmap);
	paint(&painter, 0, 0, theWidth, theHeight, colorGroup, isDefaultColor,
	      /*isSelected=*/false, /*isHovered=*/false, /*isIconButtonHovered=*/false);
	painter.end();
	return pixmap;
}

bool LinkDisplay::iconButtonAt(const TQPoint &pos) const
{
	int BUTTON_MARGIN    = kapp->style().pixelMetric(TQStyle::PM_ButtonMargin);
//	int LINK_MARGIN      = BUTTON_MARGIN + 2;
	int iconPreviewWidth = TQMAX(m_look->iconSize(), (m_look->previewEnabled() ? m_preview.width()  : 0));

	return pos.x() <= BUTTON_MARGIN - 1 + iconPreviewWidth + BUTTON_MARGIN;
}

TQRect LinkDisplay::iconButtonRect() const
{
	int BUTTON_MARGIN    = kapp->style().pixelMetric(TQStyle::PM_ButtonMargin);
//	int LINK_MARGIN      = BUTTON_MARGIN + 2;
	int iconPreviewWidth = TQMAX(m_look->iconSize(), (m_look->previewEnabled() ? m_preview.width()  : 0));

	return TQRect(0, 0, BUTTON_MARGIN - 1 + iconPreviewWidth + BUTTON_MARGIN, m_height);
}

TQFont LinkDisplay::labelFont(TQFont font, bool isIconButtonHovered) const
{
	if (m_look->italic())
		font.setItalic(true);
	if (m_look->bold())
		font.setBold(true);
	if (isIconButtonHovered) {
		if (m_look->underlineInside())
			font.setUnderline(true);
	} else {
		if (m_look->underlineOutside())
			font.setUnderline(true);
	}
	return font;
}

int LinkDisplay::heightForWidth(int width) const
{
	int BUTTON_MARGIN     = kapp->style().pixelMetric(TQStyle::PM_ButtonMargin);
	int LINK_MARGIN       = BUTTON_MARGIN + 2;
	int iconPreviewWidth  = TQMAX(m_look->iconSize(), (m_look->previewEnabled() ? m_preview.width()  : 0));
	int iconPreviewHeight = TQMAX(m_look->iconSize(), (m_look->previewEnabled() ? m_preview.height() : 0));

	TQRect textRect = TQFontMetrics(labelFont(m_font, false)).boundingRect(0, 0, width - BUTTON_MARGIN + 1 - iconPreviewWidth - LINK_MARGIN, 500000, TQt::AlignAuto | TQt::AlignTop | TQt::WordBreak, m_title);
	return TQMAX(textRect.height(), iconPreviewHeight + 2*BUTTON_MARGIN - 2);
}

TQString LinkDisplay::toHtml(const TQString &/*imageName*/) const
{
	// TODO
	return "";
}

TQString LinkDisplay::toHtml(HTMLExporter *exporter, const KURL &url, const TQString &title)
{
	TQString linkIcon;
	if (m_look->previewEnabled() && !m_preview.isNull()) {
		TQString fileName = Tools::fileNameForNewFile("preview_" + url.fileName() + ".png", exporter->iconsFolderPath);
		TQString fullPath = exporter->iconsFolderPath + fileName;
		m_preview.save(fullPath, "PNG");
		linkIcon = TQString("<img src=\"%1\" width=\"%2\" height=\"%3\" alt=\"\">")
		           .arg(exporter->iconsFolderName + fileName, TQString::number(m_preview.width()), TQString::number(m_preview.height()));
	} else {
		linkIcon = exporter->iconsFolderName + exporter->copyIcon(m_icon, m_look->iconSize());
		linkIcon = TQString("<img src=\"%1\" width=\"%2\" height=\"%3\" alt=\"\">")
		           .arg(linkIcon, TQString::number(m_look->iconSize()), TQString::number(m_look->iconSize()));
	}

	TQString linkTitle = Tools::textToHTMLWithoutP(title.isEmpty() ? m_title : title);

	return TQString("<a href=\"%1\">%2 %3</a>").arg(url.prettyURL(), linkIcon, linkTitle);
}

/** LinkLookEditWidget **/

LinkLookEditWidget::LinkLookEditWidget(KCModule *module, const TQString exTitle, const TQString exIcon,
                                       TQWidget *parent, const char *name, WFlags fl)
 : TQWidget(parent, name, fl)
{
	TQLabel      *label;
	TQVBoxLayout *layout = new TQVBoxLayout(this, KDialogBase::marginHint(), KDialogBase::spacingHint());

	m_italic = new TQCheckBox(i18n("I&talic"), this);
	layout->addWidget(m_italic);

	m_bold = new TQCheckBox(i18n("&Bold"), this);
	layout->addWidget(m_bold);

	TQGridLayout *gl = new TQGridLayout(layout, /*rows=*//*(look->canPreview() ? 5 : 4)*/5, /*columns=*//*3*/4);
	gl->addItem(new TQSpacerItem(0, 0, TQSizePolicy::Expanding), 1, /*2*/3);

	m_underlining = new TQComboBox(false, this);
	m_underlining->insertItem(i18n("Always"));
	m_underlining->insertItem(i18n("Never"));
	m_underlining->insertItem(i18n("On mouse hovering"));
	m_underlining->insertItem(i18n("When mouse is outside"));
	label = new TQLabel(m_underlining, i18n("&Underline:"), this);
	gl->addWidget(label,         0, 0);
	gl->addWidget(m_underlining, 0, 1);

	m_color = new KColorCombo2(TQRgb(), this);
	label = new TQLabel(m_color, i18n("Colo&r:"), this);
	gl->addWidget(label,   1, 0);
	gl->addWidget(m_color, 1, 1);

	m_hoverColor = new KColorCombo2(TQRgb(), this);
	label = new TQLabel(m_hoverColor, i18n("&Mouse hover color:"), this);
	gl->addWidget(label,        2, 0);
	gl->addWidget(m_hoverColor, 2, 1);

	TQHBoxLayout *icoLay = new TQHBoxLayout(/*parent=*/0L, /*margin=*/0, KDialogBase::spacingHint());
	m_iconSize = new IconSizeCombo(false, this);
	icoLay->addWidget(m_iconSize);
	label = new TQLabel(m_iconSize, i18n("&Icon size:"), this);
	gl->addWidget(label,  3, 0);
	gl->addItem(  icoLay, 3, 1);

	m_preview = new TQComboBox(false, this);
	m_preview->insertItem(i18n("None"));
	m_preview->insertItem(i18n("Icon size"));
	m_preview->insertItem(i18n("Twice the icon size"));
	m_preview->insertItem(i18n("Three times the icon size"));
	m_label = new TQLabel(m_preview, i18n("&Preview:"), this);
	m_hLabel = new HelpLabel(
		i18n("You disabled preview but still see images?"),
		i18n("<p>This is normal because there are several type of notes.<br>"
		     "This setting only applies to file and local link notes.<br>"
		     "The images you see are image notes, not file notes.<br>"
		     "File notes are generic documents, whereas image notes are pictures you can draw in.</p>"
		     "<p>When dropping files to baskets, %1 detects their type and shows you the content of the files.<br>"
		     "For instance, when dropping image or text files, image and text notes are created for them.<br>"
		     "For type of files %2 does not understand, they are shown as generic file notes with just an icon or file preview and a filename.</p>"
		     "<p>If you do not want the application to create notes depending on the content of the files you drop, "
		     "go to the \"General\" page and uncheck \"Image or animation\" in the \"View Content of Added Files for the Following Types\" group.</p>")
		// TODO: Note: you can resize down maximum size of images...
			.arg(kapp->aboutData()->programName(), kapp->aboutData()->programName()),
		this);
	gl->addWidget(m_label,   4, 0);
	gl->addWidget(m_preview, 4, 1);
	gl->addMultiCellWidget(m_hLabel, /*fromRow=*/5, /*toRow=*/5, /*fromCol=*/1, /*toCol*/2);

	TQGroupBox *gb = new TQHGroupBox(i18n("Example"), this);
	m_exLook = new LinkLook;
	m_example = new LinkLabel(exTitle, exIcon, m_exLook, 1, 1, gb);
	m_example->setSizePolicy(TQSizePolicy::Expanding, TQSizePolicy::Expanding);
	m_example->setCursor(TQCursor(TQt::PointingHandCursor));
	layout->addWidget(gb);
	m_exTitle = exTitle;
	m_exIcon  = exIcon;

	connect( m_italic,      TQT_SIGNAL(stateChanged(int)),      this,   TQT_SLOT(slotChangeLook()) );
	connect( m_bold,        TQT_SIGNAL(stateChanged(int)),      this,   TQT_SLOT(slotChangeLook()) );
	connect( m_underlining, TQT_SIGNAL(activated(int)),         this,   TQT_SLOT(slotChangeLook()) );
	connect( m_color,       TQT_SIGNAL(changed(const TQColor&)), this,   TQT_SLOT(slotChangeLook()) );
	connect( m_hoverColor,  TQT_SIGNAL(changed(const TQColor&)), this,   TQT_SLOT(slotChangeLook()) );
	connect( m_iconSize,    TQT_SIGNAL(activated(int)),         this,   TQT_SLOT(slotChangeLook()) );
	connect( m_preview,     TQT_SIGNAL(activated(int)),         this,   TQT_SLOT(slotChangeLook()) );

	connect( m_italic,      TQT_SIGNAL(stateChanged(int)),      module, TQT_SLOT(changed())        );
	connect( m_bold,        TQT_SIGNAL(stateChanged(int)),      module, TQT_SLOT(changed())        );
	connect( m_underlining, TQT_SIGNAL(activated(int)),         module, TQT_SLOT(changed())        );
	connect( m_color,       TQT_SIGNAL(changed(const TQColor&)), module, TQT_SLOT(changed())        );
	connect( m_hoverColor,  TQT_SIGNAL(changed(const TQColor&)), module, TQT_SLOT(changed())        );
	connect( m_iconSize,    TQT_SIGNAL(activated(int)),         module, TQT_SLOT(changed())        );
	connect( m_preview,     TQT_SIGNAL(activated(int)),         module, TQT_SLOT(changed())        );
}

void LinkLookEditWidget::set(LinkLook *look)
{
	m_look = look;

	m_italic->setChecked(look->italic());
	m_bold->setChecked(look->bold());
	m_underlining->setCurrentItem(look->underlining());
	m_preview->setCurrentItem(look->preview());
	m_color->setDefaultColor(m_look->defaultColor());
	m_color->setColor(m_look->color());
	m_hoverColor->setDefaultColor(m_look->defaultHoverColor());
	m_hoverColor->setColor(m_look->hoverColor());
	m_iconSize->setSize(look->iconSize());
	m_exLook = new LinkLook(*look);
	m_example->setLook(m_exLook);

	if (!look->canPreview()) {
		m_label->setEnabled(false);
		m_hLabel->setEnabled(false);
		m_preview->setEnabled(false);
	}
	slotChangeLook();
}

void LinkLookEditWidget::slotChangeLook()
{
	saveToLook(m_exLook);
	m_example->setLink(m_exTitle, m_exIcon, m_exLook); // and can't reload it at another size
}

LinkLookEditWidget::~LinkLookEditWidget()
{
}

void LinkLookEditWidget::saveChanges()
{
	saveToLook(m_look);
}

void LinkLookEditWidget::saveToLook(LinkLook *look)
{
	look->setLook( m_italic->isOn(), m_bold->isOn(), m_underlining->currentItem(),
	               m_color->color(), m_hoverColor->color(),
	               m_iconSize->iconSize(), (look->canPreview() ? m_preview->currentItem() : LinkLook::None) );
}

#include "linklabel.moc"
