/***************************************************************************
 *   Copyright (C) 2004 by Antonio Fasolato                                *
 *   Antonio.Fasolato@poste.it                                             *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.             *
 ***************************************************************************/

#ifndef MAINWIDGET_H
#define MAINWIDGET_H

#include "MainWidgetBase.h"

#include <qstring.h>
#include <qprocess.h>
#include <qstringlist.h>
#include <qpixmap.h>
#include <qclipboard.h>
#include <qevent.h>
#include <qlabel.h>

#include <kurl.h>

#include <vector>
using namespace std;

class Options;

//! The app's main widget
/*!
 * \author Antonio Fasolato <Antonio.Fasolato@poste.it>
 * \version 1.2
 * 
 * This is the widget containing all the facilities to work with the image the user is tracing. It shows the current image and potrace output (if it exists), an interface to set the options and the buttons to control the execution of potrace.
 */
class MainWidget : public MainWidgetBase
{
  Q_OBJECT
  
  //! For simplicity
  friend class potracegui;
  
public:
	//! Default constructor (all parameters are simply passed to MainWidgetBase)
	MainWidget(QWidget* parent = 0, const char* name = 0, WFlags fl = 0 );
	//! Default destructor
	~MainWidget();

private:
	//! Contains the temporary file name used while tracing
	QString m_tmpFilename;
	//! Contains the input file name for autotrace
	QString m_tmpInputFilename;
	//! Contains the real file name where to save the traced image
	QString m_outputFilename;
	//! The process of potrace (used to abort it and to control its exit status)
	QProcess tracer;
	//! To know if the tracing has exited normally, or it has been interrupted
	bool aborted;
	//! To know if the current document is different from the saved one
	bool m_changed;
	//! The history of the options (not used yet, it will permit undo/redo in the future)
	vector<Options *> m_state;
	//! The history of the previews (not used yet, it will permit undo/redo in the future)
	vector<QPixmap> m_oldPreviews;
	//!  The current options
	vector<Options *>::iterator m_currentState;
	//! The current preview image
	vector<QPixmap>::iterator m_currentPreview;
	//! Label to display the original image
	QLabel *originalLabel;
	//! Label to display the traced image
	QLabel *previewLabel;

	//! Retrives an image from an URL
	/*!
	 * \param url The url of the image
	 *
	 * Retrives an image from an URL and loads into the original image viewer: if it is a local file, it simply loads it, if it is a remote URL, first it downloads it, then it loads it.<BR>
	 * After loading the image, it resets the options for the document (unless otherwise specified by the user.)
	 */
	void getImageFromURL(KURL url);
	//! Gets the options from m_currentState and sets up all the widget from this informations
	/*!
	 * \sa getOptions()
	 */
	void readOptions();
	//! Gets the options from all the widget and saves them into m_currentState
	/*!
	 * \sa readOptions()
	 */
	void getOptions();
	//! Gets the options for potrace from all the widget and saves them into m_currentState
	/*!
	 * \sa readOptions()
	 */
	void getPotraceOptions();
	//! Gets the options for autotrace from all the widget and saves them into m_currentState
	/*!
	 * \sa readOptions()
	 */
	void getAutotraceOptions();
	//! Creates the command line for tracing
	/*!
	 * It reads m_currentState and creates from all the options a commandline to be executed
	 * \returns A QStringList that can be used as a commandline to a QProcess
	 */
	QStringList createCommandLine();
	//! Creates the command line for tracing with potrace
	/*!
	 * It reads m_currentState and creates from all the options a commandline to be executed
	 * \returns A QStringList that can be used as a commandline to a QProcess
	 */
	QStringList createPotraceCommandLine();
	//! Creates the command line for tracing with autotrace
	/*!
	 * It reads m_currentState and creates from all the options a commandline to be executed
	 * \returns A QStringList that can be used as a commandline to a QProcess
	 */
	QStringList createAutotraceCommandLine();
	//! Sets m_tmpFilename to a new temporary filename (generated through mkstemp())
	bool createTmpFilename();
	//! Deletes the file pointed by m_tmpFilename
	bool deleteTmpFilename();
	//! Shows a KDialog whit nformations
	/*!
	 * \param caption The caption of the dialog
	 * \param text The text to display into the dialog
	 * \returns The code of the dialog button pressed.
	 */
	int notificationDialog(QString caption, QString text);
	//! Accepts only a KURL
	void dragEnterEvent(QDragEnterEvent* event);
	//! Receives a KURL and loads it.
	/*!
	 * \sa getImageFromURL()
	 */
	void dropEvent(QDropEvent* event);

private slots:
	//! Opens a file (whith a KFileDialog)
	void fileOpen();
	//! Saves the current image
	void fileSave();
	//! Saves the current image whith an alternate name
	void fileSaveAs();
	//! Closes the file (and the window)
	void fileClose();
	//! Cuts the original image into the clipboard
	void cut();
	//! Copies the current image into the clipboard
	void copy();
	//! Gets an image (if any) from the clipboard
	void paste();
	//! Executed when the tracer process has ended.
	/*!
	 * Executed when the tracer process has ended. If the process was successful it loads the traced image into the preview widget (unless otherways specified by the user)
	 */
	void tracerExited();
	//! Closes tracer stdin after passing the original image to it
	void writeCompleted();
	//! Executes potrace whith the options specified by the user
	/*!
	 * \sa createCommandLine()
	 */
	void trace();
	//! Reverts to the default options
	void defaultOptions();
	//! When an output format for potrace has been selected, it shows the relevant options
	void showOptions(const QString &selection);
	//! When a backend has been selected, it shows the relevant options
	void backendChoosen(const QString &back);
	//! Syncronize margins if corresponding checkbox is checked
	void marginValueChanged( double newValue);

	
signals:
	//! Emitted to change the statusbar
	/*!
	 * \param message The message to display into the statusbar
	 */
	void signalChangeStatusbar(QString message);
protected:
    bool marginValueSynconizingNecessary;
};

#endif

