/***************************************************************************
    smb4k.cpp  -  The main class of Smb4K.
                             -------------------
    begin                : Sam Mär  1 14:57:21 CET 2003
    copyright            : (C) 2003-2007 by Alexander Reinholdt
    email                : dustpuppy@users.berlios.de
 ***************************************************************************/

/***************************************************************************
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful, but   *
 *   WITHOUT ANY WARRANTY; without even the implied warranty of            *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   General Public License for more details.                              *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,   *
 *   MA  02110-1301 USA                                                    *
 ***************************************************************************/

// Qt includes
#include <qvaluelist.h>
#include <qpopupmenu.h>

// KDE includes
#include <kstatusbar.h>
#include <kstdaction.h>
#include <klibloader.h>
#include <kmessagebox.h>
#include <kuniqueapplication.h>
#include <kiconloader.h>
#include <kactionclasses.h>
#include <klocale.h>
#include <kprogress.h>
#include <kconfigdialog.h>
#include <kaccel.h>

// application specific includes
#include "smb4k.h"
#include "smb4ksystemtray.h"
#include "dialogs/smb4kbookmarkeditor.h"
#include "core/smb4ksettings.h"
#include "core/smb4kcore.h"
#include "core/smb4kglobal.h"
#include "core/smb4kbookmark.h"
#include "core/smb4kshare.h"
#include "core/smb4kdefs.h"

using namespace Smb4KGlobal;


Smb4KApp::Smb4KApp( QWidget *parent, const char *name )
: KParts::DockMainWindow( parent, name )
{
  m_system_tray = NULL;

  m_shares_part = NULL;
  m_browser_part = NULL;
  m_search_part = NULL;

  // Set XML file:
  setXMLFile( "smb4k_shell.rc" );

  // Set up the actions.
  setupActions();

  // Set up the status bar:
  setupStatusBar();

  // Set up the view:
  setupView();

  // Setup all things that need to be done *after* createGUI()
  // was run:
  QPopupMenu *settings = static_cast<QPopupMenu *>( child( "settings", "QPopupMenu", true ) );

  if ( settings )
  {
    settings->insertItem( i18n( "&Dock Widgets" ), dockHideShowMenu(), 3, 3 );
  }

  slotSetupBookmarksMenu();

  // Set up the system tray window:
  setupSystemTray();

  // Resize main window (will be overwritten by configuration):
  resize( 800, 600 );

  // Apply main window settings:
  applyMainWindowSettings( Smb4KSettings::self()->config(), "MainWindow" );
  readDockConfig( Smb4KSettings::self()->config(), "MainWindow" );

  // We have to set the tab labels here. The writeDockConfig()
  // readDockConfig() combo will ignore changes of the localization,
  // because the first labels will be saved and then loaded again
  // and again.
  KDockWidget *dock = NULL;
  KDockTabGroup *parent_group = NULL;

  if ( (dock = manager()->getDockWidgetFromName( "NetworkBrowser" )) != NULL )
  {
    dock->setTabPageLabel( i18n( "Network Browser" ) );

    if ( (parent_group = dock->parentDockTabGroup()) != NULL )
    {
      parent_group->changeTab( dock, i18n( "Network Browser" ) );
    }
  }

  if ( (dock = manager()->getDockWidgetFromName( "SearchDialog" )) != NULL )
  {
    dock->setTabPageLabel( i18n( "Search Dialog" ) );

    if ( (parent_group = dock->parentDockTabGroup()) != NULL )
    {
      parent_group->changeTab( dock,  i18n( "Search Dialog" ) );
    }
  }

  // Connections
  connect( actionCollection(),           SIGNAL( actionHighlighted( KAction * ) ),
           this,                         SLOT( slotActionHighlighted( KAction * ) ) );

  connect( Smb4KCore::self(),            SIGNAL( runStateChanged() ),
           this,                         SLOT( slotRunStateChanged() ) );

  connect( Smb4KCore::bookmarkHandler(), SIGNAL( bookmarksUpdated() ),
           this,                         SLOT( slotSetupBookmarksMenu() ) );

  connect( Smb4KCore::mounter(),         SIGNAL( updated() ),
           this,                         SLOT( slotShareListUpdated() ) );
}


Smb4KApp::~Smb4KApp()
{
}


void Smb4KApp::setupActions()
{
  actionCollection()->setHighlightingEnabled( true );

  (void) KStdAction::quit( this, SLOT( slotQuit() ), actionCollection(), "quit_action" );

  // Set up the "Settings" menu:
  setStandardToolBarMenuEnabled( true );
  createStandardStatusBarAction();

  KActionMenu *view_modes  = new KActionMenu( i18n( "Shares Vie&w" ), "view_choose",
                             actionCollection(), "view_modes_menu" );

  KRadioAction *icon_view  = new KRadioAction( i18n( "&Icon View" ), "view_icon", CTRL+Key_I,
                             this, SLOT( slotChangeSharesView() ), actionCollection(), "icon_view_action" );
  icon_view->setExclusiveGroup( "SharesViewActions" );

  KRadioAction *list_view  = new KRadioAction( i18n( "List Vie&w" ), "view_detailed", CTRL+Key_W,
                             this, SLOT( slotChangeSharesView() ), actionCollection(), "list_view_action" );
  list_view->setExclusiveGroup( "SharesViewActions" );

  switch ( Smb4KSettings::sharesView() )
  {
    case Smb4KSettings::EnumSharesView::IconView:
    {
      icon_view->setChecked( true );

      break;
    }
    case Smb4KSettings::EnumSharesView::ListView:
    {
      list_view->setChecked( true );

      break;
    }
    default:
    {
      break;
    }
  }

  view_modes->insert( icon_view, -1 );
  view_modes->insert( list_view, -1 );

  (void) KStdAction::preferences( this, SLOT( slotConfigDialog() ), actionCollection(), "configure_action" );

  // Notes:
  // (1) Actions from the parts will be included by setupView().
  // (2) The list of dock widgets will be plugged into the menu
  //     by setupView().
  // (3) The bookmark menu is managed by slotSetupBookmarksMenu().
}


void Smb4KApp::setupStatusBar()
{
  // Insert the items:
  statusBar()->insertItem( i18n( "Ready." ), Message, 1, false );

  KProgress *progress = new KProgress( statusBar(), "StatusBarProgressBar", 0 );
  progress->setFixedWidth( 75 );
  progress->setFixedHeight( fontMetrics().height() );
  progress->setPercentageVisible( false );
  statusBar()->addWidget( progress, 0, true );

  statusBar()->insertFixedItem( QString( "Smb4K %1" ).arg( VERSION ), Version, true );

  // Align the items:
  statusBar()->setItemAlignment( Message, AlignAuto );
  statusBar()->setItemAlignment( Progress, AlignCenter );
  statusBar()->setItemAlignment( Version, AlignAuto );
}


void Smb4KApp::setupView()
{
  //
  // Main widget (shares view):
  //
  KLibFactory *shares_factory = NULL;

  m_current_shares_view = Smb4KSettings::sharesView();

  switch ( Smb4KSettings::sharesView() )
  {
    case Smb4KSettings::EnumSharesView::IconView:
    {
      shares_factory = KLibLoader::self()->factory( "libsmb4ksharesiconview" );

      break;
    }
    case Smb4KSettings::EnumSharesView::ListView:
    {
      shares_factory = KLibLoader::self()->factory( "libsmb4kshareslistview" );

      break;
    }
    default:
    {
      break;
    }
  }

  if ( shares_factory )
  {
    m_shares_part = static_cast<KParts::Part *>( shares_factory->create( this, "SharesPart", "KParts::Part" ) );

    if ( m_shares_part )
    {
      KDockWidget *main = createDockWidget( "SharesView", SmallIcon( "hdd_mount" ), 0L );
      main->setWidget( m_shares_part->widget() );
      main->setDockSite( KDockWidget::DockCorner );
      main->setEnableDocking( KDockWidget::DockNone );
      setView( main );
      setMainDockWidget( main );

      createGUI( m_shares_part );

      accel()->insert( i18n( "Jump to shares view" ), CTRL+Key_3, this, SLOT( slotJumpToSharesView() ), false, true );
    }
  }
  else
  {
    KMessageBox::error( 0, "<qt>"+KLibLoader::self()->lastErrorMessage()+"</qt>" );

    KApplication::exit( 0 );

    return;
  }

  //
  // Browser widget:
  //
  KLibFactory *browser_factory = KLibLoader::self()->factory( "libsmb4knetworkbrowser" );

  if ( browser_factory )
  {
    m_browser_part = static_cast<KParts::Part *>( browser_factory->create( this, "BrowserPart", "KParts::Part" ) );

    if ( m_browser_part )
    {
      KDockWidget *network = createDockWidget( "NetworkBrowser", SmallIcon( "network" ), 0L, i18n( "Network Browser" ) );
      network->setWidget( m_browser_part->widget() );
      network->manualDock( getMainDockWidget(), KDockWidget::DockLeft, 45 );

      factory()->addClient( m_browser_part );

      accel()->insert( i18n( "Jump to network browser" ), CTRL+Key_1, this, SLOT( slotJumpToNetworkBrowser() ), false, true );
    }
  }
  else
  {
    KMessageBox::error( 0, "<qt>"+KLibLoader::self()->lastErrorMessage()+"</qt>" );

    KApplication::exit( 0 );

    return;
  }

  //
  // Search dialog
  //
  KLibFactory *search_factory = KLibLoader::self()->factory( "libsmb4ksearchdialog" );

  if ( search_factory )
  {
    m_search_part = static_cast<KParts::Part *>( search_factory->create( this, "SearchDialogPart", "KParts::Part" ) );

    if ( m_search_part )
    {
      KDockWidget *search = createDockWidget( "SearchDialog", SmallIcon( "find" ), 0L, i18n( "Search Dialog" ) );
      search->setWidget( m_search_part->widget() );

      KDockWidget *network = manager()->getDockWidgetFromName( "NetworkBrowser" );

      if ( network )
      {
        search->manualDock( network, KDockWidget::DockCenter, 0 );
      }
      else
      {
        search->manualDock( getMainDockWidget(), KDockWidget::DockLeft, 45 );
      }

      factory()->addClient( m_search_part );

      accel()->insert( i18n( "Jump to search dialog" ), CTRL+Key_2, this, SLOT( slotJumpToSearchDialog() ), false, true );
    }
  }
  else
  {
    KMessageBox::error( 0, "<qt>"+KLibLoader::self()->lastErrorMessage()+"</qt>" );

    KApplication::exit( 0 );

    return;
  }
}


void Smb4KApp::setupSystemTray()
{
  if ( !m_system_tray )
  {
    m_system_tray = new Smb4KSystemTray( this, "SystemTray" );
  }

  connect( m_system_tray, SIGNAL( quitSelected() ),
           this,          SLOT( slotQuit() ) );

  connect( m_system_tray, SIGNAL( settingsChanged() ),
           this,          SLOT( slotSettingsChanged() ) );

  m_system_tray->embed( Smb4KSettings::embedIntoSystemTray() );
}


void Smb4KApp::changeSharesView()
{
  // Change the information about the current view:
  m_current_shares_view = Smb4KSettings::sharesView();

  // Save dock widget settings:
  writeDockConfig( Smb4KSettings::self()->config(), "MainWindow" );

  // Clear the shares view
  factory()->removeClient( m_shares_part );

  KDockWidget *dock = NULL;

  if ( (dock = manager()->getDockWidgetFromName( "SharesView" )) != NULL )
  {
    delete dock;
  }

  // Load the new shares view:
  KLibFactory *shares_factory = NULL;

  switch ( Smb4KSettings::sharesView() )
  {
    case Smb4KSettings::EnumSharesView::IconView:
    {
      shares_factory = KLibLoader::self()->factory( "libsmb4ksharesiconview" );

      break;
    }
    case Smb4KSettings::EnumSharesView::ListView:
    {
      shares_factory = KLibLoader::self()->factory( "libsmb4kshareslistview" );

      break;
    }
    default:
    {
      break;
    }
  }

  if ( shares_factory )
  {
    m_shares_part = static_cast<KParts::Part *>( shares_factory->create( this, "SharesPart", "KParts::Part" ) );

    if ( m_shares_part )
    {
      KDockWidget *main = createDockWidget( "SharesView", SmallIcon( "hdd_mount" ), this );
      main->setWidget( m_shares_part->widget() );
      main->setDockSite( KDockWidget::DockCorner );
      main->setEnableDocking( KDockWidget::DockNone );
      setView( main );
      setMainDockWidget( main );

      factory()->addClient( m_shares_part );
    }
  }
  else
  {
    KMessageBox::error( 0, "<qt>"+KLibLoader::self()->lastErrorMessage()+"</qt>" );

    KApplication::exit( 0 );

    return;
  }

  // Apply the settings to the dock widgets:
  readDockConfig( Smb4KSettings::self()->config(), "MainWindow" );
}


bool Smb4KApp::queryExit()
{
  Smb4KSettings::setStartMainWindowDocked( !isVisible() );

  saveMainWindowSettings( Smb4KSettings::self()->config(), "MainWindow" );
  writeDockConfig( Smb4KSettings::self()->config(), "MainWindow" );

  hide();

  if ( m_system_tray )
  {
    delete m_system_tray;
    m_system_tray = NULL;
  }

  // Save any options here that have to be written directly
  // before the application exits:

  return true;
}


bool Smb4KApp::queryClose()
{
  if ( !kapp->sessionSaving() && isVisible() && m_system_tray->isEmbedded() &&
       Smb4KSettings::embedIntoSystemTray() )
  {
    // This part has been 'stolen' from JuK application.
    KMessageBox::information(this,
            i18n( "Closing the main window will keep Smb4K running in the system tray. "
                  "Use \"Quit\" from the \"File\" menu to quit the application."),
            i18n( "Docking in System Tray" ), "ClosingMainWindowInfo" );

    hide();

    return false;

  }
  else
  {
    return true;
  }
}


void Smb4KApp::timerEvent( QTimerEvent * )
{
  KProgress *progress_bar = static_cast<KProgress *>( child( "StatusBarProgressBar", "KProgress", true ) );

  if ( progress_bar )
  {
    progress_bar->setProgress( progress_bar->progress() + 1 );
  }
}


/////////////////////////////////////////////////////////////////////
// SLOT IMPLEMENTATIONS
/////////////////////////////////////////////////////////////////////

void Smb4KApp::slotQuit()
{
  statusBar()->changeItem( i18n( "Exiting..." ), Message );

  if ( m_system_tray->isEmbedded() )
  {
    m_system_tray->embed( false );
  }

  KApplication::exit( 0 );
}


void Smb4KApp::slotChangeSharesView()
{
  // Let's be sure that the last action that was highlighted is
  // indeed an action that changes the shares view:
  QString action_name;

  if ( m_action )
  {
    action_name = QString( m_action->name() );

    if ( QString::compare( action_name, "icon_view_action" ) != 0 &&
         QString::compare( action_name, "list_view_action" ) != 0 )
    {
      return;
    }
  }
  else
  {
    return;
  }

  KRadioAction *shares_view_action = static_cast<KRadioAction *>( m_action );

  // Set the check mark for the action:
  if ( !shares_view_action->isChecked() )
  {
    shares_view_action->setChecked( true );
  }
  else
  {
    // Do nothing
  }

  // Change the shares view setting and save it:
  if ( QString::compare( action_name, "icon_view_action" ) == 0 )
  {
    Smb4KSettings::setSharesView( Smb4KSettings::EnumSharesView::IconView );
  }
  else if ( QString::compare( action_name, "list_view_action" ) == 0 )
  {
    Smb4KSettings::setSharesView( Smb4KSettings::EnumSharesView::ListView );
  }
  else
  {
    return;
  }

  Smb4KSettings::writeConfig();

  changeSharesView();
}


void Smb4KApp::slotConfigDialog()
{
  // If the config dialog is already created and cached,
  // we do not create a new one but show the old instead:
  if ( KConfigDialog::showDialog( "ConfigDialog" ) )
  {
    return;
  }

  // Load the factory of the config dialog:
  KLibFactory *config_factory = KLibLoader::self()->factory( "libsmb4kconfigdialog" );

  if ( config_factory )
  {
    KConfigDialog *dlg = static_cast<KConfigDialog *>( config_factory->create( this, "ConfigDialog", "KConfigDialog" ) );

    if ( dlg )
    {
      connect( dlg, SIGNAL( settingsChanged() ), this, SLOT( slotSettingsChanged() ) );

      dlg->show();
    }
  }
  else
  {
    KMessageBox::error( 0, "<qt>"+KLibLoader::self()->lastErrorMessage()+"</qt>" );

    return;
  }
}


void Smb4KApp::slotSettingsChanged()
{
  // Adjust settings for the system tray widget:
  if ( m_system_tray )
  {
    m_system_tray->loadSettings();
    m_system_tray->embed( Smb4KSettings::embedIntoSystemTray() );
  }

  // (Re-)Load the correct shares view.
  if ( m_current_shares_view != Smb4KSettings::sharesView() )
  {
    changeSharesView();
  }
  else
  {
    // Do nothing
  }

  // Notify the parts to reload the settings.
  // Note: QApplication::postEvent() will delete the QCustomEvent pointers.
  QApplication::postEvent( m_browser_part, new QCustomEvent( EVENT_LOAD_SETTINGS ) );
  QApplication::postEvent( m_search_part, new QCustomEvent( EVENT_LOAD_SETTINGS ) );
  QApplication::postEvent( m_shares_part, new QCustomEvent( EVENT_LOAD_SETTINGS ) );
}


void Smb4KApp::slotBookmarkEditor()
{
  Smb4KBookmarkEditor *dlg = static_cast<Smb4KBookmarkEditor *>( child( "BookmarkEditor", "Smb4KBookmarkEditor", true ) );

  if ( !dlg )
  {
    dlg = new Smb4KBookmarkEditor( this, "BookmarkEditor" );
  }

  dlg->show();
}


void Smb4KApp::slotRunStateChanged()
{
  // Get the progress bar:
  KProgress *progress_bar = static_cast<KProgress *>( child( "StatusBarProgressBar", "KProgress", true ) );

  // Clear the status bar:
  statusBar()->clear();

  // Set the status bar message:
  switch( Smb4KCore::currentState() )
  {
    case SCANNER_INIT:
    {
      switch ( Smb4KSettings::browseList() )
      {
        case Smb4KSettings::EnumBrowseList::LookupDomains:
        {
          statusBar()->changeItem( i18n( "Looking up workgroups and domains..." ), Message );

          break;
        }
        case Smb4KSettings::EnumBrowseList::QueryCurrentMaster:
        {
          statusBar()->changeItem( i18n( "Querying current master browser..." ), Message );

          break;
        }
        case Smb4KSettings::EnumBrowseList::QueryCustomMaster:
        {
          statusBar()->changeItem( i18n( "Querying master browser %1..." ).arg( Smb4KSettings::customMasterBrowser().upper() ), Message );

          break;
        }
        case Smb4KSettings::EnumBrowseList::ScanBroadcastAreas:
        {
          statusBar()->changeItem( i18n( "Scanning broadcast areas..." ), Message );

          break;
        }
        default:
        {
          break;
        }
      }

      break;
    }
    case SCANNER_OPENING_WORKGROUP:
    {
      statusBar()->changeItem( i18n( "Opening workgroup..." ), Message );

      break;
    }
    case SCANNER_OPENING_HOST:
    {
      statusBar()->changeItem( i18n( "Retrieving list of shares..." ), Message );

      break;
    }
    case SCANNER_RETRIEVING_INFO:
    {
      statusBar()->changeItem( i18n( "Retrieving additional information..." ), Message );

      break;
    }
    case SCANNER_SEARCHING:
    {
      statusBar()->changeItem( i18n( "Searching..." ), Message );

      break;
    }
    case SCANNER_RETRYING_OPENING_HOST:
    {
      statusBar()->changeItem( i18n( "Retrying to retrieve list of shares..." ), Message );

      break;
    }
    case SCANNER_STOP:
    {
      statusBar()->changeItem( i18n( "Done." ), Message );

      break;
    }
    case MOUNTER_MOUNTING:
    {
      statusBar()->changeItem( i18n( "Mounting share..." ), Message );

      break;
    }
    case MOUNTER_UNMOUNTING:
    {
      statusBar()->changeItem( i18n( "Unmounting share..." ), Message );

      break;
    }
    case MOUNTER_STOP:
    {
      statusBar()->changeItem( i18n( "Done." ), Message );

      break;
    }
    case PRINT_START:
    {
      statusBar()->changeItem( i18n( "Printing file..." ), Message );

      break;
    }
    case PRINT_STOP:
    {
      statusBar()->changeItem( i18n( "Done." ), Message );

      break;
    }
    case SYNCHRONIZER_START:
    {
      statusBar()->changeItem( i18n( "Synchronizing data..." ), Message );

      break;
    }
    case SYNCHRONIZER_STOP:
    {
      statusBar()->changeItem( i18n( "Done." ), Message );

      break;
    }
    case PREVIEWER_START:
    {
      statusBar()->changeItem( i18n( "Generating preview..." ), Message );

      break;
    }
    case PREVIEWER_STOP:
    {
      statusBar()->changeItem( i18n( "Done." ), Message );

      break;
    }
    case CORE_STOP:
    {
      statusBar()->changeItem( i18n( "Ready." ), Message );

      break;
    }
    default:
    {
      break;
    }
  }

  // Set up the progress bar:
  if ( Smb4KCore::isRunning() )
  {
    // If the progress bar exists and is already set to
    // a totalSteps == 0, we do not need to do anything
    // here.
    if ( progress_bar && progress_bar->totalSteps() != 0 )
    {
      progress_bar->setTotalSteps( 0 );

      m_timer_id = startTimer( TIMER_INTERVAL );
    }
  }
  else
  {
    if ( progress_bar )
    {
      killTimer( m_timer_id );

      progress_bar->setTotalSteps( 100 );
      progress_bar->reset();
    }
  }
}


void Smb4KApp::slotSetupBookmarksMenu()
{
  // Set up bookmark related actions:
  QValueList<Smb4KBookmark *> bookmarks = Smb4KCore::bookmarkHandler()->getBookmarks();

  if ( !actionCollection()->action( "edit_bookmarks_action" ) )
  {
    unplugActionList( "bookmark_actions" );

    QPtrList<KAction> bookmark_actions;

    // Create the "Edit Bookmarks" action:
    bookmark_actions.append( new KAction( i18n( "&Edit Bookmarks" ), "bookmark", CTRL+Key_E,
                             this, SLOT( slotBookmarkEditor() ), actionCollection(),
                             "edit_bookmarks_action" ) );

    // Get the "Add Bookmark" action from the browser:
    QPtrList<KXMLGUIClient> clients_list = factory()->clients();

    for ( QPtrList<KXMLGUIClient>::Iterator it = clients_list.begin(); it != clients_list.end(); ++it )
    {
      if ( (*it)->action( "bookmark_action" ) )
      {
        bookmark_actions.append( (*it)->action( "bookmark_action" ) );

        break;
      }
      else
      {
        continue;
      }
    }

    bookmark_actions.append( new KActionSeparator() );

    plugActionList( "bookmark_actions", bookmark_actions );
  }

  // Unplug the action list:
  unplugActionList( "bookmarks" );

  // Get the bookmark action list and delete all entries. We could
  // also try to keep those actions that are not obsolete, but I think
  // this is the cleanest way.
  KActionPtrList list = actionCollection()->actions( "Bookmarks" );

  for ( KActionPtrList::Iterator it = list.begin(); it != list.end(); ++it )
  {
    actionCollection()->remove( *it );
  }

  // Set up the menu:
  if ( bookmarks.isEmpty() )
  {
    actionCollection()->action( "edit_bookmarks_action" )->setEnabled( false );
  }
  else
  {
    actionCollection()->action( "edit_bookmarks_action" )->setEnabled( true );

    // Work around sorting problems:
    QStringList display_strings;

    for ( QValueListIterator<Smb4KBookmark *> it = bookmarks.begin(); it != bookmarks.end(); ++it )
    {
      if ( !(*it)->label().isEmpty() && Smb4KSettings::showCustomBookmarkLabel() )
      {
        display_strings.append( (*it)->label() );
      }
      else
      {
        display_strings.append( (*it)->bookmark() );
      }
    }

    display_strings.sort();

    // Create the bookmark list and plug it into the menu:
    QPtrList<KAction> bookmark_list;

    for ( QStringList::ConstIterator it = display_strings.begin(); it != display_strings.end(); ++it )
    {
      KAction *a = new KAction( *it, "folder", KShortcut::null(), 0, 0, actionCollection(), *it );
      a->setGroup( "Bookmarks" );
      connect( a, SIGNAL( activated() ), this, SLOT( slotBookmarkActivated() ) );
      bookmark_list.append( a );
    }

    plugActionList( "bookmarks", bookmark_list );
  }
}


void Smb4KApp::slotBookmarkActivated()
{
  if ( m_action && QString::compare( m_action->group(), "Bookmarks" ) == 0 )
  {
    Smb4KBookmark *bookmark = NULL;

    if ( !m_action->plainText().startsWith( "//" ) )
    {
      bookmark = Smb4KCore::bookmarkHandler()->findBookmarkByLabel( m_action->plainText() );
    }
    else
    {
      bookmark = Smb4KCore::bookmarkHandler()->findBookmarkByName( m_action->plainText() );
    }

    if ( bookmark )
    {
      Smb4KCore::mounter()->mountShare( bookmark->workgroup(), bookmark->host(),
                                        bookmark->ip(), bookmark->share() );
    }
  }
  else
  {
    // Do nothing
  }
}


void Smb4KApp::slotShareListUpdated()
{
  KActionPtrList list = actionCollection()->actions( "Bookmarks" );

  if ( !list.isEmpty() )
  {
    QString name;

    for ( KActionPtrList::ConstIterator it = list.begin(); it != list.end(); ++it )
    {
      if ( !(*it)->plainText().startsWith( "//" ) )
      {
        Smb4KBookmark *bookmark = Smb4KCore::bookmarkHandler()->findBookmarkByLabel( (*it)->plainText() );

        name = bookmark->bookmark();
      }
      else
      {
        name = (*it)->plainText();
      }

      QValueList<Smb4KShare> share_list = Smb4KCore::mounter()->findShareByName( name );

      bool enable = true;

      for ( QValueList<Smb4KShare>::ConstIterator i = share_list.begin(); i != share_list.end(); ++i )
      {
        if ( !(*i).isForeign() )
        {
          enable = false;

          break;
        }
        else
        {
          continue;
        }
      }

      (*it)->setEnabled( enable );
    }
  }
}


void Smb4KApp::slotActionHighlighted( KAction *action )
{
  m_action = action;
}


void Smb4KApp::slotJumpToNetworkBrowser()
{
  KDockWidget *dock = NULL;

  if ( (dock = manager()->getDockWidgetFromName( "NetworkBrowser" )) != NULL &&
       dock->isHidden() )
  {
    manager()->getDockWidgetFromName( "NetworkBrowser" )->changeHideShowState();
  }

  // Send a custom focus event to the browser, that tells it to set
  // the focus to the list view widget.
  QApplication::postEvent( m_browser_part, new QCustomEvent( EVENT_SET_FOCUS ) );
}


void Smb4KApp::slotJumpToSearchDialog()
{
  KDockWidget *dock = NULL;

  if ( (dock = manager()->getDockWidgetFromName( "SearchDialog" )) != NULL &&
       dock->isHidden() )
  {
    manager()->getDockWidgetFromName( "SearchDialog" )->changeHideShowState();
  }

  // Send a custom focus event to the search dialog, that tells it
  // to set the focus to the combo box in the tool bar.
  QApplication::postEvent( m_search_part, new QCustomEvent( EVENT_SET_FOCUS ) );
}


void Smb4KApp::slotJumpToSharesView()
{
  // Send a custom focus event to the shares view, that tells it
  // to set the focus to its main widget.
  QApplication::postEvent( m_shares_part, new QCustomEvent( EVENT_SET_FOCUS ) );
}

#include "smb4k.moc"

