/***************************************************************************
    smb4ksynchronizationdialog  -  The synchronization dialog of Smb4K
                             -------------------
    begin                : Sa Mai 19 2007
    copyright            : (C) 2007 by Alexander Reinholdt
    email                : dustpuppy@users.berlios.de
 ***************************************************************************/

/***************************************************************************
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful, but   *
 *   WITHOUT ANY WARRANTY; without even the implied warranty of            *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   General Public License for more details.                              *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,   *
 *   MA  02110-1301 USA                                                    *
 ***************************************************************************/

// Qt includes
#include <qlayout.h>
#include <qframe.h>
#include <qlabel.h>

// KDE includes
#include <klocale.h>
#include <kguiitem.h>
#include <kdebug.h>
#include <kurlrequester.h>
#include <klineedit.h>
#include <kprogress.h>

// application specific includes
#include "smb4ksynchronizationdialog.h"
#include "../core/smb4kshare.h"
#include "../core/smb4kcore.h"
#include "../core/smb4ksynchronizationinfo.h"
#include "../core/smb4ksettings.h"

Smb4KSynchronizationDialog::Smb4KSynchronizationDialog( Smb4KShare *share, QWidget *parent, const char *name )
: KDialogBase( Plain, i18n( "Synchronization" ), User2|User1|Cancel, User1, parent, name, false, true ),
m_share( share )
{
  setWFlags( Qt::WDestructiveClose );

  setButtonGuiItem( User1, KGuiItem( i18n( "Synchronize" ), "bottom", i18n( "Synchronize the destination with the source" ) ) );
  setButtonGuiItem( User2, KGuiItem( i18n( "Swap Paths" ), QString::null, i18n( "Swap source and destination" ) ) );

  QFrame *frame = plainPage();
  QGridLayout *layout = new QGridLayout( frame );
  layout->setSpacing( 5 );
  layout->setMargin( 0 );

  QLabel *source_label =       new QLabel( i18n( "Source:" ), frame, "SourceURLLabel" );
  KURLRequester *source =      new KURLRequester( m_share->path()+"/", frame, "SourceURL" );
  source->setShowLocalProtocol( false );
  source->setMode( KFile::Directory | KFile::LocalOnly );

  QLabel *destination_label =  new QLabel( i18n( "Destination:" ), frame, "DestinationURLLabel" );
  KURLRequester *destination = new KURLRequester( Smb4KSettings::rsyncPrefix(), frame, "DestinationURL" );
  destination->setShowLocalProtocol( false );
  destination->setMode( KFile::Directory | KFile::LocalOnly );

  KLineEdit *current_file =      new KLineEdit( QString::null, frame, "ProgressInfo" );
  current_file->setEnableSqueezedText( true );
  current_file->setReadOnly( true );

  KProgress *individual =         new KProgress( frame, "IndividualProgress", 0 );
  individual->setEnabled( false );

  KProgress *total =           new KProgress( frame, "TotalProgress", 0 );
  total->setEnabled( false );

  QWidget *transfer_widget =   new QWidget( frame, "TransferInfoWidget" );
  QGridLayout *trans_layout =  new QGridLayout( transfer_widget );
  trans_layout->setSpacing( 5 );
  trans_layout->setMargin( 0 );

  QLabel *file_label =         new QLabel( i18n( "Files transferred:" ), transfer_widget,
                               "FilesTransferredLabel" );
  QLabel *file_trans_label =   new QLabel( "0 / 0", transfer_widget,
                               "FilesTransferred" );

  QLabel *rate_label =         new QLabel( i18n( "Transfer rate:" ), transfer_widget,
                               "TransferRateLabel" );
  QLabel *trans_rate_label =   new QLabel( "0.00 kB/s", transfer_widget,
                               "TransferRate" );

  trans_layout->addWidget( file_label, 0, 0, 0 );
  trans_layout->addWidget( file_trans_label, 0, 1, Qt::AlignRight );
  trans_layout->addWidget( rate_label, 1, 0, 0 );
  trans_layout->addWidget( trans_rate_label, 1, 1, Qt::AlignRight );

  transfer_widget->setEnabled( false );

  layout->addWidget( source_label, 0, 0, 0 );
  layout->addWidget( source, 0, 1, 0 );
  layout->addWidget( destination_label, 1, 0, 0 );
  layout->addWidget( destination, 1, 1, 0 );
  layout->addMultiCellWidget( current_file, 2, 2, 0, 1, 0 );
  layout->addMultiCellWidget( individual, 3, 3, 0, 1, 0 );
  layout->addMultiCellWidget( total, 4, 4, 0, 1, 0 );
  layout->addMultiCellWidget( transfer_widget, 5, 6, 0, 1, 0 );

  // Connections
  connect( Smb4KCore::synchronizer(), SIGNAL( progress( const Smb4KSynchronizationInfo & ) ),
           this,                      SLOT( slotProgress( const Smb4KSynchronizationInfo & ) ) );

  connect( Smb4KCore::synchronizer(), SIGNAL( finished() ),
           this,                      SLOT( slotSynchronizationFinished() ) );


  setFixedSize( (sizeHint().width() > 350 ? sizeHint().width() : 350), sizeHint().height() );
}


Smb4KSynchronizationDialog::~Smb4KSynchronizationDialog()
{
  // Do *not* delete the share object here.
}


/////////////////////////////////////////////////////////////////////////////
//   SLOT IMPLEMENTATIONS
/////////////////////////////////////////////////////////////////////////////


void Smb4KSynchronizationDialog::slotUser1()
{
  // Synchronize!

  // Disable the URL requesters but in a way, that the information
  // proviced in them is still readable:
  KURLRequester *source = static_cast<KURLRequester *>( child( "SourceURL", "KURLRequester", true ) );
  source->lineEdit()->setReadOnly( true );
  source->button()->setEnabled( false );

  KURLRequester *destination = static_cast<KURLRequester *>( child( "DestinationURL", "KURLRequester", true ) );
  destination->lineEdit()->setReadOnly( true );
  destination->button()->setEnabled( false );

  QWidget *transfer_widget = static_cast<QWidget *>( child( "TransferInfoWidget", "QWidget", true ) );
  transfer_widget->setEnabled( true );

  enableButton( User1, false );
  enableButton( User2, false );

  // Enable the progress bars and the information widgets:
  static_cast<KProgress *>( child( "IndividualProgress", "KProgress", true ) )->setEnabled( true );
  static_cast<KProgress *>( child( "TotalProgress", "KProgress", true ) )->setEnabled( true );

  Smb4KCore::synchronizer()->synchronize( source->url(), destination->url() );
}


void Smb4KSynchronizationDialog::slotUser2()
{
  // Swap URLs.

  KURLRequester *source = static_cast<KURLRequester *>( child( "SourceURL", "KURLRequester", true ) );
  KURLRequester *destination = static_cast<KURLRequester *>( child( "DestinationURL", "KURLRequester", true ) );

  QString sourceURL = source->url();
  QString destinationURL = destination->url();

  source->setURL( destinationURL );
  destination->setURL( sourceURL );
}


void Smb4KSynchronizationDialog::slotCancel()
{
  Smb4KCore::synchronizer()->abort();

  KDialogBase::slotCancel();
}


void Smb4KSynchronizationDialog::slotProgress( const Smb4KSynchronizationInfo &info )
{
  KLineEdit *progress =   static_cast<KLineEdit *>( child( "ProgressInfo", "KLineEdit", true ) );
  KProgress *individual = static_cast<KProgress *>( child( "IndividualProgress", "KProgress", true ) );
  KProgress *total =      static_cast<KProgress *>( child( "TotalProgress", "KProgress", true ) );
  QLabel *transferred =   static_cast<QLabel *>( child( "FilesTransferred", "QLabel", true ) );
  QLabel *rate =          static_cast<QLabel *>( child( "TransferRate", "QLabel", true ) );

  if ( !info.text().isEmpty() )
  {
    progress->setSqueezedText( info.text() );
  }

  if ( info.individualProgress() != -1 )
  {
    individual->setProgress( info.individualProgress() );
  }

  if ( info.totalProgress() != -1 )
  {
    total->setProgress( info.totalProgress() );
  }

  if ( info.totalFileNumber() != -1 && info.processedFileNumber() != -1 )
  {
    transferred->setText( QString( "%1 / %2" ).arg( info.processedFileNumber() ).arg( info.totalFileNumber() ) );
  }

  if ( !info.transferRate().isEmpty() )
  {
    rate->setText( info.transferRate() );
  }
}


void Smb4KSynchronizationDialog::slotSynchronizationFinished()
{
  KProgress *individual = static_cast<KProgress *>( child( "IndividualProgress", "KProgress", true ) );
  KProgress *total =      static_cast<KProgress *>( child( "TotalProgress", "KProgress", true ) );

  if ( individual && individual->progress() != 100 )
  {
    individual->setProgress( 100 );
  }

  if ( total && total->progress() != 100 )
  {
    total->setProgress( 100 );
  }

  // Change the "Cancel" button to a "Close" button.
  setButtonGuiItem( Cancel, KStdGuiItem::close() );
}

#include "smb4ksynchronizationdialog.moc"
