/***************************************************************************
    smb4kshare  -  This is a container that holds information about
    a mounted remote share.
                             -------------------
    begin                : Do M� 4 2004
    copyright            : (C) 2004 by Franck Babin
                           (C) 2005-2007 by Alexander Reinholdt
    email                : babinfranck@yahoo.ca
                           dustpuppy@users.berlios.de
 ***************************************************************************/

/***************************************************************************
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful, but   *
 *   WITHOUT ANY WARRANTY; without even the implied warranty of            *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   General Public License for more details.                              *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,   *
 *   MA  02110-1301 USA                                                    *
 ***************************************************************************/

#ifndef SMB4KSHARE_H
#define SMB4KSHARE_H

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

// Qt includes
#include <qstring.h>
#include <qcstring.h>

// KDE includes
#include <kuser.h>


/**
 * This class is a container that holds information about a remote share
 * that was mounted on the system. It belongs to the core classes of
 * Smb4K.
 *
 * @author Franck Babin,
 * @author Alexander Reinholdt <dustpuppy@users.berlios.de>
 */

class Smb4KShare
{
  public:
    /**
     * The default constructor.
     *
     * @param name        The name of the share: //HOST/SHARE or //USER\@HOST/SHARE (FreeBSD).
     *
     * @param path        The path where the share is mounted to.
     *
     * @param filesystem  The filesystem that was used. If you use this constructor it should be "smbfs".
     *
     * @param uid         The UID of the user.
     *
     * @param gid         The GID of the user.
     *
     * @param broken      Determines whether the share is broken. "Broken" means that the share is unaccessible.
     */
    Smb4KShare( const QString &name, const QString &path, const QString &filesystem, const int uid = 0, const int gid = 0, bool broken = false );

    /**
     * The constructor for CIFS shares. It does not take the UID and GID, but the
     * user name with which the login was done.
     *
     * @param name        The name of the share: //HOST/SHARE or //USER@HOST/SHARE (FreeBSD).
     *
     * @param path        The path where the share is mounted to.
     *
     * @param filesystem  The filesystem that was used. If you use this constructor it should be "cifs".
     *
     * @param username    The the user name that had to be used for authentication. It can be different from the local user name.
     *
     * @param foreign     Determines whether the share was mounted by another user (i.e. is a foreign share).
     *
     * @param broken      Determines whether the share is broken. "Broken" means that the share is unaccessible.
     */
    Smb4KShare( const QString &name, const QString &path, const QString &filesystem, const QString &username, bool foreign = false, bool broken = false );

    /**
     * Empty constructor.
     */
    Smb4KShare() {}

    /**
     * Copy constructor.
     *
     * @param share       The share that is to be copied.
     */
    Smb4KShare( const Smb4KShare &share );

    /**
     * The destructor
     */
    ~Smb4KShare();

    /**
     * Returns the name of the share as it has been gathered by the mounter.
     *
     * @returns           The name of the share.
     */
    const QString &name() const;

    /**
     * Returns the mount point aka path of the share as it has been gathered
     * by the mounter. This is a C-type string.
     *
     * @returns           The path of the share.
     */
    const QCString &path() const;

    /**
     * This function returns the canonical path of the share. In contrast to
     * Smb4KShare::path(), it will return the absolute path without symlinks. However,
     * should the share be broken (i.e. Smb4KShare::isBroken() returns TRUE),
     * only Smb4KShare::path() is returned.
     *
     * @returns           Returns the canonical path of the share.
     */
    const QCString canonicalPath() const;

    /**
     * Returns the UID of the mounted share.
     */
    int uid() const;

    /**
     * Set the UID of the mounted share.
     *
     * @param uid         The UID of the share
     */
    void setUID( int uid );

    /**
     * Returns the GID of the mounted share.
     */
    int gid() const;

    /**
     * Set the GID of the mounted share.
     *
     * @param gid         The GID of the share
     */
    void setGID( int gid );

    /**
     * Returns the name of the user of the share.
     */
    const QString user() const;

    /**
     * Returns the name of the group of the share.
     */
    const QString group() const;

    /**
     * Returns the file system of the share.
     */
    const QString &filesystem() const;

    /**
     * Returns the CIFS login (user name).
     */
    const QString &cifsLogin() const;

    /**
     * Is TRUE if the share is/seems to be mounted by another
     * user.
     *
     * @returns           TRUE if another user mounted the share and FALSE otherwise.
     */
    bool isForeign() const;

    /**
     * This function sets the share to be foreign.
     *
     * @param foreign     TRUE if share is foreign and FALSE otherwise.
     */
    void setForeign( bool foreign );

    /**
     * Returns TRUE if the share is broken and FALSE otherwise.
     */
    bool isBroken() const;

    /**
     * Sets the share to be broken.
     *
     * @param broken      TRUE if the share is broken and FALSE otherwise.
     */
    void setBroken( bool broken );

    /**
     * This function sets the value of the total disk usage. The value has to
     * be provided in kilobytes. If the disk usage could not be determined,
     * total has to be set to -1.
     *
     * @param total       The total disk usage in kB.
     */
    void setTotalDiskSpace( double total );

    /**
     * This function sets the value of the free space on the share. The value
     * has to be provided in kilobytes. If the free space could not be determined,
     * free has to be set to -1.
     *
     * @param free        The free disk space in kB.
     */
    void setFreeDiskSpace( double free );

    /**
     * This function returns the total disk space of the share.
     *
     * @returns           The total disk space in kB.
     */
    double totalDiskSpace() const;

    /**
     * This function returns the free disk space available on the share in kB.
     *
     * @returns           the free disk space in kB.
     */
    double freeDiskSpace() const;

    /**
     * This function returns the percentage of used disk space on the
     * share.
     *
     * @returns           the percentage of disk space used on the share.
     */
    double percentage() const;

    /**
     * Compare another Smb4KShare object with this one and return TRUE if both
     * carry the same data.
     *
     * @param share       The Smb4KShare object that is compared to this one
     *
     * @returns           TRUE if the values match.
     */
    bool equals( const Smb4KShare &share );


  private:
    /**
     * The name of the share.
     */
    QString m_name;

    /**
     * The mount point / path of the share
     */
    QCString m_path;

    /**
     * The filesystem string
     */
    QString m_filesystem;

    /**
     * The user ID of the share.
     */
    KUser m_user;

    /**
     * The group ID of the share.
     */
    KUserGroup m_group;

    /**
     * The CIFS login name
     */
    QString m_cifs_login;
    bool m_foreign_mount;
    bool m_broken;
    double m_total;
    double m_free;
};

#endif
