/***************************************************************************
    smb4ksambaoptionsinfo  -  This is a container class that carries
    various information of extra options for a specific host.
                             -------------------
    begin                : Mi Okt 18 2006
    copyright            : (C) 2006 by Alexander Reinholdt
    email                : dustpuppy@mail.berlios.de
 ***************************************************************************/

/***************************************************************************
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful, but   *
 *   WITHOUT ANY WARRANTY; without even the implied warranty of            *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   General Public License for more details.                              *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,   *
 *   MA  02110-1301 USA                                                    *
 ***************************************************************************/

#ifndef SMB4KSAMBAOPTIONSINFO_H
#define SMB4KSAMBAOPTIONSINFO_H

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

// Qt includes
#include <qstring.h>

// forward declarations
class Smb4KShare;


/**
 * This class provides a container for all extra options that the user defined
 * for a certain share.
 *
 * @author  Alexander Reinholdt <dustpuppy@mail.berlios.de>
 */

class Smb4KSambaOptionsInfo
{
  public:
    /**
     * Constructor. It takes the name of the network item in the form
     * HOST or //HOST/SHARE as only argument. If you use this constructor, you
     * need to use the set* functions to add information.
     *
     * @param name              The network item's name.
     */
    Smb4KSambaOptionsInfo( const QString &name );

    /**
     * Constructor. It takes a Smb4KShare object and extracts the name and the
     * filesystem from it. All other information has to be set with the set*
     * functions.
     *
     * @param share             A Smb4KShare object representing a share.
     */
    Smb4KSambaOptionsInfo( Smb4KShare *share );

    /**
     * The copy constructor.
     *
     * @param info              A Smb4KShareOptionsInfo object
     */
    Smb4KSambaOptionsInfo( const Smb4KSambaOptionsInfo &info );

    /**
     * The destructor.
     */
    ~Smb4KSambaOptionsInfo();

    /**
     * Sets the "should be remounted" flag.
     *
     * @param rm                TRUE if the share is to be remounted and
     *                          FALSE otherwise.
     */
    void setRemount( bool rm );

    /**
     * Returns TRUE if the share is to be remounted and FALSE otherwise.
     *
     * @returns                 TRUE if the share is to be remounted and FALSE otherwise
     */
    bool remount() const { return m_remount; }

    /**
     * This function sets the item name.
     *
     * @param name              The name of the network item
     */
    void setItemName( const QString &name );

    /**
     * This function returns the name of the network item, i.e. the server or
     * share.
     *
     * @returns                 the name of the network item.
     */
    const QString &itemName() const { return m_name; }

    /**
     * This function sets the port that should be used when querying this share.
     *
     * @param port              The port number
     */
    void setPort( int port );

    /**
     * This function returns the port that should be used when working with this
     * share. Please note, that it will be returned as an integer. If no port has been
     * defined, -1 will be returned.
     *
     * @returns                 the port number
     */
    int port() const { return m_port; }

    /**
     * This function sets the protocol to use with the net command. If @p protocol
     * is equal to "auto", the protocol will automatically be set to "" internally,
     * so that Smb4KSambaOptionsInfo::protocol() returns an empty string.
     *
     * @param protocol          the protocol
     */
    void setProtocol( const QString &protocol );

    /**
     * This function returns the protocol to use with the net command.
     *
     * @retuns                  the protocol
     */
    const QString &protocol() const { return m_protocol; }

    /**
     * Set the 'Use Kerberos' flag.
     *
     * @param krb               TRUE if the user wants to use Kerberos
     *                          and FALSE otherwise.
     */
    void setKerberos( bool krb );

    /**
     * This functions returns TRUE if the user wants to use Kerberos and
     * otherwise it returns FALSE.
     *
     * @returns                 TRUE if Kerberos should be used and FALSE
     *                          otherwise.
     */
    bool kerberos() const { return m_kerberos; }

    /**
     * With this function you can set the UID you want to use for this item.
     * However, it makes only sense with shares.
     *
     * @param uid               The UID
     */
    void setUID( const QString &uid );

    /**
     * This functions returns the UID defined for this item.
     *
     * @returns                 the UID.
     */
    const QString &uid() const { return m_uid; }

    /**
     * With this function you can set the GID you want to use for this item.
     * However, it makes only sense with shares.
     *
     * @param gid               The GID
     */
    void setGID( const QString &gid );

    /**
     * This functions returns the GID defined for this item.
     *
     * @returns                 the GID.
     */
    const QString &gid() const { return m_gid; }

    /**
     * This function returns the type of the network item for which the options
     * have been defined.
     *
     * @returns the type according to the Type enumeration.
     */
    int type();

    /**
     * The Type enumeration.
     */
    enum Type { Share, Host };

#ifndef __FreeBSD__
    /**
     * This function sets the file system that is to be used when mounting the share.
     *
     * Note: This function is not available und FreeBSD.
     *
     * @param fs                the file system name
     */
    void setFilesystem( const QString &fs );

    /**
     * This function returns the file system that is to be used.
     *
     * Note: This function is not available und FreeBSD.
     *
     * @returns                 the file system name
     */
    const QString &filesystem() const { return m_filesystem; }

    /**
     * Set if the share is to be mounted read-write or read-only.
     *
     * Note: This function is not available und FreeBSD.
     *
     * @param rw                TRUE if read-write and FALSE otherwise.
     */
    void setWriteAccess( bool rw );

    /**
     * This functions returns TRUE if the user wants to mount a share read-write
     * otherwise it returns FALSE.
     *
     * Note: This function is not available und FreeBSD.
     *
     * @returns                 TRUE if read-write and FALSE otherwise.
     */
    bool writeAccess() const { return m_write_access; }
#endif

  private:
    /**
     * The share name.
     */
    QString m_name;

    /**
     * Should be remounted?
     */
    bool m_remount;

    /**
     * The port number
     */
    int m_port;

#ifndef __FreeBSD__
    /**
     * The file system
     */
    QString m_filesystem;

    /**
     * Mount read-write or read-only?
     */
    bool m_write_access;
#endif

    /**
     * The protocol
     */
    QString m_protocol;

    /**
     * Use Kerberos or not
     */
    bool m_kerberos;

    /**
     * The UID
     */
    QString m_uid;

    /**
     * The GID
     */
    QString m_gid;
};

#endif
