/***************************************************************************
    smb4knetworkitems  -  Network items used by the Smb4KScanner class
    to pass and store data.
                             -------------------
    begin                : Mi Jun 2 2004
    copyright            : (C) 2004 by Alexander Reinholdt
    email                : dustpuppy@mail.berlios.de
 ***************************************************************************/

/***************************************************************************
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful, but   *
 *   WITHOUT ANY WARRANTY; without even the implied warranty of            *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   General Public License for more details.                              *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,   *
 *   MA  02110-1301 USA                                                    *
 ***************************************************************************/

// KDE includes
#include <klocale.h>
#include <ksocketaddress.h>

// application specific includes
#include "smb4knetworkitems.h"


/****************************************************************************
   Smb4KWorkgroupItem class
****************************************************************************/

Smb4KWorkgroupItem::Smb4KWorkgroupItem( const QString &name, const QString &master, const QString &masterIP )
: m_name( name ), m_master( master ), m_pseudo( false )
{
  m_ip = ipIsValid( masterIP ) ? masterIP : QString::null;
}


Smb4KWorkgroupItem::~Smb4KWorkgroupItem()
{
}


void Smb4KWorkgroupItem::setPseudoMaster()
{
  m_pseudo = true;
}


void Smb4KWorkgroupItem::setMasterIP( const QString &ip )
{
  m_ip = ipIsValid( ip ) ? ip : QString::null;
}


void Smb4KWorkgroupItem::setMaster( const QString &name, const QString &ip, bool pseudo )
{
  m_master = name;
  m_ip = ipIsValid( ip ) ? ip : QString::null;
  m_pseudo = pseudo;
}


bool Smb4KWorkgroupItem::ipIsValid( const QString &ip )
{
  if ( !ip.isEmpty() )
  {
    KNetwork::KIpAddress ip_address = KNetwork::KIpAddress( ip );

    if ( !ip_address.isIPv4Addr() && !ip_address.isIPv6Addr() )
    {
      return false;
    }
  }
  else
  {
    return false;
  }

  return true;
}


/****************************************************************************
   Smb4KHostItem class
****************************************************************************/

Smb4KHostItem::Smb4KHostItem( const QString &workgroup, const QString &name, const QString &comment, const QString &ip )
: m_workgroup( workgroup ), m_name( name ), m_comment( comment ), m_server_string( QString::null ),
  m_os_string( QString::null ), m_master( false ), m_ip_checked( m_ip.stripWhiteSpace().isEmpty() ? false : true ),
  m_info_checked( false )
{
  m_ip = ipIsValid( ip ) ? ip : QString::null;
}


Smb4KHostItem::Smb4KHostItem( const Smb4KHostItem &host )
: m_workgroup( host.workgroup() ), m_name( host.name() ), m_comment( host.comment() ), m_ip( host.ip() ),
  m_server_string( host.serverString() ), m_os_string( host.osString() ), m_master( host.isMaster() ),
  m_ip_checked( host.ipAddressChecked() ), m_info_checked( host.infoChecked() )
{
  // NOTE: We do not check the IP address here, because that has
  // already been done by the copied Smb4KHostItem object.
}


Smb4KHostItem::~Smb4KHostItem()
{
}


void Smb4KHostItem::setServerString( const QString &server )
{
  m_server_string = server;
}


void Smb4KHostItem::setOSString( const QString &os )
{
  m_os_string = os;
}


void Smb4KHostItem::setMaster( bool master )
{
  m_master = master;
}


void Smb4KHostItem::setIPAddress( const QString &ip )
{
  m_ip = ipIsValid( ip ) ? ip : QString::null;
}


void Smb4KHostItem::setComment( const QString &comment )
{
  m_comment = comment;
}


void Smb4KHostItem::setIPAddressChecked( bool yes )
{
  m_ip_checked = yes;
}


void Smb4KHostItem::setInfoChecked( bool yes )
{
  m_info_checked = yes;
}


bool Smb4KHostItem::ipIsValid( const QString &ip )
{
  if ( !ip.isEmpty() )
  {
    KNetwork::KIpAddress ip_address = KNetwork::KIpAddress( ip );

    if ( !ip_address.isIPv4Addr() && !ip_address.isIPv6Addr() )
    {
      return false;
    }
  }
  else
  {
    return false;
  }

  return true;
}


/****************************************************************************
   Smb4KShareItem class
****************************************************************************/

Smb4KShareItem::Smb4KShareItem( const QString &workgroup, const QString &host, const QString &name, const QString &type, const QString &comment )
: m_workgroup( workgroup ), m_host( host ), m_name( name ), m_type( type ), m_comment( comment )
{
}


Smb4KShareItem::~Smb4KShareItem()
{
}


const QString Smb4KShareItem::translatedType() const
{
  QString return_string;

  if ( QString::compare( m_type, "Disk" ) == 0 )
  {
    return_string = i18n( "Disk" );
  }
  else if ( QString::compare( m_type, "Print" ) == 0 || QString::compare( m_type, "Printer" ) == 0 )
  {
    return_string = i18n( "Printer" );
  }
  else
  {
    return_string = m_type;
  }

  return return_string;
}


bool Smb4KShareItem::isHidden() const
{
  return m_name.stripWhiteSpace().endsWith( "$" );
}


bool Smb4KShareItem::isPrinter() const
{
  return (QString::compare( m_type, "Print" ) == 0 || QString::compare( m_type, "Printer" ) == 0);
}


bool Smb4KShareItem::isIPC() const
{
  return (QString::compare( m_name.stripWhiteSpace(), "IPC$" ) == 0);
}


bool Smb4KShareItem::isADMIN() const
{
  return (QString::compare( m_name.stripWhiteSpace(), "ADMIN$" ) == 0);
}

