/***************************************************************************
    smb4khomesshareshandler  -  This class handles the homes shares.
                             -------------------
    begin                : Do Aug 10 2006
    copyright            : (C) 2006 by Alexander Reinholdt
    email                : dustpuppy@mail.berlios.de
 ***************************************************************************/

/***************************************************************************
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful, but   *
 *   WITHOUT ANY WARRANTY; without even the implied warranty of            *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   General Public License for more details.                              *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,   *
 *   MA  02110-1301 USA                                                    *
 ***************************************************************************/

// Qt includes
#include <qmap.h>
#include <qlayout.h>
#include <qlabel.h>
#include <qfile.h>

// KDE includes
#include <kdebug.h>
#include <kstandarddirs.h>
#include <klocale.h>
#include <kiconloader.h>
#include <kcombobox.h>

// application specific includes
#include "smb4khomesshareshandler.h"
#include "smb4kdefs.h"
#include "smb4kerror.h"


Smb4KHomesSharesHandler::Smb4KHomesSharesHandler( QObject *parent, const char *name )
: QObject( parent, name )
{
  // First we need the directory.
  KStandardDirs *stddir = new KStandardDirs();
  QString dir = locateLocal( "data", "smb4k", KGlobal::instance() );

  if ( !stddir->exists( dir ) )
  {
    stddir->makeDir( dir );
  }

  delete stddir;

  m_dlg = NULL;
}


Smb4KHomesSharesHandler::~Smb4KHomesSharesHandler()
{
  delete m_dlg;
}


const QString Smb4KHomesSharesHandler::specifyUser( const QString &host, QWidget *parent, const char *name )
{
  QString username = QString::null;

  m_dlg = new KDialogBase( KDialogBase::Plain, i18n( "Specify User" ), KDialogBase::User1|KDialogBase::Ok|KDialogBase::Cancel, KDialogBase::Ok, parent, name, true, true );
  m_dlg->setButtonGuiItem( KDialogBase::User1, KGuiItem( i18n( "Clear List" ), "editdelete", 0, 0 ) );
  m_dlg->enableButton( KDialogBase::Ok, false );
  m_dlg->enableButton( KDialogBase::User1, false );

  // Set up the ask pass dialog.
  QFrame *frame = m_dlg->plainPage();
  QGridLayout *layout = new QGridLayout( frame );
  layout->setSpacing( 5 );

  QLabel *pic = new QLabel( frame );
  pic->setPixmap( DesktopIcon( "personal" ) );
  pic->setMargin( 10 );

  QLabel *text = new QLabel( i18n( "Please specify a user name." ), frame );

  QLabel *userLabel = new QLabel( i18n( "User:" ), frame );
  KComboBox *userCombo = new KComboBox( true, frame, "UserComboBox" );
  userCombo->setDuplicatesEnabled( false );

  QSpacerItem *spacer1 = new QSpacerItem( 10, 10, QSizePolicy::Expanding, QSizePolicy::Preferred );

  layout->addWidget( pic, 0, 0, 0 );
  layout->addMultiCellWidget( text, 0, 0, 1, 3, 0 );
  layout->addWidget( userLabel, 1, 0, 0 );
  layout->addMultiCellWidget( userCombo, 1, 1, 1, 4, 0 );
  layout->addItem( spacer1, 0, 2 );

  connect( userCombo, SIGNAL( textChanged( const QString &) ),
           this,      SLOT( slotTextChanged( const QString & ) ) );
  connect( m_dlg,     SIGNAL( user1Clicked() ),
           this,      SLOT( slotClearClicked() ) );

  // Read the list of logins, that are already defined
  // for this 'homes' share.
  QStringList list = read_names( host );

  if ( !list.isEmpty() )
  {
    userCombo->insertStringList( list, -1 );
    m_dlg->enableButton( KDialogBase::User1, true );
  }

  userCombo->setCurrentText( QString::null );

  // Do the last things before showing.
  userCombo->setFocus();
  m_dlg->setFixedSize( m_dlg->sizeHint() );

  if ( m_dlg->exec() == KDialogBase::Accepted )
  {
    // First make sure, the list is cleared:
    list.clear();

    // Write the new list of logins to the config file.
    if ( !userCombo->lineEdit()->text().isEmpty() )
    {
      list.append( userCombo->lineEdit()->text() );
    }

    int index = 0;

    while ( index < userCombo->count() )
    {
      if ( list.find( userCombo->text( index ) ) == list.end() )
      {
        list.append( userCombo->text( index ) );
      }

      index++;
    }

    list.sort();

    write_names( host, list );

    username = userCombo->currentText();
  }

  delete m_dlg;
  m_dlg = NULL;

  return username;
}


const QStringList &Smb4KHomesSharesHandler::read_names( const QString &host )
{
  // Clear the old contents of this list:
  m_names.clear();

  QFile file( locateLocal( "data", "smb4k/homes_shares", KGlobal::instance() ) );

  if ( file.open( IO_ReadOnly ) )
  {
    QTextStream ts( &file );
    ts.setEncoding( QTextStream::Locale );

    QString line;
    bool get_names = false;

    while ( !ts.atEnd() )
    {
      line = ts.readLine();

      if ( !get_names )
      {
        if ( QString::compare( line.stripWhiteSpace(), "["+host.upper()+"]" ) == 0 )
        {
          // Found the host:
          get_names = true;

          continue;
        }
        else
        {
          // No match yet...
          continue;
        }
      }
      else
      {
        if ( !line.stripWhiteSpace().isEmpty() )
        {
          // Write the names to the list:
          m_names = QStringList::split( ",", line, false );

          // This is not needed, but let's do it anyway:
          get_names = false;

          break;
        }
      }
    }

    file.close();
  }
  else
  {
    if ( file.exists() )
    {
      Smb4KError::error( ERROR_READING_FILE, file.name() );

      // The list is empty:
      return m_names;
    }
  }

  return m_names;
}


void Smb4KHomesSharesHandler::write_names( const QString &host, const QStringList &names )
{
  // First get the whole contents of the file, so that
  // we can easily modify it:
  QStringList contents;

  QFile file( locateLocal( "data", "smb4k/homes_shares", KGlobal::instance() ) );

  if ( file.open( IO_ReadOnly ) )
  {
    QTextStream ts( &file );
    ts.setEncoding( QTextStream::Locale );

    contents = QStringList::split( '\n', ts.read(), true );

    file.close();
  }
  else
  {
    if ( file.exists() )
    {
      Smb4KError::error( ERROR_READING_FILE, file.name() );

      return;
    }
  }

  // Now search for the host:
  QStringList::Iterator it;

  for ( it = contents.begin(); it != contents.end(); ++it )
  {
    if ( QString::compare( (*it).stripWhiteSpace().upper(), "["+host.upper()+"]" ) == 0 )
    {
      if ( !names.isEmpty() )
      {
        // Move to the line with the names:
        it++;
        // Change the line:
        *it = names.join( "," );
      }
      else
      {
        // Remove the host entry:
        it = contents.remove( it );
        // Remove the user names:
        it = contents.remove( it );

        // Remove the blank line following the entry:
        if ( it != contents.end() && (*it).stripWhiteSpace().isEmpty() )
        {
          contents.remove( it );
        }
      }

      break;
    }
    else
    {
      continue;
    }
  }

  // If we haven't found the host, append it to the end:
  if ( it == contents.end() )
  {
    if ( !contents.isEmpty() )
    {
      contents.append( "" );
    }

    contents.append( "["+host.upper()+"]");
    contents.append( names.join( "," ) );
  }


  // Now write or remove the file:
  if ( !contents.isEmpty() )
  {
    if ( file.open( IO_WriteOnly ) )
    {
      QTextStream ts( &file );
      ts.setEncoding( QTextStream::Locale );

      ts << contents.join( "\n" );

      file.close();
    }
    else
    {
      Smb4KError::error( ERROR_WRITING_FILE, file.name() );

      return;
    }
  }
  else
  {
    file.remove();
  }
}


/////////////////////////////////////////////////////////////////////////////
// SLOT IMPLEMENTATIONS
/////////////////////////////////////////////////////////////////////////////

void Smb4KHomesSharesHandler::slotTextChanged( const QString &text )
{
  if ( !text.isEmpty() )
  {
    m_dlg->enableButtonOK( true );
  }
  else
  {
    m_dlg->enableButtonOK( false );
  }
}


void Smb4KHomesSharesHandler::slotClearClicked()
{
  if ( m_dlg )
  {
    KComboBox *cb = (KComboBox *)m_dlg->child( "UserComboBox", "KComboBox", true );

    if ( cb )
    {
      cb->clearEdit();
      cb->clear();

      m_dlg->enableButton( KDialogBase::User1, false );
    }
  }
}

#include "smb4khomesshareshandler.moc"

