/***************************************************************************
    smb4kbookmark.h  - A bookmark container.
                             -------------------
    begin                : Feb 04 2004
    copyright            : (C) 2004 by Franck Babin
                           (C) 2005-2007 by Alexander Reinholdt
    email                : babinfranck@yahoo.ca
                           dustpuppy@mail.berlios.de
 ***************************************************************************/

/***************************************************************************
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful, but   *
 *   WITHOUT ANY WARRANTY; without even the implied warranty of            *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   General Public License for more details.                              *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,   *
 *   MA  02110-1301 USA                                                    *
 ***************************************************************************/

#ifndef SMB4KBOOKMARK_H
#define SMB4KBOOKMARK_H

// Qt includes
#include <qstring.h>

// forward declarations
class Smb4KShareItem;

/**
 * This class is a container for a set bookmark on a samba share
 */

class Smb4KBookmark
{
  public:
    /**
     * The constructor.
     *
     * @param hostname  The host name.
     *
     * @param sharename The share name.
     *
     * @param workgroup The workgroup/domain where the share is located.
     *
     * @param ip        The IP address of the host
     *
     * @param type      The type of the share, i.e. "disk" or "printer".
     *
     * @param label     An alternative label to the share name.
     */
    Smb4KBookmark( const QString &hostname,
                   const QString &sharename,
                   const QString &workgroup,
                   const QString &ip,
                   const QString &type,
                   const QString &label = QString::null );

    /**
     * An alternative constructor.
     *
     * @param item      The Smb4KShareItem for which a bookmark should be
     *                  created.
     *
     * @param ip        The IP address of the host
     *
     * @param label     An alternative label to the share name.
     */
    Smb4KBookmark( Smb4KShareItem *item,
                   const QString &ip,
                   const QString &label = QString::null );

    /**
     * The empty constructor.
     */
    Smb4KBookmark() {}

    /**
     * The destructor.
     */
    ~Smb4KBookmark();

    /**
     * This function returns the share name.
     *
     * @returns         The share name
     */
    const QString &share() const { return m_share; };

    /**
     * This function returns the workgroup/domain the share is located in.
     *
     * @returns         The workgroup
     */
    const QString &workgroup() const { return m_workgroup; }

    /**
     * This function returns the IP address of the host that carries the
     * share.
     *
     * @returns         The IP address
     */
    const QString &ip() const { return m_ip; }

    /**
     * This function returns the type of the share, i.e. either "Disk" or
     * "Printer"/"Print" and "IPC".
     *
     * @returns         The type of the share
     */
    const QString &type() const { return m_type; }

    /**
     * This function sets the share name of the bookmark. It is normally not
     * necessary to use it, because all data should be passed to the constructor.
     * In case of homes shares, however, this function is useful.
     *
     * @param name      The share name
     */
    void setShareName( const QString &name );

    /**
     * This function returns the bookmark name.
     *
     * @returns         The name of the bookmark
     */
    const QString &bookmark() const { return m_bookmark; }

    /**
     * This function returns the host name.
     *
     * @returns         The name of the host
     */
    const QString &host() const { return m_host; }

    /**
     * This function sets the IP address.
     *
     * @param ip        The IP address
     */
    void setIP( const QString &ip );

    /**
     * Return the alternative bookmark label.
     *
     * @returns the label.
     */
    const QString &label() const { return m_label; }

    /**
     * Sets the alternative bookmark label.
     *
     * @param text      The new text for the label
     */
    void setLabel( const QString &text );

  private:
    /**
     * The host name.
     */
    QString m_host;

    /**
     * The share name.
     */
    QString m_share;

    /**
     * The workgroup
     */
    QString m_workgroup;

    /**
     * The IP address
     */
    QString m_ip;

    /**
     * The type of the share;
     */
    QString m_type;

    /**
     * The bookmark string.
     */
    QString m_bookmark;

    /**
     * The alternative label
     */
    QString m_label;

    /**
     * This function checks if the IP address is valid, i.e. the
     * IP address is either IP v4 or IP v6. It returns either TRUE
     * or FALSE.
     *
     * @param ip            The IP address that's going to be checked.
     *
     * @returns TRUE if the IP address is valid and FALSE otherwise.
     */
    bool ipIsValid( const QString &ip );
};

#endif
