/***************************************************************************
 *   Copyright (C) 2003 by Sylvain Joyeux                                  *
 *   sylvain.joyeux@m4x.org                                                *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 ***************************************************************************/

#include "parsers.h"
#include "../apt.h"

#include <klocale.h>
#include <qregexp.h>

static const QString
  html_attribute_begin("<tr><td class=\"attname\">%1</td>\n\t<td>\n\t\t"),
  html_attribute_classed("<tr class=\"%1\"><td class=\"attname\">%2</td>\n\t<td>\n\t\t"),
  html_attribute_end("\n\t</td>\n</tr>\n");

// Converts the special chars in orig into its HTML equivalents
static QString text2html(const QString& orig)
{ QString ret(orig);
  ret = ret.replace("<(?!a href)", "&lt;");
  //ret = ret.replace( QRegExp("\n"), "<br />\n");
  return ret;
}

static void close_indent(int indent, QString& buffer)
{
  if (buffer.isEmpty()) return;
  if (indent)
    buffer += "\n\t\t</div>";
}
static void close_item(int indent, QString& buffer)
{
  if (buffer.isEmpty()) return;
  close_indent(indent, buffer);
  buffer += html_attribute_end;
}

static void close_table(const QString& version, int indent, QString& buffer)
{
  if (buffer.isEmpty()) return;
  close_item(indent, buffer);
  buffer = version + "<table class=\"version\">\n" + buffer + "</table>\n";
}

static QString version_header_link(const KURL& url, const QString& name)
{ return QString("\t<a class=\"links\" href=\"" + url.htmlURL() + "\">" + name + "</a>\n"); }

namespace Parsers
{

Show::Show(const QString& package, const QString& installed_version, bool act)
  : m_package(package), m_installed(installed_version), m_act(false)
{
  m_act = act;
}

void Show::operator() (AptProtocol* slave, const QString& tag, const QString& value)
{
  // Since we want to show the version first, we should wait for it
  // before sending anything
  static QString version;
  static QString buffer;
  static QString attribute;
  static int indent;
  static bool multiline = false;
  static bool first_line = false, new_paragraph = true;

  if (tag == "begin" || tag == "package" || tag == "end")
  {
    if (multiline && !new_paragraph)
      buffer += "</p>";

    if (tag == "begin")
      m_result_count = 0;
    else
    {
      m_result_count += (tag == "package");

      close_table(version, indent, buffer);
      if (!buffer.isEmpty()) *slave << buffer;
    }

    // init the state variables for the next package
    buffer = "";
    indent = 0;
    first_line = false;
    new_paragraph = true;
  }
  else if (tag == "field")
  {
    if (multiline && !new_paragraph)
      buffer += "</p>";

    attribute = value;
    if (value == "Depends" || value == "Description")
    {
      close_item(indent, buffer);
      buffer +=
      	html_attribute_begin.arg(value)
        + "&nbsp;";

      close_item(indent, buffer);
      buffer += html_attribute_begin.arg("&nbsp;");
    }
    else if (value != "Version" && value != "Package")
    {
      close_item(indent, buffer);
      buffer += html_attribute_begin.arg(value);
    }

    if (value == "Description")
    {
      multiline = true;
      new_paragraph = true;
      first_line = true;
    }
    else
    {
      multiline = false;
    }
  }
  else if (tag == "indent")
  {
    close_indent(indent, buffer);

    int new_indent = value.toInt();
    if (new_indent)
      buffer += "\n\t<div style=\"margin-left: " + value + "em\">";
    indent = new_indent;
  }
  else if (tag == "data" && attribute == "Version")
  {
    KURL action ("apt:/get");

    QString item_id = mangle_version(value);
    if (value == m_installed)
    {
      action.setQuery("remove");
      version = QString("<div class=\"version-header-installed\" id=\"%1\">").arg(item_id)
                + i18n("Installed version %1").arg(value);
    }
    else
    {
      action.setQuery("install");
      version = QString("<div class=\"version-header\" id=\"%1\">").arg(item_id)
                + i18n("Version %1").arg(value);
    }

    action.addQueryItem("package", m_package);
    action.addQueryItem("version", value);

    if (m_act)
    {
      if (value == m_installed)
        version += version_header_link(action, i18n("Remove"));
      else
        version += version_header_link(action, i18n("Install"));
    }
    version += "</div>\n";

  }
  else if (tag == "data")
  {
    if (multiline)
    {
      static const QRegExp rx_empty("^\\s*$");
      if (rx_empty.exactMatch(value))
      {
        buffer += "</p>";
        new_paragraph = true;
      }
      else if (first_line)
      {
        new_paragraph = true;
        first_line = false;
      }
      else if (new_paragraph)
      {
        buffer += "<p>";
        new_paragraph = false;
      }
    }
    buffer += text2html(QString(value).replace(QRegExp("(http://\\S+)"),QString("<a href=\"\\1\">\\1</a>")));
  }
  else if (tag == "package_link")
    buffer += "<a href=\"" + slave->buildURL("show", value).htmlURL() + "\">" + value + "</a>";
}

}
