/*
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
*/

/*
  Copyright (C) 2002 Dario Abatianni <eisfuchs@tigress.com>
  Copyright (C) 2005 Ismail Donmez <ismail@kde.org>
  Copyright (C) 2005 Peter Simonsson <psn@linux.se>
  Copyright (C) 2005 John Tapsell <johnflux@gmail.com>
  Copyright (C) 2005-2008 Eike Hein <hein@kde.org>
*/

#include "konversationmainwindow.h"
#include "konversationapplication.h"
#include "linkaddressbook/addressbook.h"
#include "konvisettingsdialog.h"
#include "viewcontainer.h"
#include "konversationstatusbar.h"
#include "konvibookmarkhandler.h"
#include "trayicon.h"
#include "serverlistdialog.h"
#include "identitydialog.h"
#include "notificationhandler.h"
#include "irccharsets.h"
#include "connectionmanager.h"
#include "awaymanager.h"
#include "dcctransfermanager.h"

#include <tqnamespace.h>
#include <tqwhatsthis.h>
#include <tqsignalmapper.h>
#include <tqobjectlist.h>

#include <kaccel.h>
#include <kaccelmanager.h>
#include <kstdaction.h>
#include <kaction.h>
#include <kdebug.h>
#include <klocale.h>
#include <kmessagebox.h>
#include <kmenubar.h>
#include <kkeydialog.h>
#include <kdeversion.h>
#include <kedittoolbar.h>
#include <kpopupmenu.h>
#include <kwin.h>
#include <kglobal.h>
#include <kstandarddirs.h>
#include <dcopclient.h>
#include <scriptmanager.h>
#include <kabc/addressbook.h>
#include <kabc/errorhandler.h>

#include <config.h>
#ifdef  USE_KNOTIFY
#include <knotifydialog.h>
#endif


KonversationMainWindow::KonversationMainWindow() : KMainWindow(0,"main_window", WStyle_ContextHelp | WType_TopLevel | WDestructiveClose)
{
    m_hasDirtySettings = false;
    m_closeApp = false;
    m_serverListDialog = 0;
    m_settingsDialog = NULL;

    m_viewContainer = new ViewContainer(this);
    setCentralWidget(m_viewContainer->getWidget());

    //used for event compression. See header file for resetHasDirtySettings()
    connect(KonversationApplication::instance(), TQT_SIGNAL(appearanceChanged()), this, TQT_SLOT(resetHasDirtySettings()));
    connect(KonversationApplication::instance(), TQT_SIGNAL(appearanceChanged()), this, TQT_SLOT(updateTrayIcon()));


    // Set up view container
    connect(KonversationApplication::instance(), TQT_SIGNAL(appearanceChanged()), m_viewContainer, TQT_SLOT(updateAppearance()));
    connect(KonversationApplication::instance(), TQT_SIGNAL(iconChanged(int)), m_viewContainer, TQT_SLOT(updateViewIcons()));
    connect(KonversationApplication::instance(), TQT_SIGNAL(serverGroupsChanged(const Konversation::ServerGroupSettings*)),
            m_viewContainer, TQT_SLOT(updateViews(const Konversation::ServerGroupSettings*)));
    connect(m_viewContainer, TQT_SIGNAL(autoJoinToggled(const Konversation::ServerGroupSettings*)),
            KonversationApplication::instance(), TQT_SIGNAL(serverGroupsChanged(const Konversation::ServerGroupSettings*)));
    connect(m_viewContainer, TQT_SIGNAL(setWindowCaption(const TQString&)), this, TQT_SLOT(setCaption(const TQString&)));
    connect(KonversationApplication::instance()->getConnectionManager(),
            TQT_SIGNAL(connectionChangedState(Server*, Konversation::ConnectionState)),
            m_viewContainer, TQT_SLOT(connectionStateChanged(Server*, Konversation::ConnectionState)));
    connect(this, TQT_SIGNAL(triggerRememberLine()), m_viewContainer, TQT_SLOT(insertRememberLine()));
    connect(this, TQT_SIGNAL(triggerRememberLines(Server*)), m_viewContainer, TQT_SLOT(insertRememberLines(Server*)));
    connect(this, TQT_SIGNAL(cancelRememberLine()), m_viewContainer, TQT_SLOT(cancelRememberLine()));
    connect(this, TQT_SIGNAL(insertMarkerLine()), m_viewContainer, TQT_SLOT(insertMarkerLine()));

    // Set up status bar
    m_statusBar = new KonversationStatusBar(this);
    connect(KonversationApplication::instance(), TQT_SIGNAL(appearanceChanged()), m_statusBar, TQT_SLOT(updateAppearance()));

    createStandardStatusBarAction();

    connect(actionCollection(), TQT_SIGNAL(actionStatusText(const TQString&)), m_statusBar, TQT_SLOT(setMainLabelTempText(const TQString&)));
    connect(actionCollection(), TQT_SIGNAL(clearStatusText()), m_statusBar, TQT_SLOT(clearMainLabelTempText()));
    actionCollection()->setHighlightingEnabled(true);

    connect(m_viewContainer, TQT_SIGNAL(resetStatusBar()), m_statusBar, TQT_SLOT(resetStatusBar()));
    connect(m_viewContainer, TQT_SIGNAL(setStatusBarTempText(const TQString&)), m_statusBar, TQT_SLOT(setMainLabelTempText(const TQString&)));
    connect(m_viewContainer, TQT_SIGNAL(clearStatusBarTempText()), m_statusBar, TQT_SLOT(clearMainLabelTempText()));
    connect(m_viewContainer, TQT_SIGNAL(setStatusBarInfoLabel(const TQString&)), m_statusBar, TQT_SLOT(updateInfoLabel(const TQString&)));
    connect(m_viewContainer, TQT_SIGNAL(clearStatusBarInfoLabel()), m_statusBar, TQT_SLOT(clearInfoLabel()));
    connect(m_viewContainer, TQT_SIGNAL(setStatusBarLagLabelShown(bool)), m_statusBar, TQT_SLOT(setLagLabelShown(bool)));
    connect(m_viewContainer, TQT_SIGNAL(updateStatusBarLagLabel(Server*, int)), m_statusBar, TQT_SLOT(updateLagLabel(Server*, int)));
    connect(m_viewContainer, TQT_SIGNAL(resetStatusBarLagLabel()), m_statusBar, TQT_SLOT(resetLagLabel()));
    connect(m_viewContainer, TQT_SIGNAL(setStatusBarLagLabelTooLongLag(Server*, int)), m_statusBar, TQT_SLOT(setTooLongLag(Server*, int)));
    connect(m_viewContainer, TQT_SIGNAL(updateStatusBarSSLLabel(Server*)), m_statusBar, TQT_SLOT(updateSSLLabel(Server*)));
    connect(m_viewContainer, TQT_SIGNAL(removeStatusBarSSLLabel()), m_statusBar, TQT_SLOT(removeSSLLabel()));


    // Actions
    KStdAction::quit(this,TQT_SLOT(quitProgram()),actionCollection());

    hideMenuBarAction = KStdAction::showMenubar(this, TQT_SLOT(toggleMenubar()), actionCollection());

    setStandardToolBarMenuEnabled(true);
    KStdAction::configureToolbars(this, TQT_SLOT(configureToolbar()), actionCollection());

    KStdAction::keyBindings(this, TQT_SLOT(openKeyBindings()), actionCollection());
    KAction *preferencesAction = KStdAction::preferences(this, TQT_SLOT(openPrefsDialog()), actionCollection());

#ifdef USE_KNOTIFY // options_configure_notifications
    KAction *configureNotificationsAction = KStdAction::configureNotifications(this,TQT_SLOT(openNotifications()), actionCollection());
#endif

    KAction* action;

    (new KAction(i18n("&Server List..."), "server", KShortcut("F2"), this, TQT_SLOT(openServerList()),
        actionCollection(), "open_server_list"))->setToolTip(i18n("Manage networks and servers"));
    (new KAction(i18n("Quick &Connect..."), "connect_creating", KShortcut("F7"), this, TQT_SLOT(openQuickConnectDialog()),
        actionCollection(), "quick_connect_dialog"))->setToolTip(i18n("Type in the address of a new IRC server to connect to"));

    action = new KAction(i18n("&Reconnect"), "connect_creating", 0, m_viewContainer, TQT_SLOT(reconnectFrontServer()), actionCollection(), "reconnect_server");
    action->setEnabled(false);
    action->setToolTip(i18n("Reconnect to the current server."));

    action = new KAction(i18n("&Disconnect"), "connect_no", 0, m_viewContainer, TQT_SLOT(disconnectFrontServer()), actionCollection(), "disconnect_server");
    action->setEnabled(false);
    action->setToolTip(i18n("Disconnect from the current server."));

    (new KAction(i18n("&Identities..."), "identity", KShortcut("F8"), this, TQT_SLOT(openIdentitiesDialog()),
        actionCollection(), "identities_dialog"))->setToolTip(i18n("Manage your nick, away and other identity settings"));

    new KToggleAction(i18n("&Watched Nicks Online"), "kontact_contacts", KShortcut("F4"), m_viewContainer, TQT_SLOT(openNicksOnlinePanel()), actionCollection(), "open_nicksonline_window");
    new KToggleAction(i18n("&DCC Status"), "2rightarrow", KShortcut("F9"), m_viewContainer, TQT_SLOT(toggleDccPanel()), actionCollection(), "open_dccstatus_window");
    action = new KAction(i18n("&Open Logfile"), "history", KShortcut("Ctrl+O"), m_viewContainer, TQT_SLOT(openLogFile()), actionCollection(), "open_logfile");
    action->setEnabled(false);
    action->setToolTip(i18n("Open the known history for this channel in a new tab"));

    action = new KAction(i18n("&Channel Settings..."), "edit", m_viewContainer, TQT_SLOT(openChannelSettings()), actionCollection(), "channel_settings");
    action->setEnabled(false);
    action->setToolTip(i18n("Open the channel settings dialog for this tab"));

    KToggleAction* channelListAction = new KToggleAction(i18n("Channel &List"), "view_text", KShortcut("F5"), m_viewContainer, TQT_SLOT(openChannelList()), actionCollection(), "open_channel_list");
    channelListAction->setEnabled(false);
    channelListAction->setToolTip(i18n("Show a list of all the known channels on this server"));

    action = new KToggleAction(i18n("&URL Catcher"), "enhanced_browsing", KShortcut("F6"), m_viewContainer, TQT_SLOT(addUrlCatcher()), actionCollection(), "open_url_catcher");
    action->setToolTip(i18n("List all URLs that have been mentioned recently in a new tab"));

    if (kapp->authorize("shell_access"))
    {
        action = new KAction(i18n("New &Konsole"), "openterm", 0, m_viewContainer, TQT_SLOT(addKonsolePanel()), actionCollection(), "open_konsole");
        action->setToolTip(i18n("Open a terminal in a new tab"));
    }

    // Actions to navigate through the different pages
    KShortcut nextShortcut = KStdAccel::tabNext();
    nextShortcut.setSeq(1, KKeySequence("Alt+Right"));
    KShortcut prevShortcut = KStdAccel::tabPrev();
    prevShortcut.setSeq(1, KKeySequence("Alt+Left"));
    action = new KAction(i18n("&Next Tab"), TQApplication::reverseLayout() ? "previous" : "next",
        TQApplication::reverseLayout() ? prevShortcut : nextShortcut,
        m_viewContainer, TQT_SLOT(showNextView()), actionCollection(), "next_tab");
    action->setEnabled(false);
    action = new KAction(i18n("&Previous Tab"), TQApplication::reverseLayout() ? "next" : "previous",
        TQApplication::reverseLayout() ? nextShortcut : prevShortcut,
        m_viewContainer, TQT_SLOT(showPreviousView()),actionCollection(),"previous_tab");
    action->setEnabled(false);
    action = new KAction(i18n("Close &Tab"),"tab_remove",KShortcut("Ctrl+w"), m_viewContainer, TQT_SLOT(closeCurrentView()),actionCollection(),"close_tab");
    action->setEnabled(false);
    action = new KAction(i18n("Next Active Tab"), 0, KShortcut("Ctrl+Alt+Space"), m_viewContainer, TQT_SLOT(showNextActiveView()),
                         actionCollection(), "next_active_tab");
    action->setEnabled(false);

    if (Preferences::tabPlacement()==Preferences::Left)
    {
        action = new KAction(i18n("Move Tab Up"), "1uparrow", KShortcut("Alt+Shift+Left"),
            m_viewContainer, TQT_SLOT(moveViewLeft()), actionCollection(), "move_tab_left");
        action->setEnabled(false);
        action->setToolTip("Move this tab");
        action = new KAction(i18n("Move Tab Down"), "1downarrow", KShortcut("Alt+Shift+Right"),
            m_viewContainer, TQT_SLOT(moveViewRight()), actionCollection(), "move_tab_right");
        action->setEnabled(false);
        action->setToolTip("Move this tab");
    }
    else
    {
        KAction* action2;
        if (TQApplication::reverseLayout())
        {
            action2 = new KAction(i18n("Move Tab Right"), "1rightarrow", KShortcut("Alt+Shift+Right"),
                          m_viewContainer, TQT_SLOT(moveViewLeft()), actionCollection(), "move_tab_left");
            action  = new KAction(i18n("Move Tab Left"), "1leftarrow", KShortcut("Alt+Shift+Left"),
                          m_viewContainer, TQT_SLOT(moveViewRight()), actionCollection(), "move_tab_right");
        }
        else
        {
            action  = new KAction(i18n("Move Tab Left"), "1leftarrow", KShortcut("Alt+Shift+Left"),
                          m_viewContainer, TQT_SLOT(moveViewLeft()), actionCollection(), "move_tab_left");
                action2 = new KAction(i18n("Move Tab Right"), "1rightarrow", KShortcut("Alt+Shift+Right"),
                          m_viewContainer, TQT_SLOT(moveViewRight()), actionCollection(), "move_tab_right");
        }

        action->setEnabled(false);
        action->setToolTip("Move this tab");
        action2->setEnabled(false);
        action2->setToolTip("Move this tab");
    }

    action = new KAction(i18n("Rejoin Channel"), 0, m_viewContainer, TQT_SLOT(rejoinChannel()), actionCollection(), "rejoin_channel");
    action->setEnabled(false);

    action = new KToggleAction(i18n("Enable Notifications"), 0, 0, m_viewContainer, TQT_SLOT(toggleViewNotifications()), actionCollection(), "tab_notifications");
    action->setEnabled(false);

    action = new KToggleAction(i18n("Join on Connect"), 0, 0, m_viewContainer, TQT_SLOT(toggleAutoJoin()), actionCollection(), "tab_autojoin");
    action->setEnabled(false);

    KSelectAction* selectAction = new KSelectAction(i18n("Set Encoding"), "charset", 0, actionCollection(), "tab_encoding");
    selectAction->setEditable(false);
    TQStringList encodingDescs = Konversation::IRCCharsets::self()->availableEncodingDescriptiveNames();
    encodingDescs.prepend(i18n("Default"));
    selectAction->setItems(encodingDescs);
    selectAction->setEnabled(false);
    connect(selectAction, TQT_SIGNAL(activated(int)), m_viewContainer, TQT_SLOT(changeViewCharset(int)));

    TQSignalMapper* tabSelectionMapper = new TQSignalMapper(this);
    connect(tabSelectionMapper, TQT_SIGNAL(mapped(int)), m_viewContainer, TQT_SLOT(goToView(int)));

    for (uint i = 1; i <= 10; ++i)
    {
        KAction* tabSelectionAction = new KAction(i18n("Go to Tab %1").arg(i), 0, KShortcut(TQString("Alt+%1").arg(i%10)),
            tabSelectionMapper, TQT_SLOT(map()), actionCollection(), TQString("go_to_tab_%1").arg(i).local8Bit());
        tabSelectionMapper->setMapping( tabSelectionAction, i-1);
    }

    action = new KAction(i18n("Clear &Marker Lines"), 0, KShortcut("CTRL+SHIFT+R"), m_viewContainer, TQT_SLOT(clearViewLines()),actionCollection(),"clear_lines");
    action->setToolTip(i18n("Clear marker lines in the current tab"));
    action->setEnabled(false);
    action = new KAction(i18n("&Clear Window"), 0, KShortcut("Ctrl+L"), m_viewContainer, TQT_SLOT(clearView()),actionCollection(),"clear_window");
    action->setToolTip(i18n("Clear the contents of the current tab"));
    action->setEnabled(false);
    action = new KAction(i18n("Clear &All Windows"),0,KShortcut("CTRL+SHIFT+L"), m_viewContainer, TQT_SLOT(clearAllViews()),actionCollection(),"clear_tabs");
    action->setToolTip(i18n("Clear the contents of all open tabs"));
    action->setEnabled(false);

    KToggleAction* awayAction = new KToggleAction(i18n("Global Away"), KShortcut("Ctrl+Shift+A"), actionCollection(), "toggle_away");
    connect(awayAction, TQT_SIGNAL(toggled(bool)), static_cast<KonversationApplication*>(kapp)->getAwayManager(), TQT_SLOT(toggleGlobalAway(bool)));
    awayAction->setEnabled(false);

    action = new KAction(i18n("&Join Channel..."), "add", KShortcut("Ctrl+J"), m_viewContainer, TQT_SLOT(showJoinChannelDialog()), actionCollection(), "join_channel");
    action->setEnabled(false);
    action->setToolTip("Join a new channel on this server");

    action = KStdAction::find(m_viewContainer, TQT_SLOT(findText()), actionCollection());
    action->setEnabled(false);
    action = KStdAction::findNext(m_viewContainer, TQT_SLOT(findNextText()), actionCollection());
    action->setEnabled(false);
    action = KStdAction::findPrev(m_viewContainer, TQT_SLOT(findPrevText()), actionCollection());
    action->setEnabled(false);

    action = new KAction(i18n("&IRC Color..."), "colorize", CTRL+Key_K, m_viewContainer, TQT_SLOT(insertIRCColor()), actionCollection(), "irc_colors");
    action->setToolTip(i18n("Set the color of your current IRC message"));
    action->setEnabled(false);
    action = new KAction(i18n("&Marker Line"), 0,  KShortcut("Ctrl+R") , m_viewContainer, TQT_SLOT(insertMarkerLine()), actionCollection(), "insert_marker_line");
    action->setToolTip(i18n("Insert a horizontal line into the current tab that only you can see"));
    action->setEnabled(false);
    action = new KAction(i18n("Special &Character..."), "char", KShortcut("Alt+Shift+C"), m_viewContainer, TQT_SLOT(insertCharacter()), actionCollection(), "insert_character");
    action->setToolTip(i18n("Insert any character into your current IRC message"));
    action->setEnabled(false);

    action = new KAction(i18n("Close &All Open Queries"), 0, KShortcut("F11"), m_viewContainer, TQT_SLOT(closeQueries()), actionCollection(), "close_queries");
    action->setEnabled(false);

    KToggleAction* toggleChannelNickListsAction = new KToggleAction(i18n("Hide Nicklist"), 0,
        KShortcut("Ctrl+H"), m_viewContainer, TQT_SLOT(toggleChannelNicklists()), actionCollection(), "hide_nicknamelist");
    if (!Preferences::showNickList())
        toggleChannelNickListsAction->setChecked(true);

    // set up system tray
    m_trayIcon = new Konversation::TrayIcon(this);
    connect(this, TQT_SIGNAL(endNotification()), m_trayIcon, TQT_SLOT(endNotification()));
    connect(KonversationApplication::instance(), TQT_SIGNAL(iconChanged(int)), m_trayIcon, TQT_SLOT(updateAppearance()));
    connect(m_trayIcon, TQT_SIGNAL(quitSelected()), this, TQT_SLOT(quitProgram()));
    KPopupMenu *trayMenu = m_trayIcon->contextMenu();
    #ifdef USE_KNOTIFY
    configureNotificationsAction->plug(trayMenu);
    #endif
    preferencesAction->plug(trayMenu);
    awayAction->plug(trayMenu);

    // decide whether to show the tray icon or not
    updateTrayIcon();

    createGUI(NULL, false);

    resize(700, 500);                             // Give the app a sane default size
    setAutoSaveSettings();

    // Apply menubar show/hide pref
    hideMenuBarAction->setChecked(Preferences::showMenuBar());
    toggleMenubar(true);

    // Bookmarks
    m_bookmarkHandler = new KonviBookmarkHandler(this);

    // set up KABC with a nice gui error dialog
    KABC::GuiErrorHandler *m_guiErrorHandler = new KABC::GuiErrorHandler(this);
    kapp->dcopClient()->setAcceptCalls( false );
    Konversation::Addressbook::self()->getAddressBook()->setErrorHandler(m_guiErrorHandler);
    kapp->dcopClient()->setAcceptCalls( true );

    if (Preferences::useNotify() && Preferences::openWatchedNicksAtStartup())
        m_viewContainer->openNicksOnlinePanel();

}

KonversationMainWindow::~KonversationMainWindow()
{
    delete m_viewContainer;
}

int KonversationMainWindow::confirmQuit()
{
    KonversationApplication* konvApp = static_cast<KonversationApplication*>(kapp);

    if (konvApp->getConnectionManager()->connectionCount() == 0)
        return KMessageBox::Continue;

    int result = KMessageBox::Cancel;

    if (!KMessageBox::shouldBeShownContinue("systemtrayquitKonversation")
         && konvApp->getDccTransferManager()->hasActiveTransfers())
    {
        result = KMessageBox::warningContinueCancel(
            this,
            i18n("<qt>You have active DCC file transfers. Are you sure you want to quit <b>Konversation</b>?</qt>"),
            i18n("Confirm Quit"),
            i18n("Quit"),
            "QuitWithActiveDccTransfers");
    }
    else
    {
        result = KMessageBox::warningContinueCancel(
            this,
            i18n("<qt>Are you sure you want to quit <b>Konversation</b>?</qt>"),
            i18n("Confirm Quit"),
            i18n("Quit"),
            "systemtrayquitKonversation");
    }

    return result;
}

void KonversationMainWindow::quitProgram()
{
    if (Preferences::showTrayIcon() &&
        sender() != m_trayIcon &&
        confirmQuit() == KMessageBox::Cancel) return;

    // will call queryClose()
    m_closeApp = true;
    close();
}

bool KonversationMainWindow::queryClose()
{
    KonversationApplication* konvApp = static_cast<KonversationApplication*>(kapp);

    if (!konvApp->sessionSaving())
    {
        if (sender() == m_trayIcon)
            m_closeApp = true;

        if (Preferences::showTrayIcon() && !m_closeApp)
        {
            KMessageBox::information( this,
                i18n("<p>Closing the main window will keep Konversation running in the system tray. "
                "Use <b>Quit</b> from the <b>Konversation</b> menu to quit the application.</p>"),
                i18n( "Docking in System Tray" ),  "HideOnCloseInfo" );
            hide();

            return false;
        }

        if (!Preferences::showTrayIcon() && confirmQuit() == KMessageBox::Cancel)
            return false;
    }

    return true;
}

void KonversationMainWindow::hideEvent(TQHideEvent *e)
{
    emit triggerRememberLine();

    m_statusBar->clearMainLabelTempText();

    KMainWindow::hideEvent(e);
}

void KonversationMainWindow::showEvent(TQShowEvent *e)
{
    emit cancelRememberLine();

    KMainWindow::showEvent(e);
}

void KonversationMainWindow::leaveEvent(TQEvent* e)
{
    m_statusBar->clearMainLabelTempText();

    KMainWindow::leaveEvent(e);
}

bool KonversationMainWindow::event(TQEvent* e)
{
    if (e->type() == TQEvent::WindowActivate)
    {
        emit endNotification();
        emit cancelRememberLine();
    }
    else if(e->type() == TQEvent::WindowDeactivate)
    {
        m_statusBar->clearMainLabelTempText();

        if (kapp->activeModalWidget() == 0)
            emit triggerRememberLine();
    }

    return KMainWindow::event(e);
}

void KonversationMainWindow::settingsChangedSlot()
{
    // This is for compressing the events. m_hasDirtySettings is set to true
    // when the settings have changed, then set to false when the app reacts to it
    // via the appearanceChanged signal.  This prevents a series of settingsChanged signals
    // causing the app expensively rereading its settings many times.
    // The appearanceChanged signal is connected to resetHasDirtySettings to reset this bool
    if (!m_hasDirtySettings) 
    {
        TQTimer::singleShot(0, KonversationApplication::instance(), TQT_SIGNAL(appearanceChanged()));
        m_hasDirtySettings = true;
    }
}

void KonversationMainWindow::resetHasDirtySettings()
{
    m_hasDirtySettings = false;
}

void KonversationMainWindow::updateTrayIcon()
{
    m_trayIcon->setNotificationEnabled(Preferences::trayNotify());

    if (Preferences::showTrayIcon())
        m_trayIcon->show();
    else
        m_trayIcon->hide();
}

void KonversationMainWindow::toggleMenubar(bool dontShowWarning)
{
    if (hideMenuBarAction->isChecked())
        menuBar()->show();
    else
    {
        if (!dontShowWarning)
        {
            TQString accel = hideMenuBarAction->shortcut().toString();
            KMessageBox::information(this,
                i18n("<qt>This will hide the menu bar completely. You can show it again by typing %1.</qt>").arg(accel),
                "Hide menu bar","HideMenuBarWarning");
        }
        menuBar()->hide();
    }

    Preferences::setShowMenuBar(hideMenuBarAction->isChecked());
}

int KonversationMainWindow::configureToolbar()
{
    saveMainWindowSettings(KGlobal::config());
    KEditToolbar dlg(actionCollection(), xmlFile(), true, this);
    connect(&dlg, TQT_SIGNAL(newToolbarConfig()), TQT_SLOT(saveToolbarConfig()));
    return dlg.exec();
}

void KonversationMainWindow::saveToolbarConfig()
{
    createGUI(xmlFile(), false);
    applyMainWindowSettings(KGlobal::config());
}

void KonversationMainWindow::focusAndShowErrorMessage(const TQString &errorMsg)
{
    show();
    KWin::demandAttention(winId());
    KWin::activateWindow(winId());
    KMessageBox::error(this, errorMsg);
}

void KonversationMainWindow::openPrefsDialog()
{
    //An instance of your dialog could be already created and could be cached,
    //in which case you want to display the cached dialog instead of creating
    //another one
    if (!m_settingsDialog)
    {
        m_settingsDialog = new KonviSettingsDialog(this);
        //User edited the configuration - update your local copies of the
        //configuration data
        connect(m_settingsDialog, TQT_SIGNAL(settingsChanged()), this, TQT_SLOT(settingsChangedSlot()));
    }
    m_settingsDialog->show();
}

void KonversationMainWindow::openKeyBindings()
{
    // Change a number of action names to make them friendlier for the shortcut list.
    actionCollection()->action("tab_notifications")->setText(i18n("Toggle Notifications"));
    actionCollection()->action("toggle_away")->setText(i18n("Toggle Away Globally"));
    actionCollection()->action("irc_colors")->setText(i18n("Insert &IRC Color..."));
    actionCollection()->action("insert_character")->setText(i18n("Insert Special &Character..."));
    actionCollection()->action("insert_marker_line")->setText(i18n("Insert &Marker Line"));
    TQString openChannelListString = actionCollection()->action("open_channel_list")->text();
    actionCollection()->action("open_channel_list")->setText(i18n("&Channel List"));
    TQString openLogFileString = actionCollection()->action("open_logfile")->text();
    actionCollection()->action("open_logfile")->setText(i18n("&Open Logfile"));

    // Open shortcut configuration dialog.
    KKeyDialog::configure(actionCollection());

    // Reset action names.
    actionCollection()->action("tab_notifications")->setText(i18n("Enable Notifications"));
    actionCollection()->action("toggle_away")->setText(i18n("Set &Away Globally"));
    actionCollection()->action("irc_colors")->setText(i18n("&IRC Color..."));
    actionCollection()->action("insert_character")->setText(i18n("Special &Character..."));
    actionCollection()->action("insert_marker_line")->setText(i18n("&Marker Line"));
    actionCollection()->action("open_channel_list")->setText(openChannelListString);
    actionCollection()->action("open_logfile")->setText(openLogFileString);
}

void KonversationMainWindow::openServerList()
{
    if (!m_serverListDialog)
    {
        m_serverListDialog = new Konversation::ServerListDialog(this);
        KonversationApplication* konvApp = static_cast<KonversationApplication*>(kapp);

        connect(m_serverListDialog, TQT_SIGNAL(serverGroupsChanged(const Konversation::ServerGroupSettings*)),
                konvApp, TQT_SIGNAL(serverGroupsChanged(const Konversation::ServerGroupSettings*)));
        connect(konvApp, TQT_SIGNAL(serverGroupsChanged(const Konversation::ServerGroupSettings*)),
                m_serverListDialog, TQT_SLOT(updateServerList()));
        connect(m_serverListDialog, TQT_SIGNAL(connectTo(Konversation::ConnectionFlag, int)),
                konvApp->getConnectionManager(), TQT_SLOT(connectTo(Konversation::ConnectionFlag, int)));
        connect(m_serverListDialog, TQT_SIGNAL(connectTo(Konversation::ConnectionFlag, ConnectionSettings&)),
                konvApp->getConnectionManager(), TQT_SLOT(connectTo(Konversation::ConnectionFlag, ConnectionSettings&)));
        connect(konvApp->getConnectionManager(), TQT_SIGNAL(closeServerList()), m_serverListDialog, TQT_SLOT(slotClose()));
    }

    m_serverListDialog->show();
}

void KonversationMainWindow::openQuickConnectDialog()
{
    emit showQuickConnectDialog();
}

// open the preferences dialog and show the watched nicknames page
void KonversationMainWindow::openNotify()
{
    openPrefsDialog();
    if (m_settingsDialog) m_settingsDialog->openWatchedNicknamesPage();
}

void KonversationMainWindow::openIdentitiesDialog()
{
    Konversation::IdentityDialog dlg(this);

    if (dlg.exec() == KDialog::Accepted)
    {
        if (m_serverListDialog)
            m_serverListDialog->updateServerList();
        m_viewContainer->updateViewEncoding(m_viewContainer->getFrontView());
    }
}

IdentityPtr KonversationMainWindow::editIdentity(IdentityPtr identity)
{
    IdentityPtr newIdentity;

    Konversation::IdentityDialog dlg(this);
    newIdentity = dlg.setCurrentIdentity(identity);

    if ((dlg.exec() == KDialog::Accepted) && m_serverListDialog)
    {
        m_serverListDialog->updateServerList();
        return newIdentity;
    }
    else
        return 0;
}

void KonversationMainWindow::openNotifications()
{
    #ifdef USE_KNOTIFY
    (void) KNotifyDialog::configure(this);
    #endif
}

void KonversationMainWindow::notifyAction(const TQString& serverName, const TQString& nick)
{
    KonversationApplication* konvApp = static_cast<KonversationApplication*>(kapp);
    Server* server = konvApp->getConnectionManager()->getServerByName(serverName);
    if (server) server->notifyAction(nick);
}

// TODO: Let an own class handle notify things
void KonversationMainWindow::setOnlineList(Server* notifyServer,const TQStringList& /*list*/, bool /*changed*/)
{
    emit nicksNowOnline(notifyServer);
    // FIXME  if (changed && nicksOnlinePanel) newText(nicksOnlinePanel, TQString::null, true);
}

TQString KonversationMainWindow::currentURL(bool passNetwork)
{
    return m_viewContainer->currentViewURL(passNetwork);
}

TQString KonversationMainWindow::currentTitle()
{
    return m_viewContainer->currentViewTitle();
}

#include "konversationmainwindow.moc"
