
/***************************************************************************
 *                                                                         *
 *   KCPULoad and KNetLoad are copyright (c) 1999-2000, Markus Gustavsson  *
 *                                       (c) 2002, Ben Burton              *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef __STATDOCK_H
#define __STATDOCK_H

#include <qcolor.h>
#include <ksystemtray.h>

class StatPopup;

/**
 * A system tray window that displays a recent history of readings.
 * A single application might have many of these windows.
 *
 * Two simultaneous sets of readings are supported; these will be
 * referred to as upper and lower readings.  When diagram splitting is
 * switched on, these readings will be displayed together on the diagram
 * with the upper readings shown above the lower readings.  When diagram
 * splitting is switched off, only the upper readings will be displayed.
 *
 * The parent window of a StatDock must be a StatPopup, which provides
 * all of the actions in this window's context menu.
 */
class StatDock : public KSystemTray {
    Q_OBJECT

public:
    /**
     * Fill style constants.
     */
    static const int fillLines;
    static const int fillBars;
    static const int fillShaded;

    /**
     * Colour constants.
     */
    static const QColor colorGrid;
    static const QColor colorGridInactive;
    static const QColor colorLabel;
    static const QColor colorLabelInactive;
    static const QColor colorLower;
    static const QColor colorLowerInactive;
    static const QColor colorBlack;

public:
    /**
     * Constructor and destructor.
     *
     * Note that the constructor will call parent->initDock().
     *
     * Parameter whichDock must be 0 or 1 to specify whether this dock
     * will become dock[0] or dock[1] in the given StatPopup parent.
     * Parameter useLabel should contain the label that will be drawn on
     * the diagram if labelling is enabled.
     */
    StatDock(int whichDock, const QString& useLabel, StatPopup *parent,
        const char *name = 0);
    ~StatDock();

    /**
     * Setting display options.
     */
    void setGrid(bool);
    void setActive(bool);
    void setSoft(bool);
    void setSplit(bool);
    void setLabelled(bool);
    void setLabel(const QString&);
    void setFill(int);
    void setColor(const QColor&);

public slots:
    /**
     * Clear the history of recent readings.
     * All readings will be reset to zero and the diagram will be
     * updated.
     */
    void clearHistory();

    /**
     * Add the given pair of readings as the most recent in our list.
     * The diagram will be updated accordingly.
     *
     * Each argument should be a percentage between 0 and 100.
     * The sum of both arguments must not exceed 100.
     *
     * If diagram splitting is switched off, the given lower reading
     * will be ignored completely and 0 will be used instead.
     *
     * @param upper the upper reading in this pair.
     * @param lower the lower reading in this pair.
     */
    void addPercentReading(int upper, int lower);

private:
    /**
     * Repaint this system tray window with a fresh diagram.
     */
    void paintEvent(QPaintEvent*);

private:
    /**
     * Display options.
     */
    bool grid;
        /**< Should the grid be displayed behind the diagram? */
    bool active;
        /**< Is this meter currently active? */
    bool soft;
        /**< Are we artificially modifying the readings to produce a
             soft curve? */
    bool split;
        /**< Are we displaying both upper and lower readings? */
    bool labelled;
        /**< Should this diagram be labelled? */
    QString label;
        /**< The specific label to draw on this diagram. */
    int fill;
        /**< Specifies which of the predefined fill styles to use. */
    QColor colorUpper;
        /**< Colour for displaying the upper readings. */
    QColor colorUpperInactive;
        /**< Colour for displaying the upper readings whilst the diagram
             is inactive. */

    /**
     * Stored readings.
     */
    int* bufUpper;
        /**< Stores our list of recent upper readings.
             This list begins at index (pos + 1) as the earliest reading
             and cycles around to index (pos) as the latest reading. */
    int* bufLower;
        /**< Stores our list of recent lower readings.
             This list begins at index (pos + 1) as the earliest reading
             and cycles around to index (pos) as the latest reading. */
    int pos;
        /**< The index in our arrays of the most recent reading. */


    /**
     * Temporaries.
     */
    int i, j, tmpPos, oldUpper, oldLower;
        /**< Temporary variables used during computations. */

protected:
    void resizeEvent ( QResizeEvent * );
};

#endif
