/***************************************************************************
 *   Copyright (C) 2005 by Nicolas Ternisien                               *
 *   nicolas.ternisien@gmail.com                                           *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.             *
 ***************************************************************************/

#include <klocale.h>


#include "ksystemlogConfig.h"
#include "systemReader.h"
#include "xorgReader.h"
#include "cronReader.h"
#include "acpidReader.h"

#include "cupsReader.h"
#include "cupsAccessReader.h"

#include "apacheReader.h"
#include "apacheAccessReader.h"

#include "sambaReader.h"

#include "readerFactory.h"

KURL ReaderFactory::lastOpenedURL;

Reader* ReaderFactory::createReader(LogMode* logMode) {
	
	/**
	 * Using System Mode in the current view
	 */
	if (logMode==Globals::systemMode) {
		Reader* reader=new SystemReader(NULL, "system_reader");
		return(reader);
	}
	
	/**
	 * Using Boot Mode in the current view
	 */
	else if (logMode==Globals::bootMode) {
		Reader* reader=new SystemReader(NULL, "boot_reader");
		return(reader);
	}
	
	/**
	 * Using Authentication Mode in the current view
	 */
	else if (logMode==Globals::authenticationMode) {
		Reader* reader=new SystemReader(NULL, "authentication_reader");
		return(reader);
	}
	
	/**
	 * Using Kernel Mode in the current view
	 */
	else if (logMode==Globals::kernelMode) {
		Reader* reader=new SystemReader(NULL, "kernel_reader");
		return(reader);
	}
	
	/**
	 * Using Daemon Mode in the current view
	 */
	else if (logMode==Globals::daemonMode) {
		Reader* reader=new SystemReader(NULL, "daemon_reader");
		return(reader);
	}
	
	/**
	 * Using Cron Mode in the current view
	 */
	else if (logMode==Globals::cronMode) {
		Reader* reader=new CronReader(NULL, "cron_reader");
		return(reader);
	}
	
	/**
	 * Using Cron Mode in the current view
	 */
	else if (logMode==Globals::xorgMode) {
		Reader* reader=new XorgReader(NULL, "xorg_reader");
		return(reader);
	}
	

	/**
	 * Using ACPID Mode in the current view
	 */
	else if (logMode==Globals::acpidMode) {
		Reader* reader=new AcpidReader(NULL, "acpid_reader");
		return(reader);
	}
	
	/**
	 * Using Cups Mode in the current view
	 */
	else if (logMode==Globals::cupsMode) {
		Reader* reader=new CupsReader(NULL, "cups_reader");
		return(reader);
	}

	/**
	 * Using Cups Access Mode in the current view
	 */
	else if (logMode==Globals::cupsAccessMode) {
		Reader* reader=new CupsAccessReader(NULL, "cups_access_reader");
		return(reader);
	}
	
	/**
	 * Using Apache Mode in the current view
	 */
	else if (logMode==Globals::apacheMode) {
		Reader* reader=new ApacheReader(NULL, "apache_reader");
		return(reader);
	}

	/**
	 * Using Apache Access Mode in the current view
	 */
	else if (logMode==Globals::apacheAccessMode) {
		Reader* reader=new ApacheAccessReader(NULL, "apache_access_reader");
		return(reader);
	}
	
	/**
	 * Using Postfix Mode in the current view
	 */
	else if (logMode==Globals::postfixMode) {
		Reader* reader=new SystemReader(NULL, "postfix_reader");
		return(reader);
	}

	/**
	 * Using Samba Mode in the current view
	 */
	else if (logMode==Globals::sambaMode) {
		Reader* reader=new SambaReader(NULL, "samba_reader");
		return(reader);
	}
	
	/**
	 * Using Opening Mode in the current view
	 */
	else if (logMode==Globals::openingMode) {
		//TODO For the moment, SystemReader is always used to parse file.
		// Implement a test case for each Reader to know who is the better to read the file
		Reader* reader=new SystemReader(NULL, "opening_reader");
		return(reader);
	}
	else {
		kdDebug() << "Error : LogMode not found : returns NULL Reader" << endl;
		return(NULL);
	}
}

LogFiles* ReaderFactory::createLogFiles(LogMode* logMode) {

	/**
	 * Using System Mode in the current view
	 */
	if (logMode==Globals::systemMode) {
		//Give the log file list to the reader
		LogFiles* list=ReaderFactory::getSystemLogFiles();
		return(list);
	}
	
	/**
	 * Using Boot Mode in the current view
	 */
	else if (logMode==Globals::bootMode) {
		//Give the log file list to the reader
		LogFile* file=ReaderFactory::getBootLogFile();
		LogFiles* list=new LogFiles();
		list->append(file);
		return(list);
	}
	
	/**
	 * Using Authentication Mode in the current view
	 */
	else if (logMode==Globals::authenticationMode) {
		//Give the log file list to the reader
		LogFile* file=ReaderFactory::getAuthenticationLogFile();
		
		LogFiles* list=new LogFiles();
		list->append(file);
		return(list);
	}
	
	/**
	 * Using Kernel Mode in the current view
	 */
	else if (logMode==Globals::kernelMode) {
		//Give the log file list to the reader
		LogFiles* list=ReaderFactory::getKernelLogFiles();
		return(list);
	}
	
	/**
	 * Using Daemon Mode in the current view
	 */
	else if (logMode==Globals::daemonMode) {
		//Give the log file list to the reader
		LogFiles* list=ReaderFactory::getDaemonLogFiles();
		return(list);
	}
	
	/**
	 * Using Cron Mode in the current view
	 */
	else if (logMode==Globals::cronMode) {
		//Give the log file list to the reader
		LogFiles* list=ReaderFactory::getCronLogFiles();
		return(list);
	}
	
	/**
	 * Using Cron Mode in the current view
	 */
	else if (logMode==Globals::xorgMode) {
		//Give the log file list to the reader
		LogFiles* list=ReaderFactory::getXorgLogFiles();
		return(list);
	}
	

	/**
	 * Using ACPID Mode in the current view
	 */
	else if (logMode==Globals::acpidMode) {
		//Give the log file list to the reader
		LogFiles* list=ReaderFactory::getAcpidLogFiles();
		return(list);

	}
	
	/**
	 * Using Cups Mode in the current view
	 */
	else if (logMode==Globals::cupsMode) {
		//Give the log file list to the reader
		LogFiles* list=ReaderFactory::getCupsLogFiles();
		return(list);
	}

	/**
	 * Using Cups Access Mode in the current view
	 */
	else if (logMode==Globals::cupsAccessMode) {
		//Give the log file list to the reader
		LogFiles* list=ReaderFactory::getCupsAccessLogFiles();
		return(list);

	}
	
	/**
	 * Using Apache Mode in the current view
	 */
	else if (logMode==Globals::apacheMode) {
		//Give the log file list to the reader
		LogFiles* list=ReaderFactory::getApacheLogFiles();
		return(list);
	}

	/**
	 * Using Apache Access Mode in the current view
	 */
	else if (logMode==Globals::apacheAccessMode) {
		//Give the log file list to the reader
		LogFiles* list=ReaderFactory::getApacheAccessLogFiles();
		return(list);

	}
	
	/**
	 * Using Postfix Mode in the current view
	 */
	else if (logMode==Globals::postfixMode) {
		//Give the log file list to the reader
		LogFiles* list=ReaderFactory::getPostfixLogFiles();
		return(list);
	}
	
	/**
	 * Using Samba Mode in the current view
	 */
	else if (logMode==Globals::sambaMode) {
		//Give the log file list to the reader
		LogFiles* list=ReaderFactory::getSambaLogFiles();
		return(list);
	}
	
	/**
	 * Using Opening Mode in the current view
	 */
	else if (logMode==Globals::openingMode) {
		//Give the file to open to the reader
		
		//A little hack... ;-) (what an oriented-object solution !)
		LogFile* logFile=new LogFile(lastOpenedURL, Globals::informationLogLevel);
		
		LogFiles* list=new LogFiles();
		list->append(logFile);
		return(list);
	}
	else {
		kdDebug() << "Error : LogFiles not found : returns NULL Reader" << endl;
		return(NULL);
	}
}





LogFile* ReaderFactory::getGenericLogFile(QString& file) {

	LogLevel* level=Globals::informationLogLevel;
	
	KURL url(file);
	if (!url.isValid()) {
		kdDebug() << i18n("URL '%1' is not valid, skipping this URL.").arg(url.path()) << endl;
		return(NULL);
	}
	
	LogFile* logFile=new LogFile(url, level);
	return(logFile);
}

LogFiles* ReaderFactory::getGenericLogFiles(QStringList& stringList, QValueList<int>& valueList) {

	LogFiles* logFiles=new LogFiles();
	
	if (stringList.size() != valueList.size()) {
		kdDebug() << i18n("The two arrays size are different, skipping the reading of log files.") << endl;
		return(logFiles);
	}

	LogFile* logFile;
	
	LogLevel* level;
	
	QStringList::Iterator itString=stringList.begin();
	QValueList<int>::Iterator itInt=valueList.begin();
	
	while(itString!=stringList.end()) {
		if (*itInt>=0 && *itInt<(int) Globals::logLevels.count())
			level=Globals::logLevels.at(*itInt);
		else
			level=Globals::informationLogLevel;
		
		KURL url(*itString);
		if (!url.isValid()) {
			kdDebug() << i18n("URL '%1' is not valid, skipping this URL.").arg(url.path()) << endl;
			continue;
			itString++;
			itInt++;
		}
		
		logFile=new LogFile(url, level);
		logFiles->push_back(logFile);
			
	
		itString++;
		itInt++;
	}
	
	return(logFiles);
}

LogFiles* ReaderFactory::getNoModeLogFiles(QStringList& stringList) {

	LogFiles* logFiles=new LogFiles();
	
	LogFile* logFile;
	
	//Default level used for No Mode logs
	LogLevel* level=Globals::noneLogLevel;
	
	QStringList::Iterator it;
	for(it=stringList.begin(); it!=stringList.end(); it++) {
		
		KURL url(*it);
		if (!url.isValid()) {
			kdDebug() << i18n("URL '%1' is not valid, skipping this URL.").arg(url.path()) << endl;
			continue;
		}
		
		logFile=new LogFile(url, level);
		logFiles->push_back(logFile);

	}
	
	return(logFiles);
}

LogFile* ReaderFactory::getBootLogFile() {
	QString file=KSystemLogConfig::bootPath();
	return(getGenericLogFile(file));
}

LogFile* ReaderFactory::getAuthenticationLogFile() {
	QString file=KSystemLogConfig::authenticationPath();
	return(getGenericLogFile(file));
}

LogFiles* ReaderFactory::getSystemLogFiles() {
	QStringList files=KSystemLogConfig::systemPaths();
	QValueList<int> levels=KSystemLogConfig::systemLevels();
	return(ReaderFactory::getGenericLogFiles(files, levels));
}

LogFiles* ReaderFactory::getAcpidLogFiles() {
	QStringList files=KSystemLogConfig::acpidPaths();
	
	return(getNoModeLogFiles(files));
}

LogFiles* ReaderFactory::getKernelLogFiles() {
	QStringList files=KSystemLogConfig::kernelPaths();
	QValueList<int> levels=KSystemLogConfig::kernelLevels();
	return(ReaderFactory::getGenericLogFiles(files, levels));
}
		
LogFiles* ReaderFactory::getDaemonLogFiles() {
	QStringList files=KSystemLogConfig::daemonPaths();
	QValueList<int> levels=KSystemLogConfig::daemonLevels();
	return(ReaderFactory::getGenericLogFiles(files, levels));
}

LogFiles* ReaderFactory::getCronLogFiles() {
	QStringList files=KSystemLogConfig::cronPaths();
	QValueList<int> levels=KSystemLogConfig::cronLevels();
	return(ReaderFactory::getGenericLogFiles(files, levels));
}


LogFiles* ReaderFactory::getXorgLogFiles() {
	QStringList stringList=KSystemLogConfig::xorgPaths();
	return(getNoModeLogFiles(stringList));
}

LogFiles* ReaderFactory::getCupsLogFiles() {
	QStringList stringList=KSystemLogConfig::cupsPaths();
	return(getNoModeLogFiles(stringList));
}

LogFiles* ReaderFactory::getCupsAccessLogFiles() {
	QStringList stringList=KSystemLogConfig::cupsAccessPaths();
	return(getNoModeLogFiles(stringList));
}

LogFiles* ReaderFactory::getApacheLogFiles() {
	QStringList stringList=KSystemLogConfig::apachePaths();
	return(getNoModeLogFiles(stringList));
}

LogFiles* ReaderFactory::getApacheAccessLogFiles() {
	QStringList stringList=KSystemLogConfig::apacheAccessPaths();
	return(getNoModeLogFiles(stringList));
}

LogFiles* ReaderFactory::getPostfixLogFiles() {
	QStringList files=KSystemLogConfig::postfixPaths();
	QValueList<int> levels=KSystemLogConfig::postfixLevels();
	return(ReaderFactory::getGenericLogFiles(files, levels));
}

LogFiles* ReaderFactory::getSambaLogFiles() {
	QStringList stringList=KSystemLogConfig::sambaPaths();
	return(getNoModeLogFiles(stringList));
}

