/***************************************************************************
 *   Copyright (C) 2005 by Nicolas Ternisien                               *
 *   nicolas.ternisien@gmail.com                                           *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.             *
 ***************************************************************************/
 
#include "fileList.h"

#include <qlayout.h>
#include <qvgroupbox.h>
#include <qbuttongroup.h>
#include <qlabel.h>
#include <qpushbutton.h>
#include <qvbox.h>
#include <qhbox.h>
#include <qwhatsthis.h>
#include <qtooltip.h>

// KDE includes
#include <klocale.h>
#include <kactioncollection.h>
#include <kbuttonbox.h>
#include <klistbox.h>
#include <kfiledialog.h>
#include <kurl.h>
#include <kmessagebox.h>
#include <kiconloader.h>

FileList::FileList(QWidget *parent, QString description) : 
	QWidget(parent) 
	{

	QHBoxLayout *layout = new QHBoxLayout(this);
	layout->setAutoAdd(true);
	
	
	QVGroupBox* dirBox = new QVGroupBox(i18n( "Files" ), this);
	
	QLabel *defaultPathLabel = new QLabel(description, dirBox);
	defaultPathLabel->setTextFormat(RichText);

	QHBox* center= new QHBox(dirBox);
	center->setSpacing(10);
		
	fileList=new KListBox(center);
	buttons=new QVBox(center);
	
	fileList->setSelectionMode(QListBox::Extended);
	
	QToolTip::add(fileList, i18n("List of files used by this log type"));
	QWhatsThis::add(fileList, i18n("<qt>Here is a list of every files that will be read by KSystemLog to display the current log lines.</qt>"));
	

	add=new QPushButton(SmallIcon("fileopen"), i18n("&Add File..."), buttons);
	connect(add, SIGNAL(clicked()), this, SLOT(addItem()));
	QToolTip::add(add, i18n("Choose a new file"));
	QWhatsThis::add(add, i18n("Opens a dialog box to choose a new file to be added to the list."));

	
	remove=new QPushButton(SmallIconSet("edit_remove"), i18n("&Remove"), buttons);
	connect(remove, SIGNAL(clicked()), this, SLOT(removeSelectedItem()));
	QToolTip::add(remove, i18n("Delete the current file(s)"));
	QWhatsThis::add(remove, i18n("Deletes the selected files of the list."));
	
	up=new QPushButton(SmallIconSet("up"), i18n("Move &Up"), buttons);
	connect(up, SIGNAL(clicked()), this, SLOT(moveUpItem()));
	QToolTip::add(up, i18n("Move up the current file(s)"));
	QWhatsThis::add(up, i18n("<qt>Moves up the selected files in the list. This option allows the files to be read <b>in first</b> by KSystemLog.</qt>"));
	
	down=new QPushButton(SmallIconSet("down"), i18n("Move &Down"), buttons);
	connect(down, SIGNAL(clicked()), this, SLOT(moveDownItem()));
	QToolTip::add(down, i18n("Move down the current file(s)"));
	QWhatsThis::add(down, i18n("<qt>Moves down the selected files in the list. This option allows the files to be read <b>at last</b> by KSystemLog.</qt>"));

	removeAll=new QPushButton(SmallIconSet("cancel"), i18n("Re&move All"), buttons);
	connect(removeAll, SIGNAL(clicked()), this, SLOT(removeAllItem()));
	QToolTip::add(removeAll, i18n("Remove all files"));
	QWhatsThis::add(removeAll, i18n("<qt>Remove all files of the list, even if they are not selected.</qt>"));

	fileListMenu=new KPopupMenu(this);
	fileListMenu->insertTitle(i18n("File List"));
	fileListMenu->insertItem(SmallIcon("fileopen"), i18n("&Add File..."), this, SLOT(addItem()), 0, ADD_FILE_MENU_ID);
	fileListMenu->insertItem(SmallIcon("edit_remove"), i18n("&Remove"), this, SLOT(removeSelectedItem()), 0, REMOVE_MENU_ID);
	fileListMenu->insertSeparator();
	fileListMenu->insertItem(SmallIcon("up"), i18n("Move &Up"), this, SLOT(moveUpItem()), 0, MOVE_UP_MENU_ID);
	fileListMenu->insertItem(SmallIcon("down"), i18n("Move &Down"), this, SLOT(moveDownItem()), 0, MOVE_DOWN_MENU_ID);
	fileListMenu->insertSeparator();
	fileListMenu->insertItem(SmallIcon("cancel"), i18n("Re&move All"), this, SLOT(removeAllItem()), 0, REMOVE_ALL_MENU_ID);
	
	connect(fileList, SIGNAL(selectionChanged()), this, SLOT(updateButtons()));
	
	connect(this, SIGNAL(fileListChanged(int)), this, SLOT(updateButtons()));
	
	connect(fileList, SIGNAL(rightButtonClicked(QListBoxItem*, const QPoint &)), this, SLOT(displayPopupMenu(QListBoxItem*, const QPoint&)));
	
	updateButtons();

}

FileList::~FileList() {

}

QString FileList::getText(int i) {
	return(fileList->text(i));
}



void FileList::insertItem(QString& item) {
	fileList->insertItem(item);
}

int FileList::count() {
	return(fileList->count());
}

void FileList::displayPopupMenu(QListBoxItem*, const QPoint& point) {
	fileListMenu->popup(point);
}

void FileList::addItem() {
	//Open a standard Filedialog
	KURL::List urlList;
	urlList=KFileDialog::getOpenURLs(DEFAULT_FOLDER, "*|" + i18n("All Files (*)") + "\n*.log|" + i18n("Log Files (*.log)"), this, i18n("Choose Log File"));
	
	KURL url;
	KURL::List::iterator it;
	
	for(it=urlList.begin(); it!=urlList.end(); ++it) {
		url=*it;
		if (isValidFile(url)) {
			fileList->insertItem(url.path());
		}
	}
	
	emit fileListChanged(fileList->count());
}

bool FileList::isValidFile(KURL url) {
	QString message;
	
	//If it is not valid
	if (!url.isValid()) {
		message=i18n("'%1' is not valid.").arg(url.path());
		KMessageBox::error(this, message, i18n("File selection failed"), KMessageBox::Notify);
		return(false);
	}

	//If it is not a local file
	if (!url.isLocalFile()) {
		message=i18n("'%1' is not a local file.").arg(url.path());
		KMessageBox::error(this, message, i18n("File selection failed"), KMessageBox::Notify);
		return(false);
	}
	
	//If it's a directory, it's not valid
	if (QDir(url.path()).exists()) {
		message=i18n("'%1' is a folder.").arg(url.path());
		KMessageBox::error(this, message, i18n("File selection failed"), KMessageBox::Notify);
		return(false);
	}
		
	return(true);
}

void FileList::removeItem(int id) {
	fileList->removeItem(id);
}

void FileList::removeSelectedItem() {
	
	for (int i=0; i<(int) fileList->count(); i++) {
		if (fileList->isSelected(i)==true) {
			this->removeItem(i);
			i--;
		}
	}
	
	fileList->setSelected(fileList->count()-1, true);
	fileList->setCurrentItem(fileList->count()-1);
	
	emit fileListChanged(fileList->count());
}

void FileList::moveUpItem() {
	int count=fileList->count();
	
	QListBoxItem* item;
	int index;
	for (int i=1; i<count; i++) {
		if (fileList->isSelected(i)==true && fileList->isSelected(i-1)==false) {
			item=fileList->item(i);
			fileList->takeItem(item);
			fileList->insertItem(item, i-1);
			
			index=fileList->index(item);
			fileList->setSelected(index, true);
			fileList->setCurrentItem(index);
		}
	}
	
	emit fileListChanged(count);
}

void FileList::moveDownItem() {
	int count=fileList->count();
	
	QListBoxItem* item;
	int index;
	for (int i=count-2; i>=0; i--) {
		if (fileList->isSelected(i)==true && fileList->isSelected(i+1)==false) {
			item=fileList->item(i);
			fileList->takeItem(item);
			fileList->insertItem(item, i+1);
			
			index=fileList->index(item);
			fileList->setSelected(index, true);
			fileList->setCurrentItem(index);
		}
	}
	
	emit fileListChanged(count);

}

void FileList::removeAllItem() {
	int count=fileList->count();
	
	for (int i=0; i<count; i++) {
		this->removeItem(0);
	}
	
	emit fileListChanged(0);
}


bool FileList::updateButtons() {
	int count=fileList->count();
	
	if (count==0)
		removeAll->setEnabled(false);
	else
		removeAll->setEnabled(true);
	
	bool selection=false;
	
	for (int i=0; i<count; i++) {
		if (fileList->isSelected(i)==true) {
			selection=true;
		}
	}

	//If nothing is selected, disabled special buttons
	remove->setEnabled(selection);
	up->setEnabled(selection);
	down->setEnabled(selection);
	
	return(selection);
}


#include "fileList.moc"
