/***************************************************************************
 *   Copyright (C) 2004-2006 by Mark Kretschmann <markey@web.de>           *
 *                      2005 by Seb Ruiz <me@sebruiz.net>                  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Steet, Fifth Floor, Boston, MA  02110-1301, USA.          *
 ***************************************************************************/

#ifndef AMAROK_SCRIPTMANAGER_H
#define AMAROK_SCRIPTMANAGER_H

#include "engineobserver.h"   //baseclass
#include "playlistwindow.h"

#include <tqmap.h>

#include <kdialogbase.h>      //baseclass
#include <kurl.h>

class MetaBundle;
class ScriptManagerBase;
class TQListViewItem;
class KArchiveDirectory;
class KProcess;
class KProcIO;


/**
 * @class ScriptManager
 * @short Script management widget and backend
 * @author Mark Kretschmann <markey@web.de>
 *
 * Script notifications, sent to stdin:
 *   configure
 *   engineStateChange: {empty|idle|paused|playing}
 *   trackChange
 *   volumeChange: newVolume (range: 0-100)
 *   fetchLyrics: artist title
 *   fetchLyricsByUrl: url
 *
 * @see http://amarok.kde.org/amarokwiki/index.php/Script-Writing_HowTo
 */

class ScriptManager : public KDialogBase, public EngineObserver
{
    Q_OBJECT

    friend class AmarokScriptNewStuff;

    public:
        ScriptManager( TQWidget *parent = 0, const char *name = 0 );
        virtual ~ScriptManager();

        static ScriptManager* instance() { return s_instance ? s_instance : new ScriptManager( PlaylistWindow::self() ); }

        /**
         * Runs the script with the given name. Used by the DCOP handler.
         * @param name The name of the script.
         * @return True if successful.
         */
        bool runScript( const TQString& name, bool silent = false );

        /**
         * Stops the script with the given name. Used by the DCOP handler.
         * @param name The name of the script.
         * @return True if successful.
         */
        bool stopScript( const TQString& name );

        /** Returns a list of all currently running scripts. Used by the DCOP handler. */
        TQStringList listRunningScripts();

       /** Custom Menu Click */
       void customMenuClicked( const TQString& message );

       /** Returns the path of the spec file of the given script */
       TQString specForScript( const TQString& name );

       /** Return name of the lyrics script currently running, or TQString::null if none */
       TQString lyricsScriptRunning() const;

       /** Returns a list of all lyrics scripts */
       TQStringList lyricsScripts() const;

       /** Sends a fetchLyrics notification to all scripts */
       void notifyFetchLyrics( const TQString& artist, const TQString& title );

       /** Sends a fetchLyrics notification to retrieve lyrics from a specific page */
       void notifyFetchLyricsByUrl( const TQString& url );

       /** Sends a playlistChange notification to all scripts */
       void notifyPlaylistChange( const TQString& change );

       /** Return name of the transcode script currently running, or TQString::null if none */
       TQString transcodeScriptRunning() const;

       /** Sends a transcode notification to all scripts */
       void notifyTranscode( const TQString& srcUrl, const TQString& filetype );

       /** Return name of the scoring script currently running, or TQString::null if none */
       TQString scoreScriptRunning() const;

       /** Returns a list of all scoring scripts */
       TQStringList scoreScripts() const;

        /** Asks the current score script to give a new score based on the parameters. */
       void requestNewScore( const TQString &url, double prevscore, int playcount, int length, float percentage, const TQString &reason );

    signals:
        /** Emitted when the lyrics script changes, so that a lyrics retry can be made */
        void lyricsScriptChanged();

    private slots:
        /** Finds all installed scripts and adds them to the listview */
        void findScripts();

        /** Enables/disables the buttons */
        void slotCurrentChanged( TQListViewItem* );

        bool slotInstallScript( const TQString& path = TQString::null );
        void slotRetrieveScript();
        void slotUninstallScript();
        bool slotRunScript( bool silent = false );
        void slotStopScript();
        void slotConfigureScript();
        void slotAboutScript();
        void slotShowContextMenu( TQListViewItem*, const TQPoint& );

        void slotReceivedStdout( KProcess*, char*, int );
        void slotReceivedStderr( KProcess*, char*, int );
        void scriptFinished( KProcess* process );

    private:
        /** Returns all scripts of the given \p type */
        TQStringList scriptsOfType( const TQString &type ) const;

        /** Returns the first running script found of \p type */
        TQString scriptRunningOfType( const TQString &type ) const;

        TQString ensureScoreScriptRunning();

        /** Terminates a process with SIGTERM and deletes the KProcIO object */
        void terminateProcess( KProcIO** proc );

        /** Sends a string message to all running scripts */
        void notifyScripts( const TQString& message );

        /** Adds a script to the listview */
        void loadScript( const TQString& path );

        /** Copies the file permissions from the tarball and loads the script */
        void recurseInstall( const KArchiveDirectory* archiveDir, const TQString& destination );

        /** EngineObserver reimplementations **/
        void engineStateChanged( Engine::State state, Engine::State oldState = Engine::Empty );
        void engineNewMetaData( const MetaBundle& /*bundle*/, bool /*trackChanged*/ );
        void engineVolumeChanged( int newVolume );

        /////////////////////////////////////////////////////////////////////////////////////
        // DATA MEMBERS
        /////////////////////////////////////////////////////////////////////////////////////
        static ScriptManager* s_instance;
        ScriptManagerBase*    m_gui;

        TQListViewItem*        m_generalCategory;
        TQListViewItem*        m_lyricsCategory;
        TQListViewItem*        m_scoreCategory;
        TQListViewItem*        m_transcodeCategory;

        bool                  m_installSuccess;

        struct ScriptItem {
            KURL           url;
            TQString        type;
            KProcIO*       process;
            TQListViewItem* li;
            TQString        log;
            ScriptItem() : process( 0 ), li( 0 ) {}
        };

        typedef TQMap<TQString, ScriptItem> ScriptMap;

        ScriptMap m_scripts;
};


inline TQStringList ScriptManager::lyricsScripts() const { return scriptsOfType( "lyrics" ); }

inline TQString ScriptManager::lyricsScriptRunning() const { return scriptRunningOfType( "lyrics" ); }

inline TQString ScriptManager::transcodeScriptRunning() const { return scriptRunningOfType( "transcode" ); }

inline TQStringList ScriptManager::scoreScripts() const { return scriptsOfType( "score" ); }

inline TQString ScriptManager::scoreScriptRunning() const { return scriptRunningOfType( "score" ); }

#endif /* AMAROK_SCRIPTMANAGER_H */


