/* -*- c-basic-offset: 4 indent-tabs-mode: nil -*- vi:set ts=8 sts=4 sw=4: */

/*
    Rosegarden
    A MIDI and audio sequencer and musical notation editor.
 
    This program is Copyright 2000-2008
        Guillaume Laurent   <glaurent@telegraph-road.org>,
        Chris Cannam        <cannam@all-day-breakfast.com>,
        Richard Bown        <richard.bown@ferventsoftware.com>
 
    The moral rights of Guillaume Laurent, Chris Cannam, and Richard
    Bown to claim authorship of this work have been asserted.
 
    This file is based on code from KGhostView, Copyright 1997-2002
        Markkhu Hihnala     <mah@ee.oulu.fi>
        and the KGhostView authors.

    Other copyrights also apply to some parts of this work.  Please
    see the AUTHORS file and individual file headers for details.
 
    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License as
    published by the Free Software Foundation; either version 2 of the
    License, or (at your option) any later version.  See the file
    COPYING included with this distribution for more information.
*/


#include "ScrollBox.h"

#include <tqapplication.h>
#include <tqframe.h>
#include <tqimage.h>
#include <tqpainter.h>
#include <tqpixmap.h>
#include <tqpoint.h>
#include <tqrect.h>
#include <tqsize.h>
#include <tqwidget.h>


namespace Rosegarden
{

ScrollBox::ScrollBox(TQWidget* parent, SizeMode sizeMode, const char* name) :
        TQFrame(parent, name),
        m_sizeMode(sizeMode)
{
    setFrameStyle(Panel | Sunken);
}

void ScrollBox::mousePressEvent(TQMouseEvent* e)
{
    m_mouse = e->pos();
    if (e->button() == RightButton)
        emit button3Pressed();
    if (e->button() == MidButton)
        emit button2Pressed();
}

void ScrollBox::mouseMoveEvent(TQMouseEvent* e)
{
    if (e->state() != LeftButton)
        return ;

    int dx = (e->pos().x() - m_mouse.x()) * m_pagesize.width() / width();
    int dy = (e->pos().y() - m_mouse.y()) * m_pagesize.height() / height();

    emit valueChanged(TQPoint(m_viewpos.x() + dx, m_viewpos.y() + dy));
    emit valueChangedRelative(dx, dy);

    m_mouse = e->pos();
}

void ScrollBox::drawContents(TQPainter* paint)
{
    if (m_pagesize.isEmpty())
        return ;

    TQRect c(contentsRect());

    paint->setPen(Qt::red);

    int len = m_pagesize.width();
    int x = c.x() + c.width() * m_viewpos.x() / len;
    int w = c.width() * m_viewsize.width() / len ;
    if (w > c.width())
        w = c.width();

    len = m_pagesize.height();
    int y = c.y() + c.height() * m_viewpos.y() / len;
    int h = c.height() * m_viewsize.height() / len;
    if (h > c.height())
        h = c.height();

    paint->drawRect(x, y, w, h);
}

void ScrollBox::setPageSize(const TQSize& s)
{
    m_pagesize = s;

    setFixedWidth(100);
    setFixedHeight(100);

    int maxWidth = int(TQApplication::desktop()->width() * 0.75);
    int maxHeight = int(TQApplication::desktop()->height() * 0.75);

    if (m_sizeMode == FixWidth) {
        int height = s.height() * width() / s.width();
        if (height > maxHeight) {
            setFixedWidth(width() * maxHeight / height);
            height = maxHeight;
        }
        setFixedHeight(height);
    } else {
        int width = s.width() * height() / s.height();
        if (width > maxWidth) {
            setFixedHeight(height() * maxWidth / width);
            width = maxWidth;
        }
        setFixedWidth(width);
    }

    repaint();
}

void ScrollBox::setViewSize(const TQSize& s)
{
    m_viewsize = s;
    repaint();
}

void ScrollBox::setViewPos(const TQPoint& pos)
{
    m_viewpos = pos;
    repaint();
}

void ScrollBox::setViewX(int x)
{
    m_viewpos = TQPoint(x, m_viewpos.y());
    repaint();
}

void ScrollBox::setViewY(int y)
{
    m_viewpos = TQPoint(m_viewpos.x(), y);
    repaint();
}

void ScrollBox::setThumbnail(TQPixmap img)
{
    setPaletteBackgroundPixmap(img.convertToImage().smoothScale(size()));
}

}
#include "ScrollBox.moc"
