
/* -*- c-basic-offset: 4 indent-tabs-mode: nil -*- vi:set ts=8 sts=4 sw=4: */

/*
    Rosegarden
    A MIDI and audio sequencer and musical notation editor.

    This program is Copyright 2000-2008
        Guillaume Laurent   <glaurent@telegraph-road.org>,
        Chris Cannam        <cannam@all-day-breakfast.com>,
        Richard Bown        <richard.bown@ferventsoftware.com>

    The moral rights of Guillaume Laurent, Chris Cannam, and Richard
    Bown to claim authorship of this work have been asserted.

    Other copyrights also apply to some parts of this work.  Please
    see the AUTHORS file and individual file headers for details.

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License as
    published by the Free Software Foundation; either version 2 of the
    License, or (at your option) any later version.  See the file
    COPYING included with this distribution for more information.
*/

#ifndef _RG_PIXMAPFUNCTIONS_H_
#define _RG_PIXMAPFUNCTIONS_H_

#include <tqbitmap.h>
#include <tqpixmap.h>
#include <utility>


namespace Rosegarden
{



class PixmapFunctions
{
public:
    /**
     * Generate a heuristic mask for the given pixmap.  Unlike
     * TQPixmap::createHeuristicMask, this removes from the mask all
     * pixels that are apparently "background" even if they appear in
     * holes in the middle of the image.  This is more usually what we
     * want than the default behaviour of createHeuristicMask.
     *
     * The rgb value specifies the colour to treat as background.
     *
     * This function is slow.
     */
    static TQBitmap generateMask(const TQPixmap &map, const QRgb &rgb);

    /**
     * Generate a heuristic mask for the given pixmap.  Unlike
     * TQPixmap::createHeuristicMask, this removes from the mask all
     * pixels that are apparently "background" even if they appear in
     * holes in the middle of the image.  This is more usually what we
     * want than the default behaviour of createHeuristicMask.
     *
     * This function calculates its own estimated colour to match as
     * background.
     *
     * This function is slow.
     */
    static TQBitmap generateMask(const TQPixmap &map);

    /**
     * Colour a greyscale pixmap with the given hue.
     * minValue specifies the minimum value (in the HSV sense) that
     * will be used for any recoloured pixel.
     */
    static TQPixmap colourPixmap(const TQPixmap &map, int hue, int minValue);

    /**
     * Make a pixmap grey, or otherwise reduce its intensity.
     */
    static TQPixmap shadePixmap(const TQPixmap &map);

    /// Return a TQPixmap that is a mirror image of map (including mask)
    static TQPixmap flipVertical(const TQPixmap &map);

    /// Return a TQPixmap that is a mirror image of map (including mask)
    static TQPixmap flipHorizontal(const TQPixmap &map);

    /// Return left and right parts of the QPixmap
    static std::pair<TQPixmap, TQPixmap> splitPixmap(const TQPixmap &original, int x);

    /**
     * Using TQPainter::drawPixmap to draw one pixmap on another does
     * not appear to take the mask into account properly.  Background
     * pixels in the second pixmap erase foreground pixels in the
     * first one, regardless of whether they're masked or not.  This
     * function does what I expect.
     *
     * Note that the source pixmap _must_ have a mask.
     */
    static void drawPixmapMasked(TQPixmap &dest, TQBitmap &destMask,
                                 int x, int y,
                                 const TQPixmap &source);
};


}

#endif
