/* -*- c-basic-offset: 4 indent-tabs-mode: nil -*- vi:set ts=8 sts=4 sw=4: */

/*
    Rosegarden
    A MIDI and audio sequencer and musical notation editor.
 
    This program is Copyright 2000-2008
        Guillaume Laurent   <glaurent@telegraph-road.org>,
        Chris Cannam        <cannam@all-day-breakfast.com>,
        Richard Bown        <richard.bown@ferventsoftware.com>
 
    The moral rights of Guillaume Laurent, Chris Cannam, and Richard
    Bown to claim authorship of this work have been asserted.
 
    Other copyrights also apply to some parts of this work.  Please
    see the AUTHORS file and individual file headers for details.
 
    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License as
    published by the Free Software Foundation; either version 2 of the
    License, or (at your option) any later version.  See the file
    COPYING included with this distribution for more information.
*/


#include "PixmapFunctions.h"

#include <tqbitmap.h>
#include <tqcolor.h>
#include <tqimage.h>
#include <tqpainter.h>
#include <tqpixmap.h>

#include <iostream>

namespace Rosegarden
{

QBitmap
PixmapFunctions::generateMask(const TQPixmap &map, const QRgb &px)
{
    TQImage i(map.convertToImage());
    TQImage im(i.width(), i.height(), 1, 2, TQImage::LittleEndian);

    for (int y = 0; y < i.height(); ++y) {
        for (int x = 0; x < i.width(); ++x) {
            if (i.pixel(x, y) != px) {
                im.setPixel(x, y, 1);
            } else {
                im.setPixel(x, y, 0);
            }
        }
    }

    TQBitmap m;
    m.convertFromImage(im);
    return m;
}

QBitmap
PixmapFunctions::generateMask(const TQPixmap &map)
{
    TQImage i(map.convertToImage());
    TQImage im(i.width(), i.height(), 1, 2, TQImage::LittleEndian);

    QRgb px0(i.pixel(0, 0));
    QRgb px1(i.pixel(i.width() - 1, 0));
    QRgb px2(i.pixel(i.width() - 1, i.height() - 1));
    QRgb px3(i.pixel(0, i.height() - 1));

    QRgb px(px0);
    if (px0 != px2 && px1 == px3)
        px = px1;

    for (int y = 0; y < i.height(); ++y) {
        for (int x = 0; x < i.width(); ++x) {
            if (i.pixel(x, y) != px) {
                im.setPixel(x, y, 1);
            } else {
                im.setPixel(x, y, 0);
            }
        }
    }

    TQBitmap m;
    m.convertFromImage(im);
    return m;
}

QPixmap
PixmapFunctions::colourPixmap(const TQPixmap &map, int hue, int minValue)
{
    // assumes pixmap is currently in shades of grey; maps black ->
    // solid colour and greys -> shades of colour

    TQImage image = map.convertToImage();

    int s, v;

    bool warned = false;

    for (int y = 0; y < image.height(); ++y) {
        for (int x = 0; x < image.width(); ++x) {

            TQColor pixel(image.pixel(x, y));

            int oldHue;
            pixel.hsv(&oldHue, &s, &v);

            if (oldHue >= 0) {
                if (!warned) {
                    std::cerr << "PixmapFunctions::recolour: Not a greyscale pixmap "
                    << "(found rgb value " << pixel.red() << ","
                    << pixel.green() << "," << pixel.blue()
                    << "), hoping for the best" << std::endl;
                    warned = true;
                }
            }

            image.setPixel
            (x, y, TQColor(hue,
                          255 - v,
                          v > minValue ? v : minValue,
                          TQColor::Hsv).rgb());
        }
    }

    TQPixmap rmap;
    rmap.convertFromImage(image);
    if (map.tqmask())
        rmap.setMask(*map.tqmask());
    return rmap;
}

QPixmap
PixmapFunctions::shadePixmap(const TQPixmap &map)
{
    TQImage image = map.convertToImage();

    int h, s, v;

    for (int y = 0; y < image.height(); ++y) {
        for (int x = 0; x < image.width(); ++x) {

            TQColor pixel(image.pixel(x, y));

            pixel.hsv(&h, &s, &v);

            image.setPixel
            (x, y, TQColor(h,
                          s,
                          255 - ((255 - v) / 2),
                          TQColor::Hsv).rgb());
        }
    }

    TQPixmap rmap;
    rmap.convertFromImage(image);
    if (map.tqmask())
        rmap.setMask(*map.tqmask());
    return rmap;
}

QPixmap
PixmapFunctions::flipVertical(const TQPixmap &map)
{
    TQPixmap rmap;
    TQImage i(map.convertToImage());
    rmap.convertFromImage(i.mirror(false, true));

    if (map.tqmask()) {
        TQImage im(map.tqmask()->convertToImage());
        TQBitmap newMask;
        newMask.convertFromImage(im.mirror(false, true));
        rmap.setMask(newMask);
    }

    return rmap;
}

QPixmap
PixmapFunctions::flipHorizontal(const TQPixmap &map)
{
    TQPixmap rmap;
    TQImage i(map.convertToImage());
    rmap.convertFromImage(i.mirror(true, false));

    if (map.tqmask()) {
        TQImage im(map.tqmask()->convertToImage());
        TQBitmap newMask;
        newMask.convertFromImage(im.mirror(true, false));
        rmap.setMask(newMask);
    }

    return rmap;
}

std::pair<TQPixmap, TQPixmap>
PixmapFunctions::splitPixmap(const TQPixmap &pixmap, int x)
{
    TQPixmap left(x, pixmap.height(), pixmap.depth());
    TQBitmap leftMask(left.width(), left.height());

    TQPixmap right(pixmap.width() - x, pixmap.height(), pixmap.depth());
    TQBitmap rightMask(right.width(), right.height());

    TQPainter paint;

    paint.begin(&left);
    paint.drawPixmap(0, 0, pixmap, 0, 0, left.width(), left.height());
    paint.end();

    paint.begin(&leftMask);
    paint.drawPixmap(0, 0, *pixmap.tqmask(), 0, 0, left.width(), left.height());
    paint.end();

    left.setMask(leftMask);

    paint.begin(&right);
    paint.drawPixmap(0, 0, pixmap, left.width(), 0, right.width(), right.height());
    paint.end();

    paint.begin(&rightMask);
    paint.drawPixmap(0, 0, *pixmap.tqmask(), left.width(), 0, right.width(), right.height());
    paint.end();

    right.setMask(rightMask);

    return std::pair<TQPixmap, TQPixmap>(left, right);
}

void
PixmapFunctions::drawPixmapMasked(TQPixmap &dest, TQBitmap &destMask,
                                  int x0, int y0,
                                  const TQPixmap &src)
{
    TQImage idp(dest.convertToImage());
    TQImage idm(destMask.convertToImage());
    TQImage isp(src.convertToImage());
    TQImage ism(src.tqmask()->convertToImage());

    for (int y = 0; y < isp.height(); ++y) {
        for (int x = 0; x < isp.width(); ++x) {

            if (x >= ism.width())
                continue;
            if (y >= ism.height())
                continue;

            if (ism.depth() == 1 && ism.pixel(x, y) == 0)
                continue;
            if (ism.pixel(x, y) == Qt::white.rgb())
                continue;

            int x1 = x + x0;
            int y1 = y + y0;
            if (x1 < 0 || x1 >= idp.width())
                continue;
            if (y1 < 0 || y1 >= idp.height())
                continue;

            idp.setPixel(x1, y1, isp.pixel(x, y));
            idm.setPixel(x1, y1, 1);
        }
    }

    dest.convertFromImage(idp);
    destMask.convertFromImage(idm);
}

}
