/* -*- c-basic-offset: 4 indent-tabs-mode: nil -*- vi:set ts=8 sts=4 sw=4: */

/*
    Rosegarden
    A MIDI and audio sequencer and musical notation editor.
 
    This program is Copyright 2000-2008
        Guillaume Laurent   <glaurent@telegraph-road.org>,
        Chris Cannam        <cannam@all-day-breakfast.com>,
        Richard Bown        <richard.bown@ferventsoftware.com>
 
    The moral rights of Guillaume Laurent, Chris Cannam, and Richard
    Bown to claim authorship of this work have been asserted.
 
    Other copyrights also apply to some parts of this work.  Please
    see the AUTHORS file and individual file headers for details.
 
    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License as
    published by the Free Software Foundation; either version 2 of the
    License, or (at your option) any later version.  See the file
    COPYING included with this distribution for more information.
*/


#include "AudioPropertiesPage.h"

#include "misc/Strings.h"
#include "ConfigurationPage.h"
#include "document/RosegardenGUIDoc.h"
#include "gui/application/RosegardenApplication.h"
#include "gui/studio/AudioPluginManager.h"
#include "sound/AudioFileManager.h"
#include "TabbedConfigurationPage.h"
#include <kconfig.h>
#include <kdiskfreesp.h>
#include <kfiledialog.h>
#include <kfile.h>
#include <tqcstring.h>
#include <tqdatastream.h>
#include <tqdialog.h>
#include <tqframe.h>
#include <tqlabel.h>
#include <tqpushbutton.h>
#include <tqstring.h>
#include <tqtabwidget.h>
#include <tqwidget.h>
#include <tqlayout.h>


namespace Rosegarden
{

AudioPropertiesPage::AudioPropertiesPage(RosegardenGUIDoc *doc,
        TQWidget *parent,
        const char *name)
        : TabbedConfigurationPage(doc, parent, name)
{
    AudioFileManager &afm = doc->getAudioFileManager();

    TQFrame *frame = new TQFrame(m_tabWidget);
    TQGridLayout *layout = new TQGridLayout(frame, 4, 3, 10, 5);
    layout->addWidget(new TQLabel(i18n("Audio file path:"), frame), 0, 0);
    m_path = new TQLabel(TQString(afm.getAudioPath().c_str()), frame);
    layout->addWidget(m_path, 0, 1);

    m_changePathButton =
        new TQPushButton(i18n("Choose..."), frame);

    layout->addWidget(m_changePathButton, 0, 2);

    m_diskSpace = new TQLabel(frame);
    layout->addWidget(new TQLabel(i18n("Disk space remaining:"), frame), 1, 0);
    layout->addWidget(m_diskSpace, 1, 1);

    m_minutesAtStereo = new TQLabel(frame);
    layout->addWidget(
        new TQLabel(i18n("Equivalent minutes of 16-bit stereo:"),
                   frame), 2, 0);

    layout->addWidget(m_minutesAtStereo, 2, 1, AlignCenter);

    layout->setRowStretch(3, 2);

    calculateStats();

    connect(m_changePathButton, TQT_SIGNAL(released()),
            TQT_SLOT(slotFileDialog()));

    addTab(frame, i18n("Modify audio path"));
}

void
AudioPropertiesPage::calculateStats()
{
    // This stolen from KDE libs kfile/kpropertiesdialog.cpp
    //
    TQString mountPoint = KIO::findPathMountPoint(m_path->text());
    KDiskFreeSp * job = new KDiskFreeSp;
    connect(job, TQT_SIGNAL(foundMountPoint(const TQString&, unsigned long, unsigned long,
                                        unsigned long)),
            this, TQT_SLOT(slotFoundMountPoint(const TQString&, unsigned long, unsigned long,
                                           unsigned long)));
    job->readDF(mountPoint);
}

void
AudioPropertiesPage::slotFoundMountPoint(const TQString&,
        unsigned long kBSize,
        unsigned long /*kBUsed*/,
        unsigned long kBAvail )
{
    m_diskSpace->setText(i18n("%1 out of %2 (%3% used)")
                         .arg(KIO::convertSizeFromKB(kBAvail))
                         .arg(KIO::convertSizeFromKB(kBSize))
                         .arg( 100 - (int)(100.0 * kBAvail / kBSize) ));


    AudioPluginManager *apm = m_doc->getPluginManager();

    int sampleRate = 48000;
    TQCString replyType;
    TQByteArray replyData;

    if (rgapp->sequencerCall("getSampleRate()", replyType, replyData)) {

        TQDataStream streamIn(replyData, IO_ReadOnly);
        unsigned int result;
        streamIn >> result;
        sampleRate = result;
    }

    // Work out total bytes and divide this by the sample rate times the
    // number of channels (2) times the number of bytes per sample (2)
    // times 60 seconds.
    //
    float stereoMins = ( float(kBAvail) * 1024.0 ) /
                       ( float(sampleRate) * 2.0 * 2.0 * 60.0 );
    TQString minsStr;
    minsStr.sprintf("%8.1f", stereoMins);

    m_minutesAtStereo->
    setText(TQString("%1 %2 %3Hz").arg(minsStr)
            .arg(i18n("minutes at"))
            .arg(sampleRate));
}

void
AudioPropertiesPage::slotFileDialog()
{
    AudioFileManager &afm = m_doc->getAudioFileManager();

    KFileDialog *fileDialog = new KFileDialog(TQString(afm.getAudioPath().c_str()),
                              TQString::null,
                              this, "file dialog", true);
    fileDialog->setMode(KFile::Directory);

    connect(fileDialog, TQT_SIGNAL(fileSelected(const TQString&)),
            TQT_SLOT(slotFileSelected(const TQString&)));

    connect(fileDialog, TQT_SIGNAL(destroyed()),
            TQT_SLOT(slotDirectoryDialogClosed()));

    if (fileDialog->exec() == TQDialog::Accepted) {
        m_path->setText(fileDialog->selectedFile());
        calculateStats();
    }
    delete fileDialog;
}

void
AudioPropertiesPage::apply()
{
    AudioFileManager &afm = m_doc->getAudioFileManager();
    TQString newDir = m_path->text();

    if (!newDir.isNull()) {
        afm.setAudioPath(qstrtostr(newDir));
        m_doc->slotDocumentModified();
    }
}

}
#include "AudioPropertiesPage.moc"
