/*********
*
* This file is part of BibleTime's source code, http://www.bibletime.info/.
*
* Copyright 1999-2006 by the BibleTime developers.
* The BibleTime source code is licensed under the GNU General Public License version 2.0.
*
**********/



//own includes
#include "ctoolclass.h"

#include "util/cresmgr.h"
#include "backend/cswordbackend.h"
#include "backend/cswordmoduleinfo.h"

//QT includes
#include <tqlabel.h>
#include <tqfile.h>
#include <tqfiledialog.h>
#include <tqtextstream.h>
#include <tqregexp.h>

//KDE includes
#include <klocale.h>
#include <kglobal.h>
#include <kiconloader.h>
#include <kstandarddirs.h>
#include <kmessagebox.h>
#include <kurl.h>
#include <kfiledialog.h>
#include <kapplication.h>


TQString CToolClass::locatehtml(const TQString &filename) {
	TQString path = locate("html", KGlobal::locale()->language() + '/' + filename);
	if (path.isNull())
		path = locate("html", "default/" + filename);
	if (path.isNull())
		path = locate("html", "en/" + filename);
	return path;
}

/** Converts HTML text to plain text */
TQString CToolClass::htmlToText(const TQString& html) {
	TQString newText = html;
	// convert some tags we need in code
	newText.replace( TQRegExp(" "),"#SPACE#" );
	newText.replace( TQRegExp("<br/?>\\s*"), "<br/>\n" );
	newText.replace( TQRegExp("#SPACE#")," " );

	TQRegExp re("<.+>");
	re.setMinimal(true);
	newText.replace( re,"" );
	return newText;
}

/** Converts text to HTML (\n to <BR>) */
TQString CToolClass::textToHTML(const TQString& text) {
	TQString newText = text;
	newText.replace( TQRegExp("<BR>\n"),"#NEWLINE#" );
	newText.replace( TQRegExp("\n"),"<BR>\n" );
	newText.replace( TQRegExp("#NEWLINE#"),"<BR>\n");
	return newText;
}

/** Creates the file filename and put text into the file.
 */
bool CToolClass::savePlainFile( const TQString& filename, const TQString& text, const bool& forceOverwrite, const TQTextStream::Encoding& fileEncoding) {
	TQFile saveFile(filename);
	bool ret;

	if (saveFile.exists()) {
		if (!forceOverwrite && KMessageBox::warningYesNo(0,
				TQString::fromLatin1("<qt><B>%1</B><BR>%2</qt>")
				.arg( i18n("The file already exists.") )
				.arg( i18n("Do you want to overwrite it?")
					)
														) == KMessageBox::No
		   ) {
			return false;
		}
		else { //either the user chose yes or forceOverwrite is set
			saveFile.remove();
		}
	};

	if ( saveFile.open(IO_ReadWrite) ) {
		TQTextStream textstream( &saveFile );
		textstream.setEncoding(fileEncoding);
		textstream << text;
		saveFile.close();
		ret = true;
	}
	else {
		KMessageBox::error(0, TQString::fromLatin1("<qt>%1<BR><B>%2</B></qt>")
						   .arg( i18n("The file couldn't be saved.") )
						   .arg( i18n("Please check permissions etc.")));
		saveFile.close();
		ret = false;
	}
	return ret;
}


/** Returns the icon used for the module given as aparameter. */
TQPixmap CToolClass::getIconForModule( CSwordModuleInfo* module_info ) {
	if (!module_info)
		return SmallIcon(CResMgr::modules::book::icon_locked, 16);

	if (module_info->category() == CSwordModuleInfo::Cult) {
		return SmallIcon("stop.png", 16);
	};


	TQPixmap img;

	switch (module_info->type()) {
		case CSwordModuleInfo::Bible:
		if (module_info->isLocked())
			img = SmallIcon(CResMgr::modules::bible::icon_locked, 16);
		else
			img = SmallIcon(CResMgr::modules::bible::icon_unlocked, 16);
		break;

		case CSwordModuleInfo::Lexicon:
		if (module_info->isLocked())
			img = SmallIcon(CResMgr::modules::lexicon::icon_locked, 16);
		else
			img = SmallIcon(CResMgr::modules::lexicon::icon_unlocked, 16);
		break;

		case CSwordModuleInfo::Commentary:
		if (module_info->isLocked())
			img = SmallIcon(CResMgr::modules::commentary::icon_locked, 16);
		else
			img = SmallIcon(CResMgr::modules::commentary::icon_unlocked, 16);
		break;

		case CSwordModuleInfo::GenericBook:
		if (module_info->isLocked())
			img = SmallIcon(CResMgr::modules::book::icon_locked, 16);
		else
			img = SmallIcon(CResMgr::modules::book::icon_unlocked, 16);
		break;

		case CSwordModuleInfo::Unknown: //fall though to default
		default:
		if (module_info->isLocked())
			img = SmallIcon(CResMgr::modules::book::icon_locked, 16);
		else
			img = SmallIcon(CResMgr::modules::book::icon_unlocked, 16);
		break;
	}


	return img;
}

TQLabel* CToolClass::explanationLabel(TQWidget* parent, const TQString& heading, const TQString& text ) {
	TQLabel* label = new TQLabel( TQString::fromLatin1("<B>%1</B><BR>%2").arg(heading).arg(text),parent );
	label->setAutoResize(true);
	label->setMargin(1);
	label->setFrameStyle(TQFrame::Box | TQFrame::Plain);
	return label;
}

/** No descriptions */
bool CToolClass::inHTMLTag(int pos, TQString & text) {
	int i1=text.findRev("<",pos);
	int i2=text.findRev(">",pos);
	int i3=text.find(">",pos);
	int i4=text.find("<",pos);


	// if ((i1>0) && (i2==-1))  //we're in th first html tag
	//  i2=i1; // not ncessary, just for explanation

	if ((i3>0) && (i4==-1))  //we're in the last html tag
		i4=i3+1;

	//  qWarning("%d > %d && %d < %d",i1,i2,i3,i4);

	if ( (i1>i2) && (i3<i4) )
		return true; //yes, we're in a tag

	return false;
}

TQString CToolClass::moduleToolTip(CSwordModuleInfo* module) {
	Q_ASSERT(module);
	if (!module) {
		return TQString::null;
	}

	TQString text;

	text = TQString("<b>%1</b> ").arg( module->name() )
		   + ((module->category() == CSwordModuleInfo::Cult) ? TQString::fromLatin1("<small><b>%1</b></small><br>").arg(i18n("Take care, this work contains cult / questionable material!")) : TQString::null);

	text += TQString("<small>(") + module->config(CSwordModuleInfo::Description) + TQString(")</small><hr>");

	text += i18n("Language") + TQString(": %1<br>").arg( module->language()->translatedName() );

	if (module->isEncrypted()) {
		text += i18n("Unlock key") + TQString(": %1<br>")
				.arg(!module->config(CSwordModuleInfo::CipherKey).isEmpty() ? module->config(CSwordModuleInfo::CipherKey) : TQString("<font COLOR=\"red\">%1</font>").arg(i18n("not set")));
	}

	if (module->hasVersion()) {
		text += i18n("Version") + TQString(": %1<br>").arg( module->config(CSwordModuleInfo::ModuleVersion) );
	}

	TQString options;
	unsigned int opts;
	for (opts = CSwordModuleInfo::filterTypesMIN; opts <= CSwordModuleInfo::filterTypesMAX; ++opts) {
		if (module->has( static_cast<CSwordModuleInfo::FilterTypes>(opts) )) {
			if (!options.isEmpty()) {
				options += TQString::fromLatin1(", ");
			}

			options += CSwordBackend::translatedOptionName(
						   static_cast<CSwordModuleInfo::FilterTypes>(opts)
					   );
		}
	}

	if (!options.isEmpty()) {
		text += i18n("Options") + TQString::fromLatin1(": <small>") + options + TQString("</small>");
	}

	if (text.right(4) == TQString::fromLatin1("<br>")) {
		text = text.left(text.length()-4);
	}

	return text;
}
