/*********
*
* This file is part of BibleTime's source code, http://www.bibletime.info/.
*
* Copyright 1999-2006 by the BibleTime developers.
* The BibleTime source code is licensed under the GNU General Public License version 2.0.
*
**********/



#include "cdragdropmgr.h"

#include "backend/cswordmoduleinfo.h"
#include "backend/cswordbackend.h"
#include "backend/cswordversekey.h"
#include "util/cpointers.h"

//Sword includes
#include "versekey.h"

//Qt includes
#include <tqevent.h>
#include <tqdom.h>

CDragDropMgr::BTDrag::BTDrag( const TQString& xml, TQWidget* dragSource, const char* name)
: TQTextDrag(xml, dragSource, name) {}
;

//static function to see whether we can decode tje given mime type
bool CDragDropMgr::BTDrag::canDecode( const TQMimeSource * mime ) {
	if ( mime->provides("BibleTime/DND") ) { //we can decode this type!
		return true;
	}
	return false; //not yet implemented
};

bool CDragDropMgr::BTDrag::provides( const char* type ) const {
	return (type == "BibleTime/DND"); //return only true if the type is BibleTime/DND
};

const char* CDragDropMgr::BTDrag::format( int i ) const {
	if ( i == 0) { //we support only one format!
	return "BibleTime/DND";
	};
	return 0;
};

bool CDragDropMgr::BTDrag::decode(const TQMimeSource* e, TQString& str) {
	if (canDecode(e)) {
		str = TQString( e->encodedData( "BibleTime/DND" ) );
		return true;
	}
	return false;
};

bool CDragDropMgr::BTDrag::decode(const TQMimeSource* e, TQString& str, TQCString& /*subtype*/) {
	return decode(e, str);
};

TQByteArray CDragDropMgr::BTDrag::encodedData( const char* /*type*/ ) const {
	return TQTextDrag::encodedData("text/plain"); //hack because TQTextDrag only accepts text/plainand not our BibleTime/DND type
};

///////////////////////////// new class //////////////////////

CDragDropMgr::Item::Item( const TQString& text )
: m_type(Text),
m_bookmarkModuleName(TQString::null),
m_bookmarkKey(TQString::null),
m_bookmarkDescription(TQString::null),
m_text(text) {}

CDragDropMgr::Item::Item( const TQString& moduleName, const TQString& key, const TQString& description  )
: m_type(Bookmark),
m_bookmarkModuleName(moduleName),
m_bookmarkKey(key),
m_bookmarkDescription(description),
m_text(TQString::null) {
	//we have to make sure the key is saved in it's english representation, so we convert it
	if (CSwordModuleInfo* mod = CPointers::backend()->findModuleByName( moduleName )) {
		if (mod->type() == CSwordModuleInfo::Bible || mod->type() == CSwordModuleInfo::Commentary) {
			CSwordVerseKey vk(0);
			vk.key( key );
			vk.setLocale("en");

			m_bookmarkKey = vk.key();
			//      qWarning("english key of %s is %s", key.latin1(), m_bookmarkKey.latin1());
		}
	}
}

CDragDropMgr::Item::~Item() {}

const CDragDropMgr::Item::Type& CDragDropMgr::Item::type() const {
	//returns the type of drag & drop action this item represents
	return m_type;
}

/** Returns the text which is used by this DragDrop Item, only valid if type() == Text */
const TQString& CDragDropMgr::Item::text() const {
	//  Q_ASSERT(!m_text.isEmpty());
	return m_text;
}

/** Returns the key, ony valid if type() == Bookmark */
const TQString& CDragDropMgr::Item::bookmarkKey() const {
	//  Q_ASSERT(!m_bookmarkKey.isEmpty());
	return m_bookmarkKey;
}

/** Returns the bookmark module, ony valid if type() == Bookmark */
const TQString& CDragDropMgr::Item::bookmarkModule() const {
	//  Q_ASSERT(!m_bookmarkModuleName.isEmpty());
	return m_bookmarkModuleName;
}

/** Returns the bookmark description, ony valid if type() == Bookmark */
const TQString& CDragDropMgr::Item::bookmarkDescription() const {
	//  Q_ASSERT(!m_bookmarkDescription.isEmpty());
	return m_bookmarkDescription;
}

////////////////////////////////// NEW CLASS //////////////////////////

CDragDropMgr::CDragDropMgr() {}

CDragDropMgr::~CDragDropMgr() {}

const bool CDragDropMgr::canDecode( const TQMimeSource* const mime ) {
	if (CDragDropMgr::BTDrag::canDecode(mime)) {
		return true;
	}
	else if( TQTextDrag::canDecode(mime) ) {
		qWarning("TQTextDrag can decode this mime!");
		return true;
	};
	return false;
};

TQDragObject* const CDragDropMgr::dragObject( CDragDropMgr::ItemList& items, TQWidget* dragSource ) {
	if ( items.count() ) {
		//process the items and set the data to the dragobject we return later
		TQDomDocument doc("DOC");
		doc.appendChild( doc.createProcessingInstruction( "xml", "version=\"1.0\" encoding=\"UTF-8\"" ) );
		TQDomElement content = doc.createElement("BibleTimeDND");
		content.setAttribute("syntaxVersion", "1.0");
		doc.appendChild(content);

		CDragDropMgr::ItemList::iterator it;
		for ( it = items.begin(); it != items.end(); ++it ) {
			Item item = (*it);
			if (item.type() == Item::Bookmark) { //a bookmark was dragged
				//append the XML stuff for a bookmark
				TQDomElement bookmark = doc.createElement("BOOKMARK");
				bookmark.setAttribute("key", item.bookmarkKey());
				bookmark.setAttribute("description", item.bookmarkDescription());
				bookmark.setAttribute("moduleName", item.bookmarkModule());

				content.appendChild(bookmark);
			}
			else if (item.type() == Item::Text) { //plain text was dragged
				//append the XML stuff for plain text
				TQDomElement plainText = doc.createElement("TEXT");
				plainText.setAttribute("text", item.text());

				content.appendChild(plainText);
			}
		}

		BTDrag* dragObject = new BTDrag( doc.toString(), dragSource );
		//    qWarning("DND data created: %s", (const char*)doc.toString().utf8());
		return dragObject;
	};
	return 0;
};

CDragDropMgr::ItemList CDragDropMgr::decode( const TQMimeSource* const  src) {
	//if the drag was started by another widget which doesn't use CDragDropMgr (a drag created by TQTextDrag)
	if (canDecode(src) && TQTextDrag::canDecode(src)) { //if we can decode but it's a TQTextDrag and not a BTDrag object
		TQString text;
		TQTextDrag::decode(src, text);
		//    qWarning(text.latin1());

		CDragDropMgr::ItemList dndItems;
		dndItems.append( Item(text) );
		return dndItems;
	}
	else if (!canDecode(src)) { //if we can't decode it
		return CDragDropMgr::ItemList();
	};

	TQString xmlData;
	BTDrag::decode(src, xmlData);

	if (xmlData.isEmpty()) { //something went wrong!
		//    qWarning("CDragDropMgr::decode: empty xml data!");
		return CDragDropMgr::ItemList();
	}
	//  else {
	//    qWarning("Drag&Drop data is: %s", xmlData.latin1());
	//  }

	//we can handle the dropEvent and have xml data to work on!
	ItemList dndItems;

	TQDomDocument doc;
	doc.setContent( xmlData );

	TQDomElement document = doc.documentElement();
	if( document.tagName() != "BibleTimeDND" ) { //BibleTime was used in syntax version 1.0
		qWarning("DragDropMgr::decode: Missing BibleTimeDND doc");
		return CDragDropMgr::ItemList();
	}
	// see if there's a section with the name MAINWINDOW
	TQDomElement elem = document.firstChild().toElement();
	while (!elem.isNull()) {
		if (elem.tagName() == "BOOKMARK") { //we found a bookmark!
			//        qWarning("found a bookmark!");
			const TQString key = elem.hasAttribute("key") ? elem.attribute("key") : TQString::null;
			const TQString moduleName = elem.hasAttribute("moduleName") ? elem.attribute("moduleName") : TQString::null;
			const TQString description = elem.hasAttribute("description") ? elem.attribute("description") : TQString::null;

			dndItems.append( CDragDropMgr::Item(moduleName, key, description) );
		}
		else if (elem.tagName() == "TEXT") { //we found a plain text passage!
			const TQString text = elem.hasAttribute("text") ? elem.attribute("text") : TQString::null;
			dndItems.append( CDragDropMgr::Item(text) );
		};
		elem = elem.nextSibling().toElement();
	};

	return dndItems;
};

/** Returns which type the given drop event has, if it's a mixed one (both bookmarks and plain text), which shouldn't happen, it return Item::Unknown. */
CDragDropMgr::Item::Type CDragDropMgr::dndType( const TQMimeSource* e ) {
	ItemList dndItems = decode(e);
	if (dndItems.isEmpty()) {//wrong dropEvent or something strange
		return Item::Unknown;
	};

	//check whether all items have the ssame type, if they do return the type
	//as soon as two items have different types return Item::Unknown
	ItemList::Iterator it;
	Item::Type type = Item::Unknown;
	for( it = dndItems.begin(); it != dndItems.end(); ++it ) {
		if( type == Item::Unknown) { //if Unknown is set this is the first loop, don't return Unknown
			type = (*it).type();
		}
		else if (type != (*it).type() ) {//items have different type, return Item::Unknown
			return Item::Unknown;
		};
	};
	return type;
}
