/*********
*
* This file is part of BibleTime's source code, http://www.bibletime.info/.
*
* Copyright 1999-2006 by the BibleTime developers.
* The BibleTime source code is licensed under the GNU General Public License version 2.0.
*
**********/



//BibleTime includes
#include "bibletime.h"

//frontend includes
#include "frontend/cmdiarea.h"
#include "frontend/cbtconfig.h"
#include "frontend/searchdialog/csearchdialog.h"

//Sword includes
#include <versekey.h>
#include <listkey.h>

//helper function
void BibleTime::syncAllModulesByType(const CSwordModuleInfo::ModuleType type, const TQString& key) {
	qDebug("Syncing modules by type to key %s", key.latin1());

	TQPtrList<TQWidget> windows = m_mdi->usableWindowList();
	for (TQWidget* w = windows.first(); w; w = windows.next()) {
		CDisplayWindow* d = dynamic_cast<CDisplayWindow*>(w);
		Q_ASSERT(d);

		if (d && d->modules().count() && d->modules().first()->type() == type) {
			d->lookup(key);
		}
	}
}

void BibleTime::closeAllModuleWindows() {
	qDebug("DCOP: close all windows now...");

	m_mdi->deleteAll();
}

void BibleTime::syncAllBibles(const TQString& key) {
	qDebug("DCOP: syncing all bibles ...");
	syncAllModulesByType(CSwordModuleInfo::Bible, key);
}

void BibleTime::syncAllCommentaries(const TQString& key) {
	qDebug("DCOP: syncing all commentaries ...");
	syncAllModulesByType(CSwordModuleInfo::Commentary, key);
}

void BibleTime::syncAllLexicons(const TQString& key) {
	qDebug("DCOP: syncing all lexicons ...");
	syncAllModulesByType(CSwordModuleInfo::Lexicon, key);
}

void BibleTime::syncAllVerseBasedModules(const TQString& key) {
	qDebug("DCOP: syncing all verse based modules ...");
	syncAllModulesByType(CSwordModuleInfo::Bible, key);
	syncAllModulesByType(CSwordModuleInfo::Commentary, key);
}

void BibleTime::openWindow(const TQString& moduleName, const TQString& key) {
	qDebug("DCOP: open window for module %s and key %s...", moduleName.latin1(), key.latin1());

	CSwordModuleInfo* module = CPointers::backend()->findModuleByName(moduleName);
	Q_ASSERT(module);
	if (module) {
		createReadDisplayWindow(module, key);
	}
}

void BibleTime::openDefaultBible(const TQString& key) {
	qDebug("DCOP: open default bible ...");
	CSwordModuleInfo* mod = CBTConfig::get
								(CBTConfig::standardBible);
	if (mod) {
		openWindow(mod->name(), key);
	}
}

TQStringList BibleTime::searchInModule(const TQString& moduleName, const TQString& searchText) {
	qDebug("DCOP: searchInModule %s ...", moduleName.latin1());
	TQStringList ret;
	CSwordModuleInfo* mod = CPointers::backend()->findModuleByName(moduleName);

	//TODO: Check this
	Q_ASSERT(mod);
	if (mod) {
		//mod->search(searchText, CSwordModuleSearch::multipleWords, sword::ListKey());
		sword::ListKey scope;
		mod->searchIndexed( searchText, scope );

		sword::ListKey result = mod->searchResult();
		const TQString lead = TQString("[%1] ").arg(moduleName);
		;
		for ( int i = 0; i < result.Count(); ++i ) {
			sword::SWKey* key = result.getElement(i);
			Q_ASSERT(key);


			if (mod->type() == CSwordModuleInfo::Bible || mod->type() == CSwordModuleInfo::Commentary) {
				sword::VerseKey vk(key->getText());
				ret << lead + TQString::fromUtf8( vk.getOSISRef() );
			}
			else {
				ret << lead + TQString::fromUtf8( key->getText() );
			}
		}
	}

	return ret;

}

TQStringList BibleTime::searchInOpenModules(const TQString& searchText) {
	qDebug("DCOP: search in open modules ...");
	TQStringList ret;

	TQWidgetList windows = m_mdi->windowList();
	for ( int i = 0; i < static_cast<int>(windows.count()); ++i ) {
		CDisplayWindow* w = dynamic_cast<CDisplayWindow*>(windows.at(i));
		if (w) {
			ListCSwordModuleInfo windowModules = w->modules();

			ListCSwordModuleInfo::iterator end_it = windowModules.end();
			for (ListCSwordModuleInfo::iterator it(windowModules.begin()); it != end_it; ++it) {
				ret += searchInModule((*it)->name(), searchText);
			};
		};
	};

	return ret;
}

TQStringList BibleTime::searchInDefaultBible(const TQString& searchText) {
	CSwordModuleInfo* bible = CBTConfig::get
								  (CBTConfig::standardBible);
	return searchInModule(bible->name(), searchText);
}

TQString BibleTime::getCurrentReference() {
	qDebug("BibleTime::getCurrentReference");
	TQString ret = TQString::null;

	CDisplayWindow* w = dynamic_cast<CDisplayWindow*>(m_mdi->activeWindow());
	Q_ASSERT(w);

	if (w) {
		TQString modType;
		Q_ASSERT(w->modules().first());
		switch (w->modules().first()->type()) {
			case CSwordModuleInfo::Bible:
				modType = "BIBLE";
				break;
			case CSwordModuleInfo::Commentary:
				modType = "COMMENTARY";
				break;
			case CSwordModuleInfo::GenericBook:
				modType = "BOOK";
				break;
			case CSwordModuleInfo::Lexicon:
				modType = "LEXICON";
				break;
			default:
				modType = "UNSUPPORTED";
				break;
		}

		ret.append("[").append(w->modules().first()->name()).append("] ");
		ret.append("[").append(modType).append("] ");

		CSwordVerseKey* vk = dynamic_cast<CSwordVerseKey*>( w->key() );
		if (vk) {
			ret.append( vk->getOSISRef() );
		}
		else {
			ret.append( w->key()->key() );
		}
	}

	return ret;
}

TQStringList BibleTime::getModulesOfType(const TQString& type) {
	TQStringList ret;

	CSwordModuleInfo::ModuleType modType = CSwordModuleInfo::Unknown;
	if (type == "BIBLES") {
		modType = CSwordModuleInfo::Bible;
	}
	else if (type == "COMMENTARIES") {
		modType = CSwordModuleInfo::Commentary;
	}
	else if (type == "LEXICONS") {
		modType = CSwordModuleInfo::Lexicon;

	}
	else if (type == "BOOKS") {
		modType = CSwordModuleInfo::GenericBook;
	}

	ListCSwordModuleInfo modList = CPointers::backend()->moduleList();
	for (ListCSwordModuleInfo::iterator it( modList.begin() ); it != modList.end(); ++it) {
		if ((*it)->type() == modType) {
			ret.append( (*it)->name() );
		}
	}

	return ret;
}

void BibleTime::reloadModules() {
	//m_backend->reloadModules();
	slotSwordSetupChanged();
}

