/***************************************************************************
 *   KT task editor window implementation                                  *
 *   --------------------------------------------------------------------  *
 *   Copyright (C) 1999, Gary Meyer <gary@meyer.net>                       *
 *   --------------------------------------------------------------------  *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 ***************************************************************************/

#include "kttask.h"

#include <tqlabel.h>
#include <tqstring.h>
#include <tqfileinfo.h>
#include <tqlayout.h>
#include <tqlineedit.h>
#include <tqcheckbox.h>
#include <tqbuttongroup.h>
#include <tqpainter.h>
#include <tqpalette.h>

#include <kapplication.h>
#include <kaccel.h>
#include <klocale.h>
#include <kfiledialog.h>
#include <kmessagebox.h>
#include <kpushbutton.h>
#include <kstdguiitem.h>

#include "cttask.h"

#include "kticon.h"

class KTPushButton : public QPushButton
{
public:
   KTPushButton(TQWidget * parent, const char * name = 0 ) 
     : TQPushButton(parent, name), isSelected(false), isDirty(false)
   {
      updatePalette();
   }

   void updatePalette()
   {
      palNormal = ((TQWidget *)parent())->palette();
      palSelected = palNormal;
      for(int cg = (int) TQPalette::Disabled; cg < (int) TQPalette::NColorGroups; cg++)
      {
        palSelected.setColor((TQPalette::ColorGroup)cg, TQColorGroup::Button, 
                     palSelected.color((TQPalette::ColorGroup)cg, TQColorGroup::Highlight));
        palSelected.setColor((TQPalette::ColorGroup)cg, TQColorGroup::ButtonText, 
                     palSelected.color((TQPalette::ColorGroup)cg, TQColorGroup::HighlightedText));
      }
      isDirty = true;
   }

   bool event( TQEvent *e)
   {
     if (e->type() == TQEvent::ParentPaletteChange)
     {
        updatePalette();
        update();
     }
     return TQPushButton::event(e);
   }
   
   void drawButton ( TQPainter *p )
   {
     if (isDirty || (isOn() != isSelected)) // Prevent infinite recursion
     {
       isDirty = false;
       isSelected = isOn();
       if (isSelected)
         setPalette(palSelected);
       else
         setPalette(palNormal);
     }
     TQPushButton::drawButton(p);
   }
   void drawButtonLabel ( TQPainter *p )
   {
     p->save();
     if (isOn())
     {
       TQFont f = p->font();
       f.setUnderline(true);
       p->setFont(f);
     }
     TQPushButton::drawButtonLabel(p);
     p->restore();
   }
   bool isSelected;
   bool isDirty;
   TQPalette palSelected;
   TQPalette palNormal;
};

KTTask::KTTask(CTTask* _cttask, const TQString & _caption)
       :KDialog( 0, "kttask", true, WStyle_DialogBorder )
{
  cttask = _cttask;

  bool everyDay(true);

  TQVBoxLayout *ml = new TQVBoxLayout( this, KDialogBase::spacingHint() );

  TQHBoxLayout *h1 = new TQHBoxLayout( ml, KDialogBase::spacingHint() );

  // user
  labUser = new TQLabel( i18n("&Run as:"), this, "labUser" );
  h1->addWidget( labUser );

  leUser = new TQLineEdit( this, "leUser");
  labUser->setBuddy(leUser);
  h1->addWidget( leUser );

  if (cttask->system())
  {
    leUser->setText(TQString::fromLocal8Bit(cttask->user.c_str()));
  }
  else
  {
    labUser->hide();
    leUser->hide();
  }

  // icon
  labIcon = new TQLabel(this, "labIcon");
  labIcon->setFixedSize(32, 32);
  h1->addStretch( 1 );
  h1->addWidget( labIcon );

  // comment
  TQHBoxLayout *h2 = new TQHBoxLayout( ml, KDialogBase::spacingHint() );

  labComment = new TQLabel( i18n("&Comment:"), this, "labComment" );
  h2->addWidget( labComment );

  leComment = new TQLineEdit(this, "leComment");
  labComment->setBuddy(leComment);
  h2->addWidget( leComment );

  leComment->setText(TQString::fromLocal8Bit(cttask->comment.c_str()));

  // command
  TQHBoxLayout *h3 = new TQHBoxLayout( ml, KDialogBase::spacingHint() );

  labCommand = new TQLabel( i18n("&Program:"), this, "labCommand" );
  h3->addWidget( labCommand );

  leCommand = new TQLineEdit(this, "leCommand");
  labCommand->setBuddy(leCommand);
  h3->addWidget( leCommand );

  leCommand->setText(TQString::fromLocal8Bit(cttask->command.c_str()));

  labComment->setFixedWidth( QMAX( labComment->width(), labCommand->width()) );
  labCommand->setFixedWidth( QMAX( labComment->width(), labCommand->width()) );

  slotCommandChanged();

  pbBrowse = new TQPushButton(this, "pbBrowse");
  pbBrowse->setText(i18n("&Browse..."));
  h3->addWidget( pbBrowse );
  
  TQHBoxLayout *h3a = new TQHBoxLayout( ml, KDialogBase::spacingHint() );

  // enabled
  chkEnabled = new TQCheckBox(i18n("&Enabled"), this, "chkEnabled");
  chkEnabled->setChecked(cttask->enabled);
  h3a->addWidget( chkEnabled );

  TQHBoxLayout *h4 = new TQHBoxLayout( ml, KDialogBase::spacingHint() );

  ml->addSpacing( 2 * KDialogBase::spacingHint() );

  // months
  bgMonth = new TQButtonGroup( i18n("Months"), this, "bgMonth");
  h4->addWidget( bgMonth );

  TQVBoxLayout *vmonths = new TQVBoxLayout( bgMonth, KDialogBase::spacingHint() );
  vmonths->addSpacing( 2 * KDialogBase::spacingHint() );

  for (int mo = 1; mo <= 12; mo++)
  {
    cbMonth[mo] = new TQCheckBox(bgMonth, "cbMonth");
    cbMonth[mo]->setText(TQString::fromLocal8Bit(cttask->month.getName(mo).c_str()));
    cbMonth[mo]->setChecked(cttask->month.get(mo));
    vmonths->addWidget( cbMonth[mo], AlignLeft );

    if (!cttask->month.get(mo)) everyDay = false;
  }
  pbAllMonths = new TQPushButton(bgMonth, "pbAllMonths");
  pbAllMonths->setText( i18n("Set All") );
  vmonths->addWidget( pbAllMonths, AlignLeft );

  TQVBoxLayout *v1 = new TQVBoxLayout( h4, KDialogBase::spacingHint() );

  // days of the month
  bgDayOfMonth = new TQButtonGroup( i18n("Days of Month"), this, "bgDayOfMonth");
  v1->addWidget( bgDayOfMonth );

  TQPushButton* day;
  TQString tmp;

  TQVBoxLayout *vdays = new TQVBoxLayout( bgDayOfMonth, KDialogBase::spacingHint() );
  vdays->addSpacing( 2 * KDialogBase::spacingHint() );
  TQHBoxLayout *hdays = 0;

  for (int dm = 1; dm <= 31; dm++)
  {
    if( (dm % 7) == 1 )
      hdays = new TQHBoxLayout( vdays, KDialogBase::spacingHint() );

    day = new KTPushButton(bgDayOfMonth);
    day->setFixedSize(25, 25);
    day->setText(tmp.setNum(dm));
    day->setToggleButton(true);
    day->setOn(cttask->dayOfMonth.get(dm));
    pbDayOfMonth[dm] = day;
    if (!cttask->dayOfMonth.get(dm)) everyDay = false;

    hdays->addWidget( day, AlignLeft );
  }
  hdays->addStretch( 1 );
  pbAllDaysOfMonth = new TQPushButton(bgDayOfMonth, "pbAllDaysOfMonth");
  pbAllDaysOfMonth->setText( i18n("Set All") );
  hdays->addWidget( pbAllDaysOfMonth, AlignLeft );

  // days of the week
  bgDayOfWeek = new TQButtonGroup( i18n("Days of Week"), this, "bgDayOfWeek");
  v1->addWidget( bgDayOfWeek );

  TQVBoxLayout *v3 = new TQVBoxLayout( bgDayOfWeek, KDialogBase::spacingHint() );
  v3->addSpacing( 2 * KDialogBase::spacingHint() );

  for (int dw = 1; dw <= 7; dw++)
  {
    cbDayOfWeek[dw] = new TQCheckBox(bgDayOfWeek);
    cbDayOfWeek[dw]->setText(TQString::fromLocal8Bit(cttask->dayOfWeek.getName(dw).c_str()));
    cbDayOfWeek[dw]->setChecked(cttask->dayOfWeek.get(dw));
    v3->addWidget( cbDayOfWeek[dw] );

    if (!cttask->dayOfWeek.get(dw)) everyDay = false;
  }
  pbAllDaysOfWeek = new TQPushButton(bgDayOfWeek, "pbAllDaysOfWeek");
  pbAllDaysOfWeek->setText( i18n("Set All") );
  v3->addWidget( pbAllDaysOfWeek, AlignLeft );

  TQVBoxLayout *v2 = new TQVBoxLayout( h4, KDialogBase::spacingHint() );

  // daily
  bgEveryDay = new TQButtonGroup( i18n("Daily"), this, "bgEveryDay");
  v2->addWidget( bgEveryDay );

  TQVBoxLayout *v9 = new TQVBoxLayout( bgEveryDay, KDialogBase::spacingHint() );
  v9->addSpacing( 2 * KDialogBase::spacingHint() );

  cbEveryDay = new TQCheckBox( i18n("Run every day"), bgEveryDay, "cbEveryDay");
  cbEveryDay->setChecked(everyDay);
  v9->addWidget( cbEveryDay );

  // hours
  bgHour = new TQButtonGroup( i18n("Hours"), this, "bgHour");
  v2->addWidget( bgHour );
  TQVBoxLayout *v4 = new TQVBoxLayout( bgHour, KDialogBase::spacingHint() );
  v4->addSpacing( 2 * KDialogBase::spacingHint() );

  labAM = new TQLabel( i18n("AM"), bgHour, "labAM");
  labAM->setAlignment(AlignRight | AlignVCenter);
  v4->addWidget( labAM );


  for (int ho = 0; ho <= 23; ho++)
  {
    pbHour[ho] = new KTPushButton(bgHour);
    pbHour[ho]->setText(tmp.setNum(ho));
    pbHour[ho]->setToggleButton(true);
    pbHour[ho]->setOn(cttask->hour.get(ho));
    pbHour[ho]->setFixedSize(25, 25);
  }

  TQHBoxLayout *hhours = new TQHBoxLayout( v4, KDialogBase::spacingHint() );
  for (int ho1 = 0; ho1 <= 11; ho1++)
  {
    if( ho1 == 6 )
      hhours = new TQHBoxLayout( v4, KDialogBase::spacingHint() );

    hhours->addWidget( pbHour[ho1] );
  }

  labPM = new TQLabel( i18n("PM"), bgHour, "labPM");
  labPM->setAlignment(AlignRight | AlignVCenter);
  v4->addWidget( labPM );

  hhours = new TQHBoxLayout( v4, KDialogBase::spacingHint() );
  for (int ho1 = 12; ho1 <= 23; ho1++)
  {
    if( ho1 == 18 )
      hhours = new TQHBoxLayout( v4, KDialogBase::spacingHint() );

    hhours->addWidget( pbHour[ho1] );
  }
  
  hhours = new TQHBoxLayout( v4, KDialogBase::spacingHint() );
  pbAllHours = new TQPushButton(bgHour, "pbAllHours");
  pbAllHours->setText( i18n("Set All") );
  hhours->addWidget( pbAllHours, AlignLeft );
  
  // minutes
  bgMinute = new TQButtonGroup( i18n("Minutes"), this, "bgMinute");
  v2->addWidget( bgMinute );
  TQVBoxLayout *vmin = new TQVBoxLayout( bgMinute, KDialogBase::spacingHint() );
  vmin->addSpacing( 2 * KDialogBase::spacingHint() );

  for (int mi = 0; mi <= 55; mi+=5)
  {
    pbMinute[mi] = new KTPushButton(bgMinute);
    pbMinute[mi]->setText(tmp.setNum(mi));
    pbMinute[mi]->setToggleButton(true);
    pbMinute[mi]->setOn(cttask->minute.get(mi));
    pbMinute[mi]->setFixedSize(25, 25);
  }

  TQHBoxLayout *hmin = new TQHBoxLayout( vmin, KDialogBase::spacingHint() );
  for (int mi1 = 0; mi1 <= 55; mi1+=5)
  {
    if( mi1 == 30 )
      hmin = new TQHBoxLayout( vmin, KDialogBase::spacingHint() );

    hmin->addWidget( pbMinute[mi1] );
  }
  
  hmin = new TQHBoxLayout( vmin, KDialogBase::spacingHint() );
  pbAllMinutes = new TQPushButton(bgMinute, "pbAllMinutes");
  pbAllMinutes->setText( i18n("Set All") );
  hmin->addWidget( pbAllMinutes, AlignLeft );

  TQHBoxLayout *h5 = new TQHBoxLayout( ml, KDialogBase::spacingHint() );
  h5->addStretch( 1 );

  // OK
  pbOk = new KPushButton(KStdGuiItem::ok(), this, "pbOk");
  pbOk->setDefault(true);
  h5->addWidget( pbOk );

  // Cancel
  pbCancel = new KPushButton(KStdGuiItem::cancel(), this, "pbCancel");
  h5->addWidget( pbCancel );

  // window
  setIcon(KTIcon::application(true));
  setCaption(_caption/*i18n("Edit Task")*/);

  // set focus to first widget
  if (cttask->system())
  {
    leUser->setFocus();
  }
  else
  {
    leComment->setFocus();
  }

  // connect them up
  connect(pbBrowse, TQT_SIGNAL(clicked()), TQT_SLOT(slotBrowse()));
  connect(leCommand, TQT_SIGNAL(textChanged(const TQString&)),
    TQT_SLOT(slotCommandChanged()));
  connect(cbEveryDay, TQT_SIGNAL(clicked()), TQT_SLOT(slotDailyChanged()));
  connect(pbOk, TQT_SIGNAL(clicked()), TQT_SLOT(slotOK()));
  connect(pbCancel, TQT_SIGNAL(clicked()), TQT_SLOT(slotCancel()));
  connect(pbAllMonths, TQT_SIGNAL(clicked()), TQT_SLOT(slotAllMonths()));
  for (int mo = 1; mo <= 12; mo++) {
    connect(cbMonth[mo], TQT_SIGNAL(clicked()), TQT_SLOT(slotMonthChanged()));
  }
  connect(pbAllDaysOfMonth, TQT_SIGNAL(clicked()), TQT_SLOT(slotAllDaysOfMonth()));
  for (int dm = 1; dm <= 31; dm++)
  {
    connect(pbDayOfMonth[dm], TQT_SIGNAL(clicked()), TQT_SLOT(slotDayOfMonthChanged()));
  }
  connect(pbAllDaysOfWeek, TQT_SIGNAL(clicked()), TQT_SLOT(slotAllDaysOfWeek()));
  for (int dw = 1; dw <= 7; dw++)
  {
    connect(cbDayOfWeek[dw], TQT_SIGNAL(clicked()), TQT_SLOT(slotDayOfWeekChanged()));
  }
  connect(pbAllHours, TQT_SIGNAL(clicked()), TQT_SLOT(slotAllHours()));
  for (int ho = 0; ho <= 23; ho++)
  {
    connect(pbHour[ho], TQT_SIGNAL(clicked()), TQT_SLOT(slotHourChanged()));
  }
  connect(pbAllMinutes, TQT_SIGNAL(clicked()), TQT_SLOT(slotAllMinutes()));
  for (int mi = 0; mi <= 55; mi+=5)
  {
    connect(pbMinute[mi], TQT_SIGNAL(clicked()), TQT_SLOT(slotMinuteChanged()));
  }
 
  // key acceleration
  key_accel = new KAccel(this);
  key_accel->insert(KStdAccel::Open, this, TQT_SLOT(slotOK()));
  key_accel->insert(KStdAccel::Close, this, TQT_SLOT(slotCancel()));
  key_accel->insert(KStdAccel::Quit, this, TQT_SLOT(slotCancel()));
  key_accel->readSettings();

  setFixedSize( minimumSize() );
  slotDailyChanged();
  slotMonthChanged();
  slotDayOfMonthChanged();
  slotDayOfWeekChanged();
  slotHourChanged();
  slotMinuteChanged();
}

KTTask::~KTTask()
{
}

void KTTask::slotCommandChanged()
{
  /*
  TQString qs(leCommand->text());

  int beginPos(qs.findRev("/", qs.length()) + 1);
  if (beginPos < 0) beginPos = 0;

  int endPos(qs.findRev(" ", qs.length()));
  if (endPos < 0) endPos = qs.length();

  TQString iconName(qs.mid(beginPos, endPos-beginPos) + ".xpm");

  TQPixmap qp(KTIcon::getIcon(iconName));
  if (qp.isNull())
    labIcon->setPixmap(KTIcon::task(false));
  else
    labIcon->setPixmap(qp);
  */

  labIcon->setPixmap(KTIcon::task(false));
  return;
}

void KTTask::slotDailyChanged()
{
  if (cbEveryDay->isChecked())
  {
    for (int mo = 1; mo <= 12; mo++)
    {
      cbMonth[mo]->setChecked(true);
      cbMonth[mo]->setEnabled(false);
    }
    for (int dm = 1; dm <= 31; dm++)
    {
      pbDayOfMonth[dm]->setOn(true);
      pbDayOfMonth[dm]->setEnabled(false);
    }
    for (int dw = 1; dw <= 7; dw++)
    {
      cbDayOfWeek[dw]->setChecked(true);
      cbDayOfWeek[dw]->setEnabled(false);
    }
    pbAllMonths->setEnabled(false);
    pbAllDaysOfMonth->setEnabled(false);
    pbAllDaysOfWeek->setEnabled(false);
  }
  else
  {
    for (int mo = 1; mo <= 12; mo++)
    {
      cbMonth[mo]->setEnabled(true);
    }
    for (int dm = 1; dm <= 31; dm++)
    {
      pbDayOfMonth[dm]->setEnabled(true);
    }
    for (int dw = 1; dw <= 7; dw++)
    {
      cbDayOfWeek[dw]->setEnabled(true);
    }
    pbAllMonths->setEnabled(true);
    pbAllDaysOfMonth->setEnabled(true);
    pbAllDaysOfWeek->setEnabled(true);
  }
  slotMonthChanged();
  slotDayOfMonthChanged();
  slotDayOfWeekChanged();
}

void KTTask::slotOK()
{
  // Make it friendly for just selecting days of the month or
  // days of the week.

  int monthDaysSelected(0);
  for (int dm = 1; dm <= 31; dm++)
  {
    if (pbDayOfMonth[dm]->isOn()) monthDaysSelected++;
  }

  int weekDaysSelected(0);
  for (int dw = 1; dw <= 7; dw++)
  {
    if (cbDayOfWeek[dw]->isChecked()) weekDaysSelected++;
  }

  if ((monthDaysSelected == 0) && (weekDaysSelected > 0))
  {
    for (int dm = 1; dm <= 31; dm++)
    {
      pbDayOfMonth[dm]->setOn(1);
    }
  }

  if ((weekDaysSelected == 0) && (monthDaysSelected > 0))
  {
    for (int dw = 1; dw <= 7; dw++)
    {
      cbDayOfWeek[dw]->setChecked(1);
    }
  }

  // Now validate
  TQString message(i18n("Please enter the following to schedule the task:\n"));
  TQString sep("\n- ");
  bool showMessage(false);

  if (leCommand->text().isEmpty())
  {
    message += sep + i18n("the program to run");
    leCommand->setFocus();
    showMessage = true;
  }

  bool valid(false);
  for (int mo = 1; mo <= 12; mo++)
  {
    if (cbMonth[mo]->isChecked()) valid = true;
  }
  if (!valid)
  {
    message += sep + i18n("the months");
    if (!showMessage)
    {
      cbMonth[1]->setFocus();
    }
    showMessage = true;
  }

  valid = false;
  for (int dm = 1; dm <= 31; dm++)
  {
    if (pbDayOfMonth[dm]->isOn()) valid = true;
  }
  for (int dw = 1; dw <= 7; dw++)
  {
    if (cbDayOfWeek[dw]->isChecked()) valid = true;
  }

  if (!valid)
  {
    message += sep +
      i18n("either the days of the month or the days of the week");
    if (!showMessage)
    {
      pbDayOfMonth[1]->setFocus();
    }
    showMessage = true;
  }

  valid = false;
  for (int ho = 0; ho <= 23; ho++)
  {
    if (pbHour[ho]->isOn()) valid = true;
  }

  if (!valid)
  {
    message += sep + i18n("the hours");
    if (!showMessage)
    {
      pbHour[0]->setFocus();
    }
    showMessage = true;
  }

  valid = false;
  for (int mi1 = 0; mi1 <= 55; mi1+=5)
  {
    if (pbMinute[mi1]->isOn()) valid = true;
  }

  if (!valid)
  {
    message += sep + i18n("the minutes");
    if (!showMessage)
    {
      pbMinute[0]->setFocus();
    }
    showMessage = true;
  }

  if (showMessage)
  {
    KMessageBox::information(this, message);
    return;
  }

  // make sure the file name is a good one if we have an
  // absolute path

  TQString qs(leCommand->text());
  if (qs.find("/") == 0)
  {
    int spacePos(qs.find(" "));
    if (spacePos < 0) spacePos = qs.length();
    TQString programName(qs.left(spacePos));
    TQFileInfo file(programName);

    if (!file.isReadable())
    {
      KMessageBox::information(this,
        i18n("Cannot locate program. Please re-enter."));
      leCommand->setFocus();
      return;
    }

    if (!file.isExecutable())
    {
      KMessageBox::information(this,
        i18n("Program is not an executable file. Please re-enter."));
      leCommand->setFocus();
      return;
    }
  }

  // save work in process
  if (!cttask->user.empty())
  {
    cttask->user = TQFile::encodeName(leUser->text());
  }
  else
  {
    cttask->user = "";
  }

  cttask->comment = (const char *)leComment->text().local8Bit();
  cttask->command = (const char *)leCommand->text().local8Bit();
  cttask->enabled = chkEnabled->isChecked();

  for (int mo = 1; mo <= 12; mo++)
  {
    cttask->month.set(mo, cbMonth[mo]->isChecked());
  }

  for (int dm = 1; dm <= 31; dm++)
  {
    cttask->dayOfMonth.set(dm, pbDayOfMonth[dm]->isOn());
  }
  for (int dw = 1; dw <= 7; dw++)
  {
    cttask->dayOfWeek.set(dw, cbDayOfWeek[dw]->isChecked());
  }
  for (int ho = 0; ho <= 23; ho++)
  {
    cttask->hour.set(ho, pbHour[ho]->isOn());
  }
  for (int mi = 0; mi <= 59; mi++)
  {
    cttask->minute.set(mi, false);
  }
  for (int mi1 = 0; mi1 <= 55; mi1+=5)
  {
    cttask->minute.set(mi1, pbMinute[mi1]->isOn());
  }

  close();
}

void KTTask::slotCancel()
{
  close();
}

void KTTask::slotBrowse()
{
  KURL url = KFileDialog::getOpenURL();

  if(!url.isEmpty())
  {
    if(url.isLocalFile())
    {
      leCommand->setText(url.path());
    }
    else
    {
      KMessageBox::sorry(this,
        i18n("Only local or mounted files can be executed by crontab."));
    }
  }

  leCommand->setFocus();
}

void KTTask::slotAllMonths()
{
  if (pbAllMonths->text() == i18n("Set All")) {
    for (int mo = 1; mo <= 12; mo++)
    {
      cbMonth[mo]->setChecked(true);
    }
  }
  else {
    for (int mo = 1; mo <= 12; mo++)
    {
      cbMonth[mo]->setChecked(false);
    }
  }
  slotMonthChanged();
}

void KTTask::slotMonthChanged()
{
  bool allChecked = true;
  bool allCleared = true;
  for (int mo = 1; mo <= 12; mo++)
  {
    if (cbMonth[mo]->isChecked()) {
      allCleared = false;
    }
    else {
      allChecked = false;
    }
  }
  if (allCleared) {
    pbAllMonths->setText( i18n("Set All") );
  }
  else {
    pbAllMonths->setText( i18n("Clear All") );
  }
}

void KTTask::slotAllDaysOfMonth()
{
  if (pbAllDaysOfMonth->text() == i18n("Set All")) {
    for (int dm = 1; dm <= 31; dm++)
    {
      pbDayOfMonth[dm]->setOn(true);
    }
  }
  else {
    for (int dm = 1; dm <= 31; dm++)
    {
      pbDayOfMonth[dm]->setOn(false);
    }
  }
  slotDayOfMonthChanged();
}

void KTTask::slotDayOfMonthChanged()
{
  bool allChecked = true;
  bool allCleared = true;
  for (int dm = 1; dm <= 31; dm++)
  {
    if (pbDayOfMonth[dm]->isOn()) {
      allCleared = false;
    }
    else {
      allChecked = false;
    }
  }
  if (allCleared) {
    pbAllDaysOfMonth->setText( i18n("Set All") );
  }
  else {
    pbAllDaysOfMonth->setText( i18n("Clear All") );
  }
 }

void KTTask::slotAllDaysOfWeek()
{
  if (pbAllDaysOfWeek->text() == i18n("Set All")) {
    for (int dw = 1; dw <= 7; dw++)
    {
      cbDayOfWeek[dw]->setChecked(true);
    }
  }
  else {
    for (int dw = 1; dw <= 7; dw++)
    {
      cbDayOfWeek[dw]->setChecked(false);
    }
  }
  slotDayOfWeekChanged();
}

void KTTask::slotDayOfWeekChanged()
{
  bool allChecked = true;
  bool allCleared = true;
  for (int dw = 1; dw <= 7; dw++)
  {
    if (cbDayOfWeek[dw]->isChecked()) {
      allCleared = false;
    }
    else {
      allChecked = false;
    }
  }
  if (allCleared) {
    pbAllDaysOfWeek->setText( i18n("Set All") );
  }
  else {
    pbAllDaysOfWeek->setText( i18n("Clear All") );
  }
 }

void KTTask::slotAllHours()
{
  if (pbAllHours->text() == i18n("Set All")) {
    for (int ho = 0; ho <= 23; ho++)
    {
      pbHour[ho]->setOn(true);
    }
  }
  else {
    for (int ho = 0; ho <= 23; ho++)
    {
      pbHour[ho]->setOn(false);
    }
  }
  slotHourChanged();
}

void KTTask::slotHourChanged()
{
  bool allChecked = true;
  bool allCleared = true;
  for (int ho = 0; ho <= 23; ho++)
  {
    if (pbHour[ho]->isOn()) {
      allCleared = false;
    }
    else {
      allChecked = false;
    }
  }
  if (allCleared) {
    pbAllHours->setText( i18n("Set All") );
  }
  else {
    pbAllHours->setText( i18n("Clear All") );
  }
 }

void KTTask::slotAllMinutes()
{
  if (pbAllMinutes->text() == i18n("Set All")) {
    for (int mi = 0; mi <= 55; mi+=5)
    {
      pbMinute[mi]->setOn(true);
    }
  }
  else {
    for (int mi = 0; mi <= 55; mi+=5)
    {
      pbMinute[mi]->setOn(false);
    }
  }
  slotMinuteChanged();
}

void KTTask::slotMinuteChanged()
{
  bool allChecked = true;
  bool allCleared = true;
  for (int mi = 0; mi <= 55; mi+=5)
  {
    if (pbMinute[mi]->isOn()) {
      allCleared = false;
    }
    else {
      allChecked = false;
    }
  }
  if (allCleared) {
    pbAllMinutes->setText( i18n("Set All") );
  }
  else {
    pbAllMinutes->setText( i18n("Clear All") );
  }
 }

#include "kttask.moc"
