/***************************************************************************
 *   Copyright (C) 1999, 2000 by Bernd Gehrmann                            *
 *   bernd@kdevelop.org                                                    *
 *   Modified for perforce 2002 by Harald Fernengel <harry@kdevelop.org>   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "commitdlg.h"

#include <tqlayout.h>
#include <tqlabel.h>
#include <tqtextedit.h>
#include <tqpushbutton.h>
#include <tqregexp.h>
#include <kprocess.h>
#include <kapplication.h>
#include <klocale.h>
#include <klineedit.h>
#include <kmessagebox.h>
#include <kdebug.h>

#include <stdlib.h>

#include "execcommand.h"

CommitDialog::CommitDialog( TQWidget *parent, const char *name )
    : KDialogBase( parent, name, true, i18n("Perforce Submit"), Ok|Cancel|Details )
{
    TQWidget *w = new TQWidget( this, "main widget" );
    setMainWidget( w );

    edit = new TQTextEdit( w );
    TQFontMetrics fm(edit->fontMetrics());
    edit->setMinimumSize(fm.width("0")*40, fm.lineSpacing()*3);

    TQVBoxLayout *layout = new TQVBoxLayout( w, 0, spacingHint() );
    TQLabel *editLabel = new TQLabel(i18n("&Enter description:"), w);
    editLabel->setBuddy(edit);
    layout->addWidget(editLabel);
    layout->addWidget(edit);

    w = new TQWidget( this, "details widget" );

    clientEdit = new KLineEdit( w );
    userEdit = new KLineEdit( w );
    filesBox = new KListBox( w );

    layout = new TQVBoxLayout( w, 0, spacingHint() );
    TQLabel *clientLabel = new TQLabel(i18n("C&lient:"), w);
    clientLabel->setBuddy(clientEdit);
    layout->addWidget(clientLabel);
    layout->addWidget( clientEdit );
    TQLabel *userLabel = new TQLabel(i18n("&User:"), w);
    userLabel->setBuddy(userEdit);
    layout->addWidget( userLabel );
    layout->addWidget( userEdit );
    TQLabel *filesLabel = new TQLabel(i18n("&File(s):"), w);
    filesLabel->setBuddy(filesBox);
    layout->addWidget( filesLabel );
    layout->addWidget( filesBox );

    setDetailsWidget( w );
    autoGuess();
    edit->setFocus();
}

CommitDialog::~CommitDialog()
{
}

void CommitDialog::autoGuess()
{
    char *cenv;

    cenv = getenv( "P4USER" );
    if ( cenv ) {
        setUser( TQString::fromLocal8Bit( cenv ) );
    }

    cenv = getenv( "P4CLIENT" );
    if ( cenv ) {
        setClient( TQString::fromLocal8Bit( cenv ) );
    }
}

void CommitDialog::setFiles( const TQStringList& lst )
{
    filesBox->clear();
    setDepotFiles( lst );
}

void CommitDialog::setDepotFiles( const TQStringList& lst )
{
    TQStringList args;

    args << "files";
    for ( TQStringList::ConstIterator it = lst.begin(); it != lst.end(); ++it ) {
        args << (*it);
    }

    ExecCommand* cmd = new ExecCommand( "p4", args, TQString::null, TQStringList(), this );
    connect( cmd, TQT_SIGNAL(finished( const TQString&, const TQString& )),
	     this, TQT_SLOT(getFilesFinished( const TQString&, const TQString& )) );
}

void CommitDialog::getFilesFinished( const TQString& out, const TQString& /* err */ )
{
    TQStringList lst = TQStringList::split( TQChar('\n'), out );
    for ( TQStringList::ConstIterator it = lst.begin(); it != lst.end(); ++it ) {
        int pos = (*it).find( TQChar('#') );
        if ( pos > 1 && (*it).startsWith( "//" ) ) {
            filesBox->insertItem( (*it).left( pos ) );
        }
    }
}

TQString CommitDialog::changeList() const
{
    TQString lst;

    lst += "Change: new\n"
           "Client: " + client() + "\n"
           "User: " + user() + "\n"
           "Status: new\n"
           "Description:\n        ";

    lst += logMessage().replace(TQRegExp("\n"), "\n        ") + "\n\n";

    lst += "Files:\n";

    for ( uint i = 0; i < filesBox->count(); ++i ) {
        lst += "       " + filesBox->text( i ) + "\n";
    }

    return lst;
}

void CommitDialog::accept()
{
    if ( client().isEmpty() ) {
        setDetails( true );
        KMessageBox::error( this, i18n("Please enter the P4 client name.") );
        clientEdit->setFocus();
    } else if ( user().isEmpty() ) {
        setDetails( true );
        KMessageBox::error( this, i18n("Please enter the P4 user.") );
        userEdit->setFocus();
    } else if ( filesBox->count() == 0 ) {
        setDetails( true );
        KMessageBox::error( this, i18n("The changelist does not contain any files.") );
    } else {
        KDialogBase::accept();
    }
}

#include "commitdlg.moc"
