/***************************************************************************
 *   Copyright (C) 2002 by Bernd Gehrmann                                  *
 *   bernd@kdevelop.org                                                    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "toolsconfigwidget.h"

#include <tqcheckbox.h>
#include <tqlistbox.h>
#include <tqtimer.h>

#include <klineedit.h>
#include <kconfig.h>
#include <kdeversion.h>
#include <kdebug.h>
#include <kdesktopfile.h>
#include <kiconloader.h>
#include <kmessagebox.h>
#include <kurl.h>
#include <kurldrag.h>

#include "addtooldlg.h"
#include "kapplicationtree.h"


struct ToolsConfigEntry
{
    TQString menutext;
    TQString cmdline;
    bool isdesktopfile;
    bool captured;
    bool isEmpty() const {
        return ( menutext.isEmpty() && cmdline.isEmpty() );
    }
};


ToolsConfigWidget::ToolsConfigWidget(TQWidget *parent, const char *name)
    : ToolsConfigWidgetBase(parent, name)
{
    m_toolsmenuEntries.setAutoDelete(true);
    m_filecontextEntries.setAutoDelete(true);
    m_dircontextEntries.setAutoDelete(true);

    toolsmenuBox->setAcceptDrops(true);
    toolsmenuBox->installEventFilter(this);
    toolsmenuBox->viewport()->setAcceptDrops(true);
    toolsmenuBox->viewport()->installEventFilter(this);

    readConfig();
}


ToolsConfigWidget::~ToolsConfigWidget()
{}


void ToolsConfigWidget::readGroup(const TQString &group, TQDict<ToolsConfigEntry> *entryDict)
{
    KConfig *config = ToolsFactory::instance()->config();
    config->setGroup("External Tools");
    TQStringList list = config->readListEntry(group);

    TQStringList::ConstIterator it;
    for (it = list.begin(); it != list.end(); ++it) {
        config->setGroup(group + " " + (*it));
        TQString cmdline = config->readPathEntry("CommandLine");
        bool isdesktopfile = config->readBoolEntry("DesktopFile");
        bool captured = config->readBoolEntry("Captured");
        ToolsConfigEntry *entry = new ToolsConfigEntry;
        entry->menutext = (*it);
        entry->cmdline = cmdline;
        entry->isdesktopfile = isdesktopfile;
        entry->captured = captured;
        entryDict->insert(*it, entry);
    }
}


void ToolsConfigWidget::storeGroup(const TQString &group, const TQDict<ToolsConfigEntry> &entryDict)
{
    KConfig *config = ToolsFactory::instance()->config();

    TQStringList list;

    TQDictIterator<ToolsConfigEntry> it(entryDict);
    for (; it.current(); ++it) {
        ToolsConfigEntry *entry = it.current();
        list << entry->menutext;
        config->setGroup(group + " " + entry->menutext);
        config->writePathEntry("CommandLine", entry->cmdline);
        config->writeEntry("DesktopFile", entry->isdesktopfile);
        config->writeEntry("Captured", entry->captured);
    }

    config->setGroup("External Tools");
    config->writeEntry(group, list);
}



void ToolsConfigWidget::fillListBox(TQListBox *lb, const TQDict<ToolsConfigEntry> &entryDict)
{
    lb->clear();

    TQDictIterator<ToolsConfigEntry> it(entryDict);
    for (; it.current(); ++it) {
        ToolsConfigEntry *entry = it.current();
        if (entry->isdesktopfile) {
            KDesktopFile df(entry->cmdline);
            lb->insertItem(SmallIcon(df.readIcon()), entry->menutext);
        } else {
            lb->insertItem(entry->menutext);
        }
    }
}


bool ToolsConfigWidget::addEntry(ToolsConfigEntry *entry, TQDict<ToolsConfigEntry> *entryDict)
{
    TQString menutext = entry->menutext;
    if (entryDict->find(menutext)) {
        delete entry;
        KMessageBox::sorry(this, i18n("An entry with this title exists already."));
        return false;
    }

    entryDict->insert(menutext, entry);

    updateListBoxes();
    return true;
}


void ToolsConfigWidget::readConfig()
{
    readGroup("Tool Menu", &m_toolsmenuEntries);
    readGroup("File Context", &m_filecontextEntries);
    readGroup("Dir Context", &m_dircontextEntries);

    updateListBoxes();
}


void ToolsConfigWidget::storeConfig()
{
    storeGroup("Tool Menu", m_toolsmenuEntries);
    storeGroup("File Context", m_filecontextEntries);
    storeGroup("Dir Context", m_dircontextEntries);
}


void ToolsConfigWidget::updateListBoxes()
{
    fillListBox(toolsmenuBox, m_toolsmenuEntries);
    fillListBox(filecontextBox, m_filecontextEntries);
    fillListBox(dircontextBox, m_dircontextEntries);
}


void ToolsConfigWidget::toolsmenuaddClicked()
{
    AddToolDialog dlg(this);
    dlg.setCaption(i18n("Add to Tools Menu"));
    dlg.tree->setFocus();
    while (dlg.exec()) {
        ToolsConfigEntry *entry = new ToolsConfigEntry;
        entry->menutext = dlg.menutextEdit->text();
        entry->cmdline = dlg.getApp().stripWhiteSpace();
        entry->isdesktopfile = false;
        entry->captured = dlg.capturedBox->isChecked();
        if ( entry->isEmpty() )
            delete entry;
        else if (addEntry(entry, &m_toolsmenuEntries))
            return;
    }
}


void ToolsConfigWidget::toolsmenuremoveClicked()
{
    TQString menutext = toolsmenuBox->currentText();
    m_toolsmenuEntries.remove(menutext);
    updateListBoxes();
}


void ToolsConfigWidget::filecontextaddClicked()
{
    AddToolDialog dlg(this);
    dlg.setCaption(i18n("Add to File Context Menus"));
    dlg.tree->setFocus();
    while (dlg.exec()) {
        ToolsConfigEntry *entry = new ToolsConfigEntry;
        entry->menutext = dlg.menutextEdit->text();
        entry->cmdline = dlg.getApp().stripWhiteSpace();
        entry->isdesktopfile = false;
        entry->captured = dlg.capturedBox->isChecked();
        if ( entry->isEmpty() )
            delete entry;

        else if (addEntry(entry, &m_filecontextEntries))
            return;
    }
}


void ToolsConfigWidget::filecontextremoveClicked()
{
    TQString menutext = filecontextBox->currentText();
    m_filecontextEntries.remove(menutext);
    updateListBoxes();
}


void ToolsConfigWidget::dircontextaddClicked()
{
    AddToolDialog dlg(this);
    dlg.setCaption(i18n("Add to Directory Context Menus"));
    dlg.tree->setFocus();
    if (dlg.exec()) {
        ToolsConfigEntry *entry = new ToolsConfigEntry;
        entry->menutext = dlg.menutextEdit->text();
        entry->cmdline = dlg.getApp().stripWhiteSpace();
        entry->isdesktopfile = false;
        entry->captured = dlg.capturedBox->isChecked();
        if ( entry->isEmpty() )
            delete entry;
        else if (addEntry(entry, &m_dircontextEntries))
            return;
    }
}


void ToolsConfigWidget::dircontextremoveClicked()
{
    TQString menutext = dircontextBox->currentText();
    m_dircontextEntries.remove(menutext);
    updateListBoxes();
}


bool ToolsConfigWidget::eventFilter(TQObject *o, TQEvent *e)
{
    if (e->type() == TQEvent::DragEnter || e->type() == TQEvent::DragMove) {
        TQDragMoveEvent *dme = static_cast<TQDragMoveEvent*>(e);
        if (KURLDrag::canDecode(dme))
            dme->accept();
        return true;
    } else if (e->type() == TQEvent::Drop) {
        TQDropEvent *de = static_cast<TQDropEvent*>(e);
        KURL::List fileList;
        if (KURLDrag::decode(de, fileList)) {
            KURL::List::ConstIterator it;
            for (it = fileList.begin(); it != fileList.end(); ++it) {
                if ((*it).isLocalFile() && KDesktopFile::isDesktopFile((*it).path())) {
                    KDesktopFile df((*it).path());
                    ToolsConfigEntry *entry = new ToolsConfigEntry;
                    entry->menutext = df.readName();
                    entry->cmdline = (*it).path();
                    entry->isdesktopfile = true;
                    entry->captured = false;
                    addEntry(entry, &m_toolsmenuEntries);
                }
            }
        }
        return true;
    }

    return ToolsConfigWidgetBase::eventFilter(o, e);
}


void ToolsConfigWidget::accept()
{
    storeConfig();
}
#include "toolsconfigwidget.moc"
