/* This file is part of the KDE project
   Copyright (C) 2006 David Nolden <david.nolden.kdevelop@art-master.de>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public License
   along with this library; see the file COPYING.LIB.  If not, write to
   the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
   Boston, MA 02110-1301, USA.
*/

#ifndef FANCYLISTVIEWITEM
#define FANCYLISTVIEWITEM

#include <tqvaluevector.h>
#include <tqpainter.h>
#include <tqfont.h>
#include <tqlistview.h>
#include <klistview.h>

class TextPaintStyleStore {
    public:
        class Item {
            public:
            TQFont font;
            TQColor color;
            TQColor background;
            
            Item(const TQFont& f = TQFont(), const TQColor& c = TQColor(), const TQColor b = TQColor() ) : font(f), color(c), background(b) {
            }
            
            bool bgValid() {
                return background.isValid();
            }
            
            bool colValid() {
                return color.isValid();
            }
        };
        
        typedef TQMap<int, Item> Store ;
    
        TextPaintStyleStore( TQFont defaultFont=TQFont() ) {
            m_styles.insert( 0, Item( defaultFont ) );
        }
        
        Item& getStyle( int num ) {
            Store::Iterator it = m_styles.find( num );
            if( it != m_styles.end() ) return *it;
            return m_styles[0];
        }
        
        void addStyle( int num, Item& style ) {
            m_styles[num] = style;
        }
        
        void addStyle( int num, const TQFont& font ) {
            m_styles[num] = Item( font );
        }
        
        bool hasStyle( int num ) {
            Store::Iterator it = m_styles.find( num );
            return ( it != m_styles.end() );
        }
        
    private:
        Store m_styles;
};

class TextPaintItem {
    public:
        struct Item {
            TQString text;
            int style;
            
            Item( const TQString& t = "", int st = 0 ) : text(t), style(st) {
            }
            
        };
        typedef TQValueList<Item> Chain;
        
        Chain& items() {
            return m_chain;
        }
        
        TextPaintItem(const TQString& text="") {
            addItem( text );
        }
        
        Item& addItem(const TQString& item, int style = 0) {
            m_chain.append( Item(item, style) );
            return m_chain.back();
        }
        
        void clear() {
            m_chain.clear();
        }
        
        operator TQString () const {
            TQString ret;
            Chain::const_iterator it = m_chain.begin();
            while(it != m_chain.end()) {
                ret += (*it).text;
                ++it;
            }
            return ret;
        }
        
    private:
        Chain m_chain;
};

///does not support multiple column, a "column" represents a part of the real first column
///KListViewItem is only needed for the background-color

class FancyListViewItem : public KListViewItem
{
    public:
        FancyListViewItem(TextPaintStyleStore& styles, TQListView *parent, const TQString &label1, const TQString &label2="") : KListViewItem(parent, label1, label2), m_styles(styles) {
            init(label1, label2);
        }
        
        FancyListViewItem(TextPaintStyleStore& styles, TQListViewItem *parent, const TQString &label1, const TQString &label2="") : KListViewItem(parent, label1, label2), m_styles(styles) {
            init(label1, label2);
        }
        
        virtual void paintCell(TQPainter *painter, const TQColorGroup &cg, int column, int width, int align);
        virtual int width(const TQFontMetrics &fm, const TQListView *lv, int column);
        virtual void setText ( int column, const TQString & text );
        virtual TQString text(int column) const;
        
        inline void clear() {
            m_items.clear();
        }
        
        inline TextPaintItem& item(int column = 0) {
            if(m_items.isEmpty()) {
                m_items.append( TextPaintItem() );
            }
            
            return m_items[column];
        }
        
        void setItem(int column, TextPaintItem item);
    private:
        virtual TQColor backgroundColor(int col);
        void init(const TQString& label1, const TQString& label2);
        int textWidth(const TQFont& font, const TQString& text);
        TQValueVector<TextPaintItem> m_items;
    protected:
        TextPaintStyleStore& m_styles;
};

#endif
