/* This file is part of KCachegrind.
   Copyright (C) 2002, 2003 Josef Weidendorfer <Josef.Weidendorfer@gmx.de>

   KCachegrind is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation, version 2.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; see the file COPYING.  If not, write to
   the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
   Boston, MA 02110-1301, USA.
*/

/*
 * Configuration Dialog for KCachegrind
 */

#include <tqcombobox.h>
#include <tqcheckbox.h>
#include <tqlineedit.h>
#include <tqlistview.h>
#include <tqdict.h>
#include <tqmessagebox.h>

#include <kcolorbutton.h>
#include <kfiledialog.h>
#include <klocale.h>
#include <knumvalidator.h>

#include "configdlg.h"
#include "tracedata.h"
#include "configuration.h"


ConfigDlg::ConfigDlg(Configuration* c, TraceData* data,
                     TQWidget* parent, const char* name)
  :ConfigDlgBase(parent, name)
{
  _config = c;
  _data = data;
  _objectCS = 0;
  _classCS = 0;
  _fileCS = 0;
  KIntValidator * numValidator = new KIntValidator( this );
  maxListEdit->setValidator(numValidator );
  symbolCount->setValidator(numValidator );
  symbolLength->setValidator(numValidator );
  precisionEdit->setValidator(numValidator );
  contextEdit->setValidator(numValidator );

#if 0
  TQListViewItem *oItem, *fItem, *cItem, *fnItem;
  oItem = new(colorList, i18n("ELF Objects"));

  fItem = new(colorList, i18n("Source Files"));
  cItem = new(colorList, i18n("C++ Classes"));
  fnItem = new(colorList, i18n("Function (no Grouping)"));
#endif

  connect(objectCombo, TQT_SIGNAL(activated(const TQString &)),
          this, TQT_SLOT(objectActivated(const TQString &)));
  connect(objectCombo, TQT_SIGNAL(textChanged(const TQString &)),
          this, TQT_SLOT(objectActivated(const TQString &)));
  connect(objectCheck, TQT_SIGNAL(toggled(bool)),
          this, TQT_SLOT(objectCheckChanged(bool)));
  connect(objectColor, TQT_SIGNAL(changed(const TQColor &)),
          this, TQT_SLOT(objectColorChanged(const TQColor &)));

  connect(classCombo, TQT_SIGNAL(activated(const TQString &)),
          this, TQT_SLOT(classActivated(const TQString &)));
  connect(classCombo, TQT_SIGNAL(textChanged(const TQString &)),
          this, TQT_SLOT(classActivated(const TQString &)));
  connect(classCheck, TQT_SIGNAL(toggled(bool)),
          this, TQT_SLOT(classCheckChanged(bool)));
  connect(classColor, TQT_SIGNAL(changed(const TQColor &)),
          this, TQT_SLOT(classColorChanged(const TQColor &)));

  connect(fileCombo, TQT_SIGNAL(activated(const TQString &)),
          this, TQT_SLOT(fileActivated(const TQString &)));
  connect(fileCombo, TQT_SIGNAL(textChanged(const TQString &)),
          this, TQT_SLOT(fileActivated(const TQString &)));
  connect(fileCheck, TQT_SIGNAL(toggled(bool)),
          this, TQT_SLOT(fileCheckChanged(bool)));
  connect(fileColor, TQT_SIGNAL(changed(const TQColor &)),
          this, TQT_SLOT(fileColorChanged(const TQColor &)));

  TQString objectPrefix = TraceCost::typeName(TraceCost::Object);
  TQString classPrefix = TraceCost::typeName(TraceCost::Class);
  TQString filePrefix = TraceCost::typeName(TraceCost::File);

  objectCombo->setDuplicatesEnabled(false);
  classCombo->setDuplicatesEnabled(false);
  fileCombo->setDuplicatesEnabled(false);
  objectCombo->setAutoCompletion(true);
  classCombo->setAutoCompletion(true);
  fileCombo->setAutoCompletion(true);

  // first unspecified cost items from data
  TraceObjectMap::Iterator oit;
  TQStringList oList;
  for ( oit = data->objectMap().begin();
        oit != data->objectMap().end(); ++oit )
    oList.append((*oit).prettyName());

  TraceClassMap::Iterator cit;
  TQStringList cList;
  for ( cit = data->classMap().begin();
        cit != data->classMap().end(); ++cit )
    cList.append((*cit).prettyName());

  TraceFileMap::Iterator fit;
  TQStringList fList;
  for ( fit = data->fileMap().begin();
        fit != data->fileMap().end(); ++fit )
    fList.append((*fit).prettyName());

  // then already defined colors (have to check for duplicates!)
  TQDictIterator<Configuration::ColorSetting> it( c->_colors );
  for( ; it.current(); ++it ) {
    if ((*it)->automatic) continue;

    TQString n = it.currentKey();
    if (n.startsWith(objectPrefix)) {
      n = n.remove(0, objectPrefix.length()+1);
      if (oList.findIndex(n) == -1) oList.append(n);
    }
    else if (n.startsWith(classPrefix)) {
      n = n.remove(0, classPrefix.length()+1);
      if (cList.findIndex(n) == -1) cList.append(n);
    }
    else if (n.startsWith(filePrefix)) {
      n = n.remove(0, filePrefix.length()+1);
      if (fList.findIndex(n) == -1) fList.append(n);
    }
  }

  oList.sort();
  cList.sort();
  fList.sort();
  objectCombo->insertStringList(oList);
  classCombo->insertStringList(cList);
  fileCombo->insertStringList(fList);

  objectActivated(objectCombo->currentText());
  classActivated(classCombo->currentText());
  fileActivated(fileCombo->currentText());

  maxListEdit->setText(TQString::number(c->_maxListCount));

  _dirItem = 0;

  TQListViewItem* i = new TQListViewItem(dirList, i18n("(always)"));
  i->setOpen(true);
  TQStringList::Iterator sit = c->_generalSourceDirs.begin();
  for(; sit != c->_generalSourceDirs.end(); ++sit ) {
    TQString d = (*sit);
    if (d.isEmpty()) d = "/";
    new TQListViewItem(i, d);
  }
  for ( oit = data->objectMap().begin();
        oit != data->objectMap().end(); ++oit ) {
    TQString n = (*oit).name();
    i = new TQListViewItem(dirList, n);
    i->setOpen(true);
    TQStringList* dirs = c->_objectSourceDirs[n];
    if (!dirs) continue;

    sit = dirs->begin();
    for(; sit != dirs->end(); ++sit ) {
      TQString d = (*sit);
      if (d.isEmpty()) d = "/";
      new TQListViewItem(i, d);
    }
  }

  connect(dirList, TQT_SIGNAL(selectionChanged(TQListViewItem*)),
          this, TQT_SLOT(dirsItemChanged(TQListViewItem*)));
  connect(addDirButton, TQT_SIGNAL(clicked()),
          this, TQT_SLOT(dirsAddPressed()));
  connect(deleteDirButton, TQT_SIGNAL(clicked()),
          this, TQT_SLOT(dirsDeletePressed()));
  dirList->setSelected(dirList->firstChild(), true);

  symbolCount->setText(TQString::number(c->_maxSymbolCount));
  symbolLength->setText(TQString::number(c->_maxSymbolLength));
  precisionEdit->setText(TQString::number(c->_percentPrecision));
  contextEdit->setText(TQString::number(c->_context));
}

ConfigDlg::~ConfigDlg()
{
}

bool ConfigDlg::configure(Configuration* c, TraceData* d, TQWidget* p)
{
  ConfigDlg dlg(c, d, p);

  if (dlg.exec()) {

    bool ok;
    int newValue = dlg.maxListEdit->text().toUInt(&ok);
    if (ok && newValue < 500)
      c->_maxListCount = newValue;
    else
      TQMessageBox::warning(p, i18n("KCachegrind Configuration"),
                           i18n("The Maximum Number of List Items should be below 500."
                                "The previous set value (%1) will still be used.")
                           .arg(TQString::number(c->_maxListCount)),
                           TQMessageBox::Ok, 0);

    c->_maxSymbolCount = dlg.symbolCount->text().toInt();
    c->_maxSymbolLength = dlg.symbolLength->text().toInt();
    c->_percentPrecision = dlg.precisionEdit->text().toInt();
    c->_context = dlg.contextEdit->text().toInt();
    return true;
  }
  return false;
}

void ConfigDlg::objectActivated(const TQString & s)
{
//  qDebug("objectActivated: %s", s.ascii());

  if (s.isEmpty()) { _objectCS=0; return; }

  TQString n = TraceCost::typeName(TraceCost::Object) + "-" + s;

  Configuration* c = Configuration::config();
  Configuration::ColorSetting* cs = c->_colors[n];
  if (!cs)
    cs = Configuration::color(n);
//  else
//    qDebug("found color %s", n.ascii());

  _objectCS = cs;

  objectCheck->setChecked(cs->automatic);
  objectColor->setColor(cs->color);

  /*
  qDebug("Found Color %s, automatic to %s",
         _objectCS->name.ascii(),
         _objectCS->automatic ? "true":"false");
  */
}


void ConfigDlg::objectCheckChanged(bool b)
{
  if (_objectCS) {
    _objectCS->automatic = b;
    /*
    qDebug("Set Color %s automatic to %s",
           _objectCS->name.ascii(),
           _objectCS->automatic ? "true":"false");
    */
  }
}

void ConfigDlg::objectColorChanged(const TQColor & c)
{
  if (_objectCS) _objectCS->color = c;
}

void ConfigDlg::classActivated(const TQString & s)
{
//  qDebug("classActivated: %s", s.ascii());

  if (s.isEmpty()) { _classCS=0; return; }

  TQString n = TraceCost::typeName(TraceCost::Class) + "-" + s;

  Configuration* c = Configuration::config();
  Configuration::ColorSetting* cs = c->_colors[n];
  if (!cs)
    cs = Configuration::color(n);

  _classCS = cs;

  classCheck->setChecked(cs->automatic);
  classColor->setColor(cs->color);

}


void ConfigDlg::classCheckChanged(bool b)
{
  if (_classCS) _classCS->automatic = b;
}

void ConfigDlg::classColorChanged(const TQColor & c)
{
  if (_classCS) _classCS->color = c;
}


void ConfigDlg::fileActivated(const TQString & s)
{
//  qDebug("fileActivated: %s", s.ascii());

  if (s.isEmpty()) { _fileCS=0; return; }

  TQString n = TraceCost::typeName(TraceCost::File) + "-" + s;

  Configuration* c = Configuration::config();
  Configuration::ColorSetting* cs = c->_colors[n];
  if (!cs)
    cs = Configuration::color(n);

  _fileCS = cs;

  fileCheck->setChecked(cs->automatic);
  fileColor->setColor(cs->color);
}


void ConfigDlg::fileCheckChanged(bool b)
{
  if (_fileCS) _fileCS->automatic = b;
}

void ConfigDlg::fileColorChanged(const TQColor & c)
{
  if (_fileCS) _fileCS->color = c;
}


void ConfigDlg::dirsItemChanged(TQListViewItem* i)
{
  _dirItem = i;
  deleteDirButton->setEnabled(i->depth() == 1);
  addDirButton->setEnabled(i->depth() == 0);
}

void ConfigDlg::dirsDeletePressed()
{
  if (!_dirItem || (_dirItem->depth() == 0)) return;
  TQListViewItem* p = _dirItem->parent();
  if (!p) return;

  Configuration* c = Configuration::config();
  TQString objName = p->text(0);

  TQStringList* dirs;
  if (objName == i18n("(always)"))
    dirs = &(c->_generalSourceDirs);
  else
    dirs = c->_objectSourceDirs[objName];
  if (!dirs) return;

  dirs->remove(_dirItem->text(0));
  delete _dirItem;
  _dirItem = 0;

  deleteDirButton->setEnabled(false);
}

void ConfigDlg::dirsAddPressed()
{
  if (!_dirItem || (_dirItem->depth() >0)) return;

  Configuration* c = Configuration::config();
  TQString objName = _dirItem->text(0);

  TQStringList* dirs;
  if (objName == i18n("(always)"))
    dirs = &(c->_generalSourceDirs);
  else {
    dirs = c->_objectSourceDirs[objName];
    if (!dirs) {
      dirs = new TQStringList;
      c->_objectSourceDirs.insert(objName, dirs);
    }
  }

  TQString newDir;
  newDir = KFileDialog::getExistingDirectory(TQString::null,
                                             this,
                                             i18n("Choose Source Folder"));
  if (newDir.isEmpty()) return;

  // even for "/", we strip the tailing slash
  if (newDir.endsWith("/"))
    newDir = newDir.left(newDir.length()-1);

  if (dirs->findIndex(newDir)>=0) return;

  dirs->append(newDir);
  if (newDir.isEmpty()) newDir = TQString("/");
  new TQListViewItem(_dirItem, newDir);
}

#include "configdlg.moc"
