/* ****************************************************************************
  This file is part of KBabel

  Copyright (C) 2000 by Matthias Kiefer
                            <matthias.kiefer@gmx.de>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.

  In addition, as a special exception, the copyright holders give
  permission to link the code of this program with any edition of
  the Qt library by Trolltech AS, Norway (or with modified versions
  of Qt that use the same license as Qt), and distribute linked
  combinations including the two.  You must obey the GNU General
  Public License in all respects for all of the code used other than 
  Qt. If you modify this file, you may extend this exception to
  your version of the file, but you are not obligated to do so.  If
  you do not wish to do so, delete this exception statement from
  your version.

**************************************************************************** */


#include "kbabeldictview.h"
#include "kbabeldictbox.h"
#include "searchengine.h"


#include <tqcheckbox.h>
#include <tqgroupbox.h>
#include <tqhbox.h>
#include <tqlabel.h>
#include <tqlayout.h>
#include <tqptrlist.h>
#include <tqpushbutton.h>
#include <tqstringlist.h>
#include <tqwidgetstack.h>
#include <tqsplitter.h>
#include <tqvbox.h>

#include <kcombobox.h>
#include <kconfig.h>
#include <kdialog.h>
#include <kglobal.h>
#include <klineedit.h>
#include <klocale.h>
#include <kprogress.h>
#include <kseparator.h>
#include <kdebug.h>
#include <kiconloader.h>

/* 
 *  Constructs a KBabelDictView which is a child of 'parent', with the 
 *  name 'name' and widget flags set to 'f' 
 */
KBabelDictView::KBabelDictView( TQWidget* parent,  const char* name, WFlags fl )
    : TQWidget( parent, name, fl )
{
    TQVBoxLayout    *mainLayout = new TQVBoxLayout(this);
    mainLayout->setSpacing(KDialog::spacingHint());
    mainLayout->setMargin(KDialog::marginHint());

    splitter = new TQSplitter(this);
    mainLayout->addWidget(splitter);

    TQWidget *w = new TQWidget(splitter);
    TQVBoxLayout *wLayout= new TQVBoxLayout(w);
    wLayout->setSpacing(KDialog::spacingHint());
    wLayout->setMargin(KDialog::marginHint());
    
    TQHBoxLayout *hbox = new TQHBoxLayout(wLayout);
    TQLabel *label = new TQLabel(i18n("Search in module:"), w);
    hbox->addWidget(label);
    moduleCombo = new KComboBox(w);
    hbox->addWidget(moduleCombo);

    TQWidget *temp = new TQWidget(w);
    hbox->addWidget(temp);
    hbox->setStretchFactor(temp,2);
    editButton = new TQPushButton(i18n("&Edit"),w);
    editButton->setEnabled(false);
    hbox->addWidget(editButton);

    // added a button "clear search" here
    hbox = new TQHBoxLayout(wLayout);
    TQPushButton* clearButton = new TQPushButton(w);
    clearButton->setFlat(true);
    clearButton->setPixmap(SmallIcon("locationbar_erase"));
    hbox->addWidget(clearButton);
    textEdit = new KLineEdit(w,"textedit");
    textEdit->setFocus();
    hbox->addWidget(textEdit);

    hbox = new TQHBoxLayout(wLayout);
    startButton = new TQPushButton(i18n("&Start Search"),w);
    hbox->addWidget(startButton);
    inTransButton = new TQCheckBox(i18n("Sea&rch in translations"),w);
    hbox->addWidget(inTransButton);
    hbox->addStretch(1);
    stopButton = new TQPushButton(i18n("S&top"),w);
    stopButton->setEnabled(false);
    hbox->addWidget(stopButton);

    KSeparator *sep = new KSeparator(w);
    wLayout->addWidget(sep);
    dictBox = new KBabelDictBox(w,"kbabeldictbox");
    wLayout->addWidget(dictBox);

    prefWidget = new TQWidget(splitter);
    TQVBoxLayout *tempLayout= new TQVBoxLayout(prefWidget);
    tempLayout->setSpacing(KDialog::spacingHint());
    tempLayout->setMargin(KDialog::marginHint());

    label = new TQLabel(i18n("Settings:"),prefWidget);
    tempLayout->addWidget(label);
    
    prefStack = new TQWidgetStack(prefWidget);
    tempLayout->addWidget(prefStack);
    tempLayout->addStretch(1);

    KConfig *config = KGlobal::config();
    dictBox->readSettings(config);
    dictBox->setAutoUpdateOptions(true);
           
    TQStringList modules = dictBox->moduleNames();
    moduleCombo->insertStringList(modules);

    TQPtrList<PrefWidget> prefs = dictBox->modPrefWidgets(prefStack);
    prefs.setAutoDelete(false);

    PrefWidget *p;
    int i=0;
    for(p = prefs.first(); p != 0; p=prefs.next())
    {
        prefStack->addWidget(p,i);
        i++;
    }

    int active=dictBox->activeModule();
    prefStack->raiseWidget(active);
    moduleCombo->setCurrentItem(active);


    TQHBox *h = new TQHBox(this);
    h->setSpacing(KDialog::spacingHint());
    mainLayout->addWidget(h);
    progressLabel = new TQLabel(h);
    progressBar = new KProgress(h);

    connect(textEdit,TQT_SIGNAL(returnPressed()),startButton,TQT_SLOT(animateClick()));
    connect(startButton,TQT_SIGNAL(clicked()),this, TQT_SLOT(startSearch()));
    connect(stopButton, TQT_SIGNAL(clicked()), dictBox,TQT_SLOT(slotStopSearch()));
    connect(editButton, TQT_SIGNAL(clicked()), dictBox, TQT_SLOT(edit()));
    connect(dictBox, TQT_SIGNAL(searchStarted()), this, TQT_SLOT(searchStarted()));
    connect(dictBox, TQT_SIGNAL(searchStopped()), this, TQT_SLOT(searchStopped()));
    connect(dictBox, TQT_SIGNAL(progressed(int)), progressBar, TQT_SLOT(setProgress(int)));
    connect(dictBox, TQT_SIGNAL(activeModuleChanged(bool))
            , editButton, TQT_SLOT(setEnabled(bool)));
    
    connect(dictBox, TQT_SIGNAL(progressStarts(const TQString&))
            , this, TQT_SLOT(progressStarted(const TQString&)));
    connect(dictBox, TQT_SIGNAL(progressEnds()), this, TQT_SLOT(progressStopped()));
    
    connect(moduleCombo, TQT_SIGNAL(activated(int)), 
                    dictBox, TQT_SLOT(setActiveModule(int)));
    connect(dictBox, TQT_SIGNAL(activeModuleChanged(int))
                    , this, TQT_SLOT(switchModule(int)));
    connect(clearButton, TQT_SIGNAL(clicked()), this, TQT_SLOT(slotClearSearch()));
}

/*  
 *  Destroys the object and frees any allocated resources
 */
KBabelDictView::~KBabelDictView()
{
    // no need to delete child widgets, Qt does it all for us
    KConfig *config = KGlobal::config();
    dictBox->saveSettings(config);
}

/* 
 * public slot
 */
void KBabelDictView::startSearch()
{
    TQString text = textEdit->text();
    
    if(!text.isEmpty())
    {
        if(inTransButton->isChecked())
        {
            dictBox->startTranslationSearch(text);
        }
        else
        {
            dictBox->startSearch(text);
        }
    }


}

void KBabelDictView::progressStopped()
{
    progressBar->setProgress(0);
    progressLabel->setText("");
    startButton->setEnabled(true);
    stopButton->setEnabled(false);
}

void KBabelDictView::progressStarted(const TQString& msg)
{
    progressLabel->setText(msg);
    startButton->setEnabled(false);
    stopButton->setEnabled(true);
}

void KBabelDictView::searchStopped()
{
    progressLabel->setText("");
    progressBar->setProgress(0);
    startButton->setEnabled(true);
    stopButton->setEnabled(false);
}

void KBabelDictView::searchStarted()
{
    progressLabel->setText(i18n("Searching"));
    startButton->setEnabled(false);
    stopButton->setEnabled(true);
}

void KBabelDictView::switchModule(int m)
{
    moduleCombo->blockSignals(true);
    moduleCombo->setCurrentItem(m);
    moduleCombo->blockSignals(false);

    prefStack->raiseWidget(m);
}


void KBabelDictView::about()
{
    dictBox->about();
}


void KBabelDictView::aboutModule()
{
    dictBox->aboutActiveModule();
}

int KBabelDictView::togglePref()
{
    int prefWidth=0;
    if(prefWidget->isVisibleTo(this))
    {
        prefWidget->hide();
        prefWidth=prefWidget->width();
    }
    else
    {
        prefWidget->show();
        prefWidth=prefWidget->width();
    }

    return prefWidth;
}

bool KBabelDictView::prefVisible() 
{
    return prefWidget->isVisibleTo(this);
}

void KBabelDictView::slotClearSearch()
{
  textEdit->clear();
  textEdit->setFocus();
}

#include "kbabeldictview.moc"

