/* ****************************************************************************
  This file is part of KBabel

  Copyright (C) 1999-2000 by Matthias Kiefer
                            <matthias.kiefer@gmx.de>
		2001-2005 by Stanislav Visnovsky
			    <visnovsky@kde.org>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.

  In addition, as a special exception, the copyright holders give
  permission to link the code of this program with any edition of
  the Qt library by Trolltech AS, Norway (or with modified versions
  of Qt that use the same license as Qt), and distribute linked
  combinations including the two.  You must obey the GNU General
  Public License in all respects for all of the code used other than
  Qt. If you modify this file, you may extend this exception to
  your version of the file, but you are not obligated to do so.  If
  you do not wish to do so, delete this exception statement from
  your version.

**************************************************************************** */
#include "klisteditor.h"
#include "toolselectionwidget.h"
#include "projectprefwidgets.h"
#include "resources.h"
#include "kbabeldictbox.h"
#include "toolaction.h"
#include "cmdedit.h"
#include "kbprojectsettings.h"

#include <kcombobox.h>
#include <kdatatool.h>
#include <klocale.h>
#include <kdialog.h>
#include <kfiledialog.h>
#include <knuminput.h>
#include <kmessagebox.h>
#include <klineedit.h> 
#include <kurlcompletion.h>
#include <kfontdialog.h>
#include <kcolorbutton.h>
#include <kparts/componentfactory.h>
#include <kregexpeditorinterface.h>
#include <ksconfig.h>
#include <kurldrag.h>
#include <kurlrequester.h>

#include <tqlayout.h>
#include <tqobjectlist.h>
#include <tqlabel.h>
#include <tqvbox.h>
#include <tqlineedit.h>
#include <tqcheckbox.h>
#include <tqgroupbox.h>
#include <tqhbuttongroup.h>
#include <tqvbuttongroup.h>
#include <tqpushbutton.h>
#include <tqcombobox.h>
#include <tqradiobutton.h>
#include <tqspinbox.h>
#include <tqtextcodec.h>
#include <tqwhatsthis.h>

using namespace KBabel;

static TQSize sizeHintForWidget(const TQWidget* widget)
{
  //
  // The size is computed by adding the sizeHint().height() of all
  // widget children and taking the width of the widest child and adding
  // layout()->margin() and layout()->spacing()
  //

  TQSize size;

  int numChild = 0;
  TQObjectList *l = (TQObjectList*)(widget->children());

  for( uint i=0; i < l->count(); i++ )
  {
    TQObject *o = l->at(i);
    if( o->isWidgetType() )
    {
      numChild += 1;
      TQWidget *w=((TQWidget*)o);

      TQSize s = w->sizeHint();
      if( s.isEmpty() == true )
      {
          s = TQSize( 50, 100 ); // Default size
      }
      size.setHeight( size.height() + s.height() );
      if( s.width() > size.width() ) { size.setWidth( s.width() ); }
    }
  }

  if( numChild > 0 )
  {
    size.setHeight( size.height() + widget->layout()->spacing()*(numChild-1) );
    size += TQSize( widget->layout()->margin()*2, widget->layout()->margin()*2 + 1 );
  }
  else
  {
    size = TQSize( 1, 1 );
  }

  return( size );
}




SavePreferences::SavePreferences(TQWidget *parent)
    :  KTabCtl(parent)
{
    TQWidget* page = new TQWidget(this);
    TQVBoxLayout* layout=new TQVBoxLayout(page);
    layout->setSpacing(KDialog::spacingHint());
    layout->setMargin(KDialog::marginHint());

    TQGroupBox* box=new TQGroupBox(1,Qt::Horizontal,page);
    layout->addWidget(box);

    box->setMargin(KDialog::marginHint());
    _updateButton = new TQCheckBox(i18n("&Update header when saving"),box, "kcfg_AutoUpdate");
    _descriptionButton = new TQCheckBox(i18n("Update &description comment when saving"),box, "kcfg_UpdateDescription");
    _autoCheckButton = new TQCheckBox(i18n("Chec&k syntax of file when saving"),box, "kcfg_AutoSyntaxCheck");
    _saveObsoleteButton = new TQCheckBox(i18n("Save &obsolete entries"),box, "kcfg_SaveObsolete");
    
    TQGroupBox* descBox=new TQGroupBox(1,Qt::Horizontal,i18n("De&scription"),page);
    layout->addWidget(descBox);
    
    descBox->setMargin(KDialog::marginHint());
    _descriptionEdit = new TQLineEdit(descBox, "kcfg_DescriptionString");

    TQGroupBox* encodingBox = new TQGroupBox(1,Qt::Horizontal,i18n("Encoding")
                                  ,page);
    encodingBox->setMargin(KDialog::marginHint());
    layout->addWidget(encodingBox);
    TQHBox *b = new TQHBox(encodingBox);

    TQLabel* tempLabel=new TQLabel(i18n("Default:"),b);
    _encodingBox = new TQComboBox(b, "kcfg_Encoding");
    b->setStretchFactor(_encodingBox,2);
    b->setSpacing(KDialog::spacingHint());

    TQString defaultName=charsetString(ProjectSettingsBase::Locale);
    defaultName+=" "+i18n("(default)");
    TQString utf8Name=charsetString(ProjectSettingsBase::UTF8);
    TQString utf16Name=charsetString(ProjectSettingsBase::UTF16);

    _encodingBox->insertItem(defaultName,(int)ProjectSettingsBase::Locale);
    _encodingBox->insertItem(utf8Name,(int)ProjectSettingsBase::UTF8);

    // KBabel seems to crash somehow, when saving in utf16, so
    // it's better to disable this, since it is useless anyway
    // at the moment
    //_encodingBox->insertItem(utf16Name,(int)UTF16);

    tempLabel->setBuddy(_encodingBox);

    _oldEncodingButton = new TQCheckBox(i18n("Kee&p the encoding of the file")
                     ,encodingBox, "kcfg_UseOldEncoding");

    _autoSaveBox = new TQGroupBox( 1, Qt::Horizontal, i18n( "Automatic Saving" ), page );
    _autoSaveBox->setMargin( KDialog::marginHint( ) );
    layout->addWidget( _autoSaveBox );
    _autoSaveDelay = new KIntNumInput( _autoSaveBox, "kcfg_AutoSaveDelay" );
    _autoSaveDelay->setRange( 0, 60 );
    _autoSaveDelay->setSuffix( i18n( "Short for minutes", " min" ) );
    _autoSaveDelay->setSpecialValueText( i18n( "No autosave" ) );

    layout->addStretch(1);
    page->setMinimumSize(sizeHintForWidget(page));
    addTab(page, i18n("&General"));
    
    page = new TQWidget(this);
    layout=new TQVBoxLayout(page);
    layout->setSpacing(KDialog::spacingHint());
    layout->setMargin(KDialog::marginHint());

    TQGroupBox* gridBox = new TQGroupBox(2,Qt::Horizontal,i18n("Fields to Update"),page);
    layout->addWidget(gridBox);
    gridBox->setMargin(KDialog::marginHint());

    _revisionButton = new TQCheckBox(i18n("Re&vision-Date"),gridBox, "kcfg_UpdateRevisionDate");
    _lastButton = new TQCheckBox(i18n("Last-&Translator"),gridBox, "kcfg_UpdateLastTranslator");
    _languageButton = new TQCheckBox(i18n("&Language"),gridBox, "kcfg_UpdateLanguageTeam");
    _charsetButton = new TQCheckBox(i18n("Char&set"),gridBox, "kcfg_UpdateCharset");
    _encodingButton = new TQCheckBox(i18n("&Encoding"),gridBox, "kcfg_UpdateEncoding");
    _projectButton = new TQCheckBox(i18n("Pro&ject"),gridBox, "kcfg_UpdateProject");

    TQButtonGroup* dateBox = new TQButtonGroup(2,Qt::Horizontal,i18n("Format of Revision-Date"),page, "kcfg_DateFormat");
    layout->addWidget(dateBox);
    box->setMargin(KDialog::marginHint());

    // we remove/insert default date button to correctly map Qt::DateFormat to our Ids
    _defaultDateButton = new TQRadioButton( i18n("De&fault date format"),dateBox );
    dateBox->remove (_defaultDateButton);
    _localDateButton = new TQRadioButton( i18n("Local date fo&rmat"),dateBox );
    dateBox->remove (_localDateButton);
    _customDateButton = new TQRadioButton( i18n("Custo&m date format:"),dateBox );

    dateBox->insert (_defaultDateButton);
    dateBox->insert (_localDateButton);

    _dateFormatEdit = new TQLineEdit(dateBox, "kcfg_CustomDateFormat");
    _dateFormatEdit->setEnabled(false);

    connect( _customDateButton, TQT_SIGNAL(toggled(bool)), this, TQT_SLOT( customDateActivated(bool) ) );

    TQGroupBox* projectBox = new TQGroupBox(1,Qt::Horizontal,i18n("Project String")
                                  ,page);
    projectBox->setMargin(KDialog::marginHint());
    layout->addWidget(projectBox);
    b = new TQHBox(projectBox);

    tempLabel=new TQLabel(i18n("Project-Id:"),b);
    _projectEdit = new TQLineEdit(b, "kcfg_ProjectString");
    b->setStretchFactor(_projectEdit,2);
    b->setSpacing(KDialog::spacingHint());
    tempLabel->setBuddy(_projectEdit);
    
    layout->addStretch(1);
    page->setMinimumSize(sizeHintForWidget(page));
    addTab(page, i18n("&Header"));
    
    page = new TQWidget(this);
    layout=new TQVBoxLayout(page);
    layout->setSpacing(KDialog::spacingHint());
    layout->setMargin(KDialog::marginHint());

    TQGroupBox* translatorCopyrightBox = new TQGroupBox(1,Qt::Horizontal, page);
    translatorCopyrightBox->setMargin(KDialog::marginHint());
    _translatorCopyrightButton = 
	new TQCheckBox(i18n("Update &translator copyright")
	    ,translatorCopyrightBox, "kcfg_UpdateTranslatorCopyright");
    layout->addWidget(translatorCopyrightBox);

    TQGroupBox* fsfBox=new TQButtonGroup(1,Qt::Horizontal,i18n("Free Software Foundation Copyright"),page, "kcfg_FSFCopyright");
    layout->addWidget(fsfBox);

    fsfBox->setMargin(KDialog::marginHint());
    _removeFSFButton = new TQRadioButton(i18n("&Remove copyright if empty"),fsfBox);
    _updateFSFButton = new TQRadioButton(i18n("&Update copyright"),fsfBox);
    _nochangeFSFButton = new TQRadioButton(i18n("Do &not change"),fsfBox);

    layout->addStretch(1);
    page->setMinimumSize(sizeHintForWidget(page));
    addTab(page, i18n("Cop&yright"));

    TQWhatsThis::add(_updateButton,
        i18n("<qt><p><b>Update Header</b></p>\n"
             "<p>Check this button to update the header "
             "information of the file "
             "every time it is saved.</p>\n"
             "<p>The header normally keeps information about "
             "the date and time the file was last\n"
             "updated, the last translator etc.</p>\n"
             "<p>You can choose which information you want to update from the checkboxes below.\n"
             "Fields that do not exist are added to the header.\n"
             "If you want to add additional fields to the header, you can edit the header manually by choosing\n"
             "<b>Edit->Edit Header</b> in the editor window.</p></qt>"));

    TQWhatsThis::add(gridBox,i18n("<qt><p><b>Fields to update</b></p>\n"
 "<p>Choose which fields in the header you want to have updated when saving.\n"
 "If a field does not exist, it is appended to the header.</p>\n"
 "<p>If you want to add other information to the header, you have to edit the header manually\n"
 "by choosing <b>Edit->Edit Header</b> in the editor window.</p>\n"
 "<p>Deactivate <b>Update Header</b> above if you do not want to have the header\n"
 "updated when saving.</p></qt>"));

    TQWhatsThis::add(encodingBox,i18n("<qt><p><b>Encoding</b></p>"
"<p>Choose how to encode characters when saving to a file. If you are unsure "
"what encoding to use, please ask your translation coordinator.</p>"
"<ul><li><b>%1</b>: this is the encoding that fits the character "
"set of your system language.</li>"
"<li><b>%2</b>: uses Unicode (UTF-8) encoding.</li>"
"</ul></qt>").arg(defaultName).arg(utf8Name) );


    TQWhatsThis::add(_oldEncodingButton
        ,i18n("<qt><p><b>Keep the encoding of the file</b></p>"
        "<p>If this option is activated, files are always saved in the "
        "same encoding as they were read in. Files without charset "
        "information in the header (e.g. POT files) are saved in the "
        "encoding set above.</p></qt>"));

    TQWhatsThis::add(_autoCheckButton,i18n("<qt><p><b>Check syntax of file when saving</b></p>\n"
"<p>Check this to automatically check syntax of file with \"msgfmt --statistics\"\n"
"when saving a file. You will only get a message, if an error occurred.</p></qt>"));

    TQWhatsThis::add(_saveObsoleteButton,i18n("<qt><p><b>Save obsolete entries</b></p>\n"
"<p>If this option is activated, obsolete entries found when the file was open\n"
"will be saved back to the file. Obsolete entries are marked by #~ and are\n"
"created when the msgmerge does not need the translation anymore.\n"
"If the text will appear again, the obsolete entries will be activated again.\n"
"The main drawback is the size of the saved file.</p></qt>"));


    TQWhatsThis::add(dateBox, i18n("<qt><p><b>Format of Revision-Date</b></p>"
"<p>Choose in which format the date and time of the header field\n"
"<i>PO-Revision-Date</i> is saved: <ul>\n"
"<li><b>Default</b> is the format normally used in PO files.</li>\n"
"<li><b>Local</b> is the format specific to your country.\n"
"It can be configured in KDE's Control Center.</li>\n"
"<li><b>Custom</b> lets you define your own format.</li></ul></p> "
"<p>It is recommended that you use the default format to avoid creating non-standard PO files.</p>"
"<p>For more information, see section <b>The Preferences Dialog</b> "
"in the online help.</p>"
"</qt>") );

    setMinimumSize(sizeHint());
}


void SavePreferences::defaults(const KBabel::SaveSettings& _settings)
{
   _updateButton->setChecked(_settings.autoUpdate);

   _lastButton->setChecked(_settings.updateLastTranslator);
   _revisionButton->setChecked(_settings.updateRevisionDate);
   _languageButton->setChecked(_settings.updateLanguageTeam);
   _charsetButton->setChecked(_settings.updateCharset);
   _encodingButton->setChecked(_settings.updateEncoding);
   _projectButton->setChecked(_settings.updateProject);

   _encodingBox->setCurrentItem(_settings.encoding);
   _oldEncodingButton->setChecked(_settings.useOldEncoding);

   _projectEdit->setText(_settings.projectString);
   
   _descriptionButton->setChecked(_settings.updateDescription);
   _descriptionEdit->setText(_settings.descriptionString);
   _translatorCopyrightButton->setChecked(_settings.updateTranslatorCopyright);
   
   switch(_settings.FSFCopyright)
   {
      case ProjectSettingsBase::Update:
         _updateFSFButton->setChecked(true);
         break;
      case ProjectSettingsBase::Remove:
         _removeFSFButton->setChecked(true);
         break;
      case ProjectSettingsBase::NoChange:
         _nochangeFSFButton->setChecked(true);
         break;
      case ProjectSettingsBase::RemoveLine:
         break;
   }
   
   _autoCheckButton->setChecked(_settings.autoSyntaxCheck);
   _saveObsoleteButton->setChecked(_settings.saveObsolete);

   _dateFormatEdit->setText(_settings.customDateFormat);

   switch(_settings.dateFormat)
   {
      case Qt::ISODate:
         _defaultDateButton->setChecked(true);
         break;
      case Qt::LocalDate:
         _localDateButton->setChecked(true);
         break;
      case Qt::TextDate:
         _customDateButton->setChecked(true);
         break;
   }

   _autoSaveDelay->setValue( _settings.autoSaveDelay );
}


void SavePreferences::customDateActivated(bool on)
{
   _dateFormatEdit->setEnabled(on);
   _dateFormatEdit->setFocus();
}

void SavePreferences::setAutoSaveVisible( const bool on )
{
    if( on ) _autoSaveBox->show();
    else _autoSaveBox->hide();
}



IdentityPreferences::IdentityPreferences(TQWidget* parent, const TQString& project)
         : TQWidget(parent)
{
    TQWidget* page = this;
    TQVBoxLayout* layout=new TQVBoxLayout(page);
    layout->setSpacing(KDialog::spacingHint());
    layout->setMargin(KDialog::marginHint());

    if( !project.isEmpty() )
    {
	// show the project name in the widget at the top
	layout->addWidget(new TQLabel(i18n("<font size=\"+1\">Project: %1</font>").arg(project),page));
    }

    TQGroupBox* group = new TQGroupBox(2,Qt::Horizontal,page);
    layout->addWidget(group);
    group->setMargin(KDialog::marginHint());
    
    TQLabel* tempLabel=new TQLabel(i18n("&Name:"),group);
    _nameEdit = new TQLineEdit(group, "kcfg_AuthorName");
    tempLabel->setBuddy(_nameEdit);

    tempLabel=new TQLabel(i18n("Localized na&me:"),group);
    _localNameEdit = new TQLineEdit(group, "kcfg_LocalAuthorName");
    tempLabel->setBuddy(_localNameEdit);

    tempLabel=new TQLabel(i18n("E&mail:"),group);
    _mailEdit = new TQLineEdit(group, "kcfg_AuthorEmail");
    tempLabel->setBuddy(_mailEdit);


    tempLabel=new TQLabel(i18n("&Full language name:"),group);

    TQHBox *hbox = new TQHBox(group);
    hbox->setSpacing(KDialog::spacingHint());
    _langEdit = new TQLineEdit(hbox, "kcfg_Language");
    tempLabel->setBuddy(_langEdit);
    tempLabel=new TQLabel(i18n("Lan&guage code:"),hbox);
    _langCodeEdit = new TQLineEdit(hbox, "kcfg_LanguageCode");
    tempLabel->setBuddy(_langCodeEdit);
    connect(_langCodeEdit,TQT_SIGNAL(textChanged(const TQString&)), this
            , TQT_SLOT(checkTestPluralButton()));

    tempLabel=new TQLabel(i18n("&Language mailing list:"),group);
    _listEdit = new TQLineEdit(group, "kcfg_Mailinglist");
    _listEdit->setMinimumSize(100,_listEdit->sizeHint().height());
    tempLabel->setBuddy(_listEdit);
 
    tempLabel=new TQLabel(i18n("&Timezone:"), group);
    _timezoneEdit = new TQLineEdit(group, "kcfg_Timezone");
    _timezoneEdit->setMinimumSize(100,_timezoneEdit->sizeHint().height());
    tempLabel->setBuddy(_timezoneEdit);


    TQString whatsThisMsg=i18n("<qt><p><b>Identity</b></p>\n"
"<p>Fill in information about you and your translation team.\n"
"This information is used when updating the header of a file.</p>\n"
"<p>You can find the options if and what fields in the header should be updated\n"
"on page <b>Save</b> in this dialog.</p></qt>");

    TQWhatsThis::add(group,whatsThisMsg);


    group = new TQGroupBox(1,Qt::Horizontal,page);
    layout->addWidget(group);
    group->setMargin(KDialog::marginHint());

    hbox = new TQHBox(group);
    hbox->setSpacing(KDialog::spacingHint());

    TQLabel *label = new TQLabel(i18n("&Number of singular/plural forms:"), hbox);
    _pluralFormsBox = new TQSpinBox(0,100,1,hbox, "kcfg_PluralForms");
    _pluralFormsBox->setSpecialValueText(
            i18n("automatic choose number of plural forms","Automatic"));
    label->setBuddy(_pluralFormsBox);
    connect(_pluralFormsBox,TQT_SIGNAL(valueChanged(int)), this
            , TQT_SLOT(checkTestPluralButton()));
    
    hbox->setStretchFactor(_pluralFormsBox,1);

    _testPluralButton = new TQPushButton(i18n("Te&st"),hbox);
    _testPluralButton->setEnabled(false);
    connect(_testPluralButton, TQT_SIGNAL(clicked()), this
            , TQT_SLOT(testPluralForm()));

    const TQString msg=i18n("<qt><p><b>Number of singular/plural forms</b></p>"
            "<p><b>Note</b>: This option is KDE specific. "
            "If you are not translating a KDE application, you can safely "
            "ignore this option.</p>"
            "<p>Choose here how many singular and plural forms are used in "
            "your language. "
            "This number must correspond to the settings of your language "
            "team.</p>"
            "<p>Alternatively, you can set this option to "
            "<i>Automatic</i> and KBabel will try to get this information "
            "automatically from KDE. Use the <i>Test</i> button "
            "to test if it can find it out.</p></qt>");
    TQWhatsThis::add(_pluralFormsBox,msg);
    TQWhatsThis::add(_testPluralButton,msg);
    
    TQVBox* vbox = new TQVBox(group);
    vbox->setSpacing(KDialog::spacingHint());

    label = new TQLabel(i18n("&GNU plural form header:"), vbox);
    
    hbox = new TQHBox(vbox);
    hbox->setSpacing(KDialog::spacingHint());
    
    _gnuPluralFormHeaderEdit = new TQLineEdit(hbox, "kcfg_PluralFormsHeader");
    label->setBuddy(_gnuPluralFormHeaderEdit);
    
    hbox->setStretchFactor(_gnuPluralFormHeaderEdit,1);

    _testGnuPluralFormButton = new TQPushButton(i18n("&Lookup"),hbox);
    connect(_testGnuPluralFormButton, TQT_SIGNAL(clicked()), this
            , TQT_SLOT(lookupGnuPluralFormHeader()));

    _checkPluralArgumentBox = new TQCheckBox( i18n("Re&quire plural form arguments in translation")
	, group, "kcfg_CheckPluralArgument" );
    TQWhatsThis::add(_checkPluralArgumentBox, 
	i18n("<qt><p><b>Require plural form arguments in translation</b></p>\n"
            "<p><b>Note</b>: This option is KDE specific at the moment. "
            "If you are not translating a KDE application, you can safely "
            "ignore this option.</p>\n"
	    "<p>If is this option enabled, the validation check will "
	    "require the %n argument to be present in the message.</p></qt>"));

    TQWhatsThis::add(_gnuPluralFormHeaderEdit, 
	i18n("<qt><p><b>GNU plural form header</b></p>\n"
            "<p>Here you can fill a header entry for GNU plural form handling; "
            "if you leave the entry empty, the entry in the PO file will not be "
	    "changed or added.</p>\n"
	    "<p>KBabel can automatically try to determine value suggested by the "
	    "GNU gettext tools for currently set language; just press the <b>Lookup</b> "
	    "button.</p></qt>"));

    layout->addStretch(1);

    page->setMinimumSize(sizeHintForWidget(page));

    setMinimumSize(sizeHint());

    _mailEdit->installEventFilter(this);
    _listEdit->installEventFilter(this);
}

void IdentityPreferences::defaults(const IdentitySettings& settings)
{
    _nameEdit->setText(settings.authorName);
    _localNameEdit->setText(settings.authorLocalizedName);
    _langEdit->setText(settings.languageName);
    _langCodeEdit->setText(settings.languageCode);
    _listEdit->setText(settings.mailingList);
    _timezoneEdit->setText(settings.timeZone);
    _pluralFormsBox->setValue(settings.numberOfPluralForms);
    _gnuPluralFormHeaderEdit->setText(settings.gnuPluralFormHeader);
    _checkPluralArgumentBox->setChecked(settings.checkPluralArgument);
}

bool IdentityPreferences::eventFilter(TQObject *o, TQEvent *e)
{
    if(e->type() == TQEvent::Drop)
    {
        TQDropEvent *de = static_cast<TQDropEvent*>(e);
        KURL::List urlList;
        if(de && KURLDrag::decode(de,urlList))
        {
            KURL url(urlList.first());
            if(url.protocol()== "mailto")
            {
                TQString mail=url.path();

                bool handled=false;
                if(o == _mailEdit)
                {
                    handled=true;
                    _mailEdit->setText(mail);
                }
                else if(o == _listEdit)
                {
                    handled=true;
                    _listEdit->setText(mail);
                }

                if(handled)
                    return true;
            }
        }
    }

    return false;
}

void IdentityPreferences::checkTestPluralButton()
{
    int val = _pluralFormsBox->value();
    TQString lang=_langCodeEdit->text();
    
    _testPluralButton->setEnabled(val==0 && !lang.isEmpty());
}

void IdentityPreferences::testPluralForm()
{
    TQString lang=_langCodeEdit->text();

    if(lang.isEmpty())
    {
        KMessageBox::sorry(this,i18n("Please insert a language code first."));
        return;
    }

    int number=Catalog::getNumberOfPluralForms(lang);

    TQString msg;
    
    if(number < 0)
    {
        msg = i18n("It is not possible to find out the number "
                "of singular/plural forms automatically for the "
                "language code \"%1\".\n"
                "Do you have kdelibs.po installed for this language?\n"
                "Please set the correct number manually.").arg(lang);
    }
    else
    {
        msg = i18n("The number of singular/plural forms found for "
                "the language code \"%1\" is %2.").arg(lang).arg(number);
    }

    if(!msg.isEmpty())
    {
        KMessageBox::information(this,msg);
    }
}

void IdentityPreferences::lookupGnuPluralFormHeader()
{
    TQString lang=_langCodeEdit->text();

    if(lang.isEmpty())
    {
        KMessageBox::sorry(this,i18n("Please insert a language code first."));
        return;
    }

    TQString header=GNUPluralForms(lang);
    
    if( header.isEmpty() )
    {
	KMessageBox::information(this, i18n("It was not possible to determine "
	"GNU header for plural forms. Maybe your GNU gettext tools are too "
	"old or they do not contain a suggested value for your language.") );
    }
    else
    {
	_gnuPluralFormHeaderEdit->setText( header );
    }
}


MiscPreferences::MiscPreferences(TQWidget *parent)
                : TQWidget(parent), _regExpEditDialog(0)
{
    TQWidget* page = this;

    TQVBoxLayout* layout=new TQVBoxLayout(page);
    layout->setSpacing(KDialog::spacingHint());
    layout->setMargin(KDialog::marginHint());

    TQGroupBox* box=new TQGroupBox(1,Qt::Horizontal,page);
    box->setMargin(KDialog::marginHint());
    layout->addWidget(box);

    TQHBox *hbox = new TQHBox(box);
    hbox->setSpacing(KDialog::spacingHint());

    TQLabel *label = new TQLabel(i18n("&Marker for keyboard accelerator:"),hbox);
    accelMarkerEdit = new KLineEdit(hbox, "kcfg_AccelMarker");
    accelMarkerEdit->setMaxLength(1);
    label->setBuddy(accelMarkerEdit);
    hbox->setStretchFactor(accelMarkerEdit,1);
    TQString msg=i18n("<qt><p><b>Marker for keyboard accelerator</b></p>"
        "<p>Define here, what character marks the following "
        "character as keyboard accelerator. For example in Qt it is "
        "'&amp;' and in Gtk it is '_'.</p></qt>");
    TQWhatsThis::add(label,msg);
    TQWhatsThis::add(accelMarkerEdit,msg);


    hbox = new TQHBox(box);
    hbox->setSpacing(KDialog::spacingHint());

    label = new TQLabel(i18n("&Regular expression for context information:")
            ,hbox);
    contextInfoEdit = new KLineEdit(hbox, "kcfg_ContextInfo");
    label->setBuddy(contextInfoEdit);
    hbox->setStretchFactor(contextInfoEdit,1);

    msg=i18n("<qt><p><b>Regular expression for context information</b></p>"
        "<p>Enter a regular expression here which defines what is "
        "context information in the message and must not get "
        "translated.</p></qt>");
    TQWhatsThis::add(label,msg);
    TQWhatsThis::add(contextInfoEdit,msg);

    if( !KTrader::self()->query("KRegExpEditor/KRegExpEditor").isEmpty() )
    {
	_regExpButton = new TQPushButton( i18n("&Edit..."), hbox );
	connect( _regExpButton, TQT_SIGNAL( clicked() ), this, TQT_SLOT( regExpButtonClicked()));
    }


    // preferences for mail attachments
    TQVButtonGroup* vbgroup = new TQVButtonGroup(page);
    vbgroup->setTitle(i18n("Compression Method for Mail Attachments"));
    vbgroup->setRadioButtonExclusive(true);
    vbgroup->setMargin(KDialog::marginHint());
    layout->addWidget(vbgroup);

    bzipButton = new TQRadioButton(i18n("tar/&bzip2"), vbgroup, "kcfg_BZipCompression");
    gzipButton = new TQRadioButton(i18n("tar/&gzip"), vbgroup);

    compressSingle = new TQCheckBox(i18n("&Use compression when sending "
					"a single file"), vbgroup, "kcfg_CompressSingleFile");

    layout->addStretch(1);
    page->setMinimumSize(sizeHintForWidget(page));
}

void MiscPreferences::defaults(const MiscSettings& settings)
{
    accelMarkerEdit->setText(settings.accelMarker);
    contextInfoEdit->setText(settings.contextInfo.pattern());
    if( settings.useBzip )
	bzipButton->setChecked (true);
    else
	gzipButton->setChecked (true);
	
    compressSingle->setChecked(settings.compressSingleFile);
}

TQString MiscPreferences::contextInfo() const
{
    TQString temp=contextInfoEdit->text();

    bool quoted=false;
    TQString newStr;

    for(uint i=0; i<temp.length(); i++)
    {
        if(temp[i]=='n')
        {
            quoted=!quoted;
            newStr+=temp[i];
        }
        else if(temp[i]=='n' && quoted)
        {
            newStr[newStr.length()-1]='\n';
            quoted=false;
        }
        else
        {
            quoted=false;
            newStr+=temp[i];
        }
    }

    return newStr;
}

void MiscPreferences::setContextInfo(TQString reg)
{
    reg.replace("\n","\\n");
    contextInfoEdit->setText(reg);
}

void MiscPreferences::regExpButtonClicked()
{
    if ( _regExpEditDialog==0 )
      _regExpEditDialog = KParts::ComponentFactory::createInstanceFromQuery<TQDialog>
	("KRegExpEditor/KRegExpEditor", TQString::null, this );

    KRegExpEditorInterface *iface = dynamic_cast<KRegExpEditorInterface *>( _regExpEditDialog );
    if( iface )
    {
	iface->setRegExp( contextInfoEdit->text() );
	if( _regExpEditDialog->exec() == TQDialog::Accepted )
	    contextInfoEdit->setText( iface->regExp() );
    }
}


SpellPreferences::SpellPreferences(TQWidget* parent)
         : TQWidget(parent)
{
    TQWidget* page = this;
    TQVBoxLayout* layout=new TQVBoxLayout(page);
    layout->setSpacing(KDialog::spacingHint());
    layout->setMargin(KDialog::marginHint());


    onFlyBtn = new TQCheckBox(i18n("On the &fly spellchecking"),page, "kcfg_OnFlySpellCheck");
    layout->addWidget(onFlyBtn);

    TQWhatsThis::add(onFlyBtn, i18n("<qt><p><b>On the fly spellchecking</b></p>"
        "<p>Activate this to let KBabel spell check the text "
	"as you type. Mispelled words will be colored by the error color.</p></qt>"));
	
    spellConfig = new KSpellConfig(page,"spellConfigWidget",0,false);
    layout->addWidget(spellConfig);
    remIgnoredBtn = new TQCheckBox(i18n("&Remember ignored words"),page, "kcfg_RememberIgnored");
    layout->addWidget(remIgnoredBtn);
    
    connect( spellConfig, TQT_SIGNAL( configChanged() )
	, this, TQT_SIGNAL ( settingsChanged() ) );

    TQLabel *tempLabel = new TQLabel(i18n("F&ile to store ignored words:"),page);
    layout->addWidget(tempLabel);
    ignoreURLEdit = new KURLRequester(page, "kcfg_IgnoreURL");
    layout->addWidget(ignoreURLEdit);
    tempLabel->setBuddy(ignoreURLEdit);

    connect(remIgnoredBtn,TQT_SIGNAL(toggled(bool)),ignoreURLEdit
                        ,TQT_SLOT(setEnabled(bool)));


    TQString msg = i18n("<qt><p><b>Remember ignored words</b></p>"
        "<p>Activate this, to let KBabel ignore the words, where you have "
        "chosen <i>Ignore All</i> in the spell check dialog, "
        "in every spell check.</p></qt>");

    TQWhatsThis::add(remIgnoredBtn,msg);
    TQWhatsThis::add(tempLabel,msg);
    TQWhatsThis::add(ignoreURLEdit,msg);

    layout->addStretch(1);

    page->setMinimumSize(sizeHintForWidget(page));

    setMinimumSize(sizeHint());
}



void SpellPreferences::updateWidgets(const SpellcheckSettings& settings)
{
    spellConfig->setClient(settings.spellClient);
    spellConfig->setNoRootAffix(settings.noRootAffix);
    spellConfig->setRunTogether(settings.runTogether);
    spellConfig->setEncoding(settings.spellEncoding);
    spellConfig->setDictionary(settings.spellDict);
}


void SpellPreferences::mergeSettings(SpellcheckSettings& settings) const
{
    settings.noRootAffix=spellConfig->noRootAffix();
    settings.runTogether=spellConfig->runTogether();
    settings.spellClient=spellConfig->client();
    settings.spellEncoding=spellConfig->encoding();
    settings.spellDict=spellConfig->dictionary();

    settings.valid=true;
}

void SpellPreferences::defaults(const SpellcheckSettings& settings)
{
    remIgnoredBtn->setChecked(settings.rememberIgnored);
    ignoreURLEdit->setURL(settings.ignoreURL);
    
    onFlyBtn->setChecked(settings.onFlySpellcheck);

    KSpellConfig spCfg;
    *spellConfig = spCfg;
}

CatmanPreferences::CatmanPreferences(TQWidget* parent)
         : TQWidget(parent)
{
    TQWidget* page = this;

    TQVBoxLayout* layout=new TQVBoxLayout(page);
    layout->setSpacing(KDialog::spacingHint());
    layout->setMargin(KDialog::marginHint());

    TQGroupBox* box=new TQGroupBox(1,Qt::Horizontal,page);
    box->setMargin(KDialog::marginHint());
    layout->addWidget(box);

    TQLabel* label=new TQLabel(i18n("&Base folder of PO files:"),box);
    TQHBox* hbox = new TQHBox(box);
    hbox->setSpacing(KDialog::spacingHint());

    const KFile::Mode mode = static_cast<KFile::Mode>( KFile::Directory | KFile::ExistingOnly | KFile::LocalOnly );
	
												
    _poDirEdit = new KURLRequester(hbox, "kcfg_PoBaseDir");
    _poDirEdit->setMode( mode );
    _poDirEdit->setMinimumSize(250,_poDirEdit->sizeHint().height());
    label->setBuddy(_poDirEdit);


    label=new TQLabel(i18n("Ba&se folder of POT files:"),box);
    hbox = new TQHBox(box);
    hbox->setSpacing(KDialog::spacingHint());

    _potDirEdit = new KURLRequester(hbox, "kcfg_PotBaseDir");
    _potDirEdit->setMode( mode );
    _potDirEdit->setMinimumSize(250,_potDirEdit->sizeHint().height());
    label->setBuddy(_potDirEdit);



    TQWhatsThis::add(box,i18n("<qt><p><b>Base folders</b></p>\n"
     "<p>Type in the folders which contain all your PO and POT files.\n"
     "The files and the folders in these folders will then be merged into one\n"
     "tree.</p></qt>"));


    box=new TQGroupBox(1,Qt::Horizontal,page);
    box->setMargin(KDialog::marginHint());
    layout->addWidget(box);

    _openWindowButton = new TQCheckBox(i18n("O&pen files in new window"),box, "kcfg_OpenWindow");


    TQWhatsThis::add(_openWindowButton,i18n("<qt><p><b>Open files in new window</b></p>\n"
"<p>If this is activated all files that are opened from the Catalog Manager are opened\n"
"in a new window.</p></qt>"));

    _killButton = new TQCheckBox( i18n("&Kill processes on exit") , box, "kcfg_KillCmdOnExit" );

    TQWhatsThis::add( _killButton , i18n("<qt><p><b>Kill processes on exit</b></p>\n"
"<p>If you check this, KBabel tries to kill the processes, that have not exited already when KBabel exits,\n"
"by sending a kill signal to them.</p>\n"
"<p>NOTE: It is not guaranteed that the processes will be killed.</p></qt>") );


    _indexButton = new TQCheckBox( i18n("Create inde&x for file contents"), box, "kcfg_IndexWords" );
    
    TQWhatsThis::add( _indexButton , i18n("<qt><p><b>Create index for file contents</b></p>\n"
"<p>If you check this, KBabel will create an index for each PO file to speed up the find/replace functions.</p>\n"
"<p>NOTE: This will slow down updating the file information considerably.</p></qt>") );

    m_msgfmtButton = new TQCheckBox( i18n("Run &msgfmt before processing a file"), box, "kcfg_msgfmt" );

    TQWhatsThis::add( m_msgfmtButton, i18n("<qt><p><b>Run msgfmt before processing a file</b></p>"
        "<p>If you enable this, KBabel will run Gettext's "
        "msgfmt tool before processing a file.</p>"
        "<p>Enabling this setting is recommended, even if it causes processing to be slower. "
        "This setting is enabled by default.</p>"
        "<p>Disabling is useful for slow computers and when you want "
        "to translate PO files that are not supported by the current version "
        "of the Gettext tools that are on your system. "
        "The drawback of disabling is that hardly any syntax checking is done by the processing code, "
        "so invalid PO files could be shown as good ones, "
        "even if Gettext tools would reject such files.</p></qt>") );

    layout->addStretch(1);

    page->setMinimumSize(sizeHintForWidget(page));

    setMinimumSize(sizeHint());
}


void CatmanPreferences::defaults(const CatManSettings& settings)
{
   _poDirEdit->setURL(settings.poBaseDir);
   _potDirEdit->setURL(settings.potBaseDir);

   _openWindowButton->setChecked(settings.openWindow);

   _killButton->setChecked(settings.killCmdOnExit );
   _indexButton->setChecked(settings.indexWords );
   m_msgfmtButton->setChecked( settings.msgfmt );
}

DirCommandsPreferences::DirCommandsPreferences(TQWidget* parent)
         : TQWidget(parent)
{
    TQWidget* page = this;

    TQVBoxLayout* layout=new TQVBoxLayout(page);
    layout->setSpacing(KDialog::spacingHint());
    layout->setMargin(KDialog::marginHint());

    TQGroupBox* box = new TQGroupBox( 1 , Qt::Horizontal , i18n("Commands for Folders") , page );
    box->setMargin( KDialog::marginHint() );
    layout->addWidget( box );

    _dirCmdEdit = new CmdEdit( box );
    new TQLabel( i18n("Replaceables:\n@PACKAGE@, @PODIR@, @POTDIR@\n"
	"@POFILES@, @MARKEDPOFILES@"), box);
	
    connect (_dirCmdEdit, TQT_SIGNAL(widgetChanged()), this, TQT_SIGNAL(settingsChanged()));

    TQWhatsThis::add( box , i18n("<qt><p><b>Commands for folders</b></p>"
"<p>Insert here the commands you want to execute in folders from "
"the Catalog Manager. The commands are then shown in the submenu "
"<b>Commands</b> in the Catalog Manager's context menu.</p>"
"<p>The following strings will be replaced in a command:<ul>"
"<li>@PACKAGE@: The name of the folder without path</li>"
"<li>@PODIR@: The name of the PO-folder with path</li>"
"<li>@POTDIR@: The name of the template folder with path</li>"
"<li>@POFILES@: The names of the PO files with path</li>"
"<li>@MARKEDPOFILES@: The names of the marked PO files with path</li>"
"</ul></p>"
"</qt>") );



    layout->addStretch(1);
    page->setMinimumSize(sizeHintForWidget(page));

    setMinimumSize(sizeHint());
}


DirCommandsPreferences::~DirCommandsPreferences()
{
}


void DirCommandsPreferences::updateWidgets(const CatManSettings& settings)
{
   _dirCmdEdit->setCommands( settings.dirCommands , settings.dirCommandNames );
}


void DirCommandsPreferences::mergeSettings(CatManSettings& settings) const
{
    _dirCmdEdit->commands( settings.dirCommands , settings.dirCommandNames );
}

void DirCommandsPreferences::defaults(const CatManSettings& settings)
{
   _dirCmdEdit->setCommands( settings.dirCommands, settings.dirCommandNames );
}


FileCommandsPreferences::FileCommandsPreferences(TQWidget* parent)
         : TQWidget(parent)
{
    TQWidget* page = this;

    TQVBoxLayout* layout=new TQVBoxLayout(page);
    layout->setSpacing(KDialog::spacingHint());
    layout->setMargin(KDialog::marginHint());

    TQGroupBox* box=new TQGroupBox( 1 , Qt::Horizontal , i18n("Commands for Files") , page );
    box->setMargin( KDialog::marginHint() );
    layout->addWidget( box );

    _fileCmdEdit = new CmdEdit( box );
    new TQLabel( i18n("Replaceables:\n"
"@PACKAGE@, @POFILE@,@POTFILE@,\n@PODIR@, @POTDIR@"), box);

    connect (_fileCmdEdit, TQT_SIGNAL(widgetChanged()), this, TQT_SIGNAL(settingsChanged()));

    TQWhatsThis::add( box , i18n("<qt><p><b>Commands for files</b></p>"
"<p>Insert here the commands you want to execute on files from "
"the Catalog Manager. The commands are then shown in the submenu "
"<b>Commands</b> in the Catalog Manager's context menu.</p>"
"<p>The following strings will be replaced in a command:<ul>"
"<li>@PACKAGE@: The name of the file without path and extension</li>"
"<li>@POFILE@: The name of the PO-file with path and extension</li>"
"<li>@POTFILE@: The name of the corresponding template file with path "
"and extension</li>"
"<li>@POEMAIL@: The name and email address of the last translator</li>"
"<li>@PODIR@: The name of the folder the PO-file is in, with path</li>"
"<li>@POTDIR@: The name of the folder the template file is in, with "
"path</li></ul></p></qt>") );



    layout->addStretch(1);
    page->setMinimumSize(sizeHintForWidget(page));

    setMinimumSize(sizeHint());
}


FileCommandsPreferences::~FileCommandsPreferences()
{
}


void FileCommandsPreferences::updateWidgets(const CatManSettings& settings)
{
   _fileCmdEdit->setCommands( settings.fileCommands , settings.fileCommandNames );
}


void FileCommandsPreferences::mergeSettings(CatManSettings& settings) const
{
    _fileCmdEdit->commands( settings.fileCommands , settings.fileCommandNames );
}

void FileCommandsPreferences::defaults(const CatManSettings& settings)
{
   _fileCmdEdit->setCommands( settings.fileCommands, settings.fileCommandNames );
}

ViewPreferences::ViewPreferences(TQWidget* parent)
         : TQWidget(parent)
{
    TQWidget* page = this;

    TQVBoxLayout* layout=new TQVBoxLayout(page);
    layout->setSpacing(KDialog::spacingHint());
    layout->setMargin(KDialog::marginHint());

    TQGroupBox* box=new TQGroupBox(2, Qt::Horizontal,i18n("Shown Columns"),page);
    box->setMargin(KDialog::marginHint());
    layout->addWidget(box);

    _flagColumnCheckbox = new TQCheckBox( i18n("Fla&g"), box, "kcfg_ShowFlagColumn" );
    _fuzzyColumnCheckbox = new TQCheckBox( i18n("&Fuzzy"), box, "kcfg_ShowFuzzyColumn" );
    _untranslatedColumnCheckbox = new TQCheckBox( i18n("&Untranslated"), box, "kcfg_ShowUntranslatedColumn" );
    _totalColumnCheckbox = new TQCheckBox( i18n("&Total"), box, "kcfg_ShowTotalColumn" );
    _cvsColumnCheckbox = new TQCheckBox( i18n("SVN/&CVS status"), box, "kcfg_ShowCVSColumn" );
    _revisionColumnCheckbox = new TQCheckBox( i18n("Last &revision"), box, "kcfg_ShowRevisionColumn" );
    _translatorColumnCheckbox = new TQCheckBox( i18n("Last t&ranslator"), box, "kcfg_ShowTranslatorColumn" );

    TQWhatsThis::add(box,i18n("<qt><p><b>Shown columns</b></p>\n"
     "<p></p></qt>"));
     
    layout->addStretch(1);

    page->setMinimumSize(sizeHintForWidget(page));

    setMinimumSize(sizeHint());
}


void ViewPreferences::defaults(const CatManSettings& _settings)
{
    _flagColumnCheckbox->setChecked(_settings.flagColumn);
    _fuzzyColumnCheckbox->setChecked(_settings.fuzzyColumn);
    _untranslatedColumnCheckbox->setChecked(_settings.untranslatedColumn);
    _totalColumnCheckbox->setChecked(_settings.totalColumn);
    _cvsColumnCheckbox->setChecked(_settings.cvsColumn);
    _revisionColumnCheckbox->setChecked(_settings.revisionColumn);
    _translatorColumnCheckbox->setChecked(_settings.translatorColumn);
}

SourceContextPreferences::SourceContextPreferences(TQWidget* parent): TQWidget(parent)
{
    TQWidget* page = this;
    TQVBoxLayout* layout=new TQVBoxLayout(page);
    layout->setSpacing(KDialog::spacingHint());
    layout->setMargin(KDialog::marginHint());
    
    TQHBox* box = new TQHBox(page);
    box->setSpacing(KDialog::spacingHint());
    TQLabel* tempLabel=new TQLabel(i18n("&Base folder for source code:"),box);

    const KFile::Mode mode = static_cast<KFile::Mode>( KFile::Directory | KFile::ExistingOnly | KFile::LocalOnly );
    _coderootEdit = new KURLRequester ( box, "kcfg_CodeRoot" );
    _coderootEdit->setMode( mode );
    _coderootEdit->setMinimumSize( 250, _coderootEdit->sizeHint().height() );
    tempLabel->setBuddy( _coderootEdit );
    layout->addWidget(box);
												
    // FIXME: use KConfigXT    
    _pathsEditor = new KListEditor(page);
    _pathsEditor->setTitle(i18n("Path Patterns"));
    layout->addWidget(_pathsEditor);
    
    connect ( _pathsEditor, TQT_SIGNAL (itemsChanged ())
	, this, TQT_SIGNAL (itemsChanged ()));
    
    _pathsEditor->installEventFilter(this);
				    
    setMinimumSize(sizeHint());
}

SourceContextPreferences::~SourceContextPreferences()
{
}

void SourceContextPreferences::mergeSettings(KBabel::SourceContextSettings& settings) const
{
    settings.sourcePaths=_pathsEditor->list();
}

void SourceContextPreferences::updateWidgets(const KBabel::SourceContextSettings& settings)
{
    _pathsEditor->setList(settings.sourcePaths);
}

void SourceContextPreferences::defaults(const KBabel::SourceContextSettings& settings)
{
    _pathsEditor->setList(settings.sourcePaths);
}

bool SourceContextPreferences::eventFilter( TQObject *, TQEvent *e )
{
    if( e->type() == TQEvent::KeyPress )
    {
        TQKeyEvent *ke = dynamic_cast<TQKeyEvent*>(e);
        if( ke->key() == Key_Return || ke->key() == Key_Enter )
            return true;
    }
    return false;
}

#include "projectprefwidgets.moc"
