/*
    This file is part of libkcal.

    Copyright (c) 1998 Preston Brown <pbrown@kde.org>
    Copyright (c) 2001,2003 Cornelius Schumacher <schumacher@kde.org>
    Copyright (c) 2002 Jan-Pascal van Best <janpascal@vanbest.org>
    Copyright (C) 2003-2004 Reinhold Kainhofer <reinhold@kainhofer.com>

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to
    the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
    Boston, MA 02110-1301, USA.
*/

#ifndef KCAL_RESOURCECALENDAR_H
#define KCAL_RESOURCECALENDAR_H

#include <tqstring.h>
#include <tqdatetime.h>
#include <tqptrlist.h>

#include <kconfig.h>

#include "alarm.h"
#include "todo.h"
#include "event.h"
#include "journal.h"
#include "calendar.h"
#include "exceptions.h"

#include <kresources/resource.h>
#include <kresources/manager.h>
#include <kabc/lock.h>
#include <kdepimmacros.h>

namespace KCal {

class CalFormat;

/**
  This class provides the interfaces for a calendar resource. It makes use of
  the kresources framework.

  \warning This code is still under heavy development. Don't expect source or
  binary compatibility in future versions.
*/
class LIBKCAL_EXPORT ResourceCalendar : public KRES::Resource
{
  Q_OBJECT
  public:
    ResourceCalendar( const KConfig * );
    virtual ~ResourceCalendar();

    /**
      Clears the exception status.
    */
    void clearException();

    /**
      Set exception for this object. This is used by the functions of this
      class to report errors.
    */
    void setException( ErrorFormat *error );

    /**
      Returns an exception, if there is any, containing information about the
      last error that occurred.
    */
    ErrorFormat *exception();

    void setResolveConflict( bool b);

    virtual void writeConfig( KConfig* config );

    /**
      Return rich text with info about the resource. Adds standard info and
      then calls addInfoText() to add info about concrete resources.
    */
    virtual TQString infoText() const;

    /**
      Load resource data. After calling this function all data is accessible by
      calling the incidence/event/todo/etc. accessor functions.

      Whether data is actually loaded within this function or the loading is delayed
      until it is accessed by another function depends on the implementation of
      the resource.

      If loading the data takes significant time, the resource should return
      cached values if available, and return the results via the resourceChanged
      signal. When the resource has finished loading, the resourceLoaded() signal
      is emitted.

      Calling this function multiple times should have the same effect as
      calling it once, given that the data isn't changed between calls.

      This function calls doLoad() which has to be reimplented by the resource
      to do the actual loading.
    */
    bool load();

    /**
      Save resource data. After calling this function it is safe to close the
      resource without losing data.

      Whether data is actually saved within this function or saving is delayed
      depends on the implementation of the resource.

      If saving the data takes significant time, the resource should return from
      the function, do the saving in the background and notify the end of the
      save by emitting the signal resourceSaved().

      This function calls doSave() which has to be reimplented by the resource
      to do the actual saving.

      @param incidence if given as 0, doSave() is called to save all incidences,
             else doSave(incidence) is called to save only the given one
    */
    bool save( Incidence *incidence = 0 );

    /**
      Return true if a save operation is still in progress, otherwise return
      false.
    */
    virtual bool isSaving() { return false; }

    /**
      Return object for locking the resource.
    */
    virtual KABC::Lock *lock() = 0;

    /**
      Add incidence to resource.
      @deprecated use addIncidence(Incidence *,const TQString &) instead.
    */
    virtual KDE_DEPRECATED bool addIncidence( Incidence * );

    /**
      Add incidence to resource and subresource.
    */
    virtual bool addIncidence( Incidence *, const TQString &subresource );

    /**
      Delete incidence from resource.
    */
    virtual bool deleteIncidence( Incidence * );

    /**
      Return incidence with given unique id. If there is no incidence with that
      uid, return 0.
    */
    Incidence *incidence( const TQString &uid );

    /**
      Add event to resource.
      @deprecated use addEvent(Event *,const TQString&) instead.
    */
    virtual KDE_DEPRECATED bool addEvent( Event *event ) = 0;
    virtual bool addEvent( Event *event, const TQString &subresource ) = 0;

    /**
      Delete event from this resource.
    */
    virtual bool deleteEvent( Event * ) = 0;

    /**
      Retrieves an event on the basis of the unique string ID.
    */
    virtual Event *event( const TQString &uid ) = 0;

    /**
      Return unfiltered list of all events in calendar. Use with care,
      this can be a bad idea for server-based calendars.
    */
    virtual Event::List rawEvents( EventSortField sortField = EventSortUnsorted, SortDirection sortDirection = SortDirectionAscending ) = 0;

    /**
      Builds and then returns a list of all events that match the
      date specified. Useful for dayView, etc. etc.
    */
    virtual Event::List rawEventsForDate( const TQDate &date, EventSortField sortField = EventSortUnsorted, SortDirection sortDirection = SortDirectionAscending ) = 0;

    /**
      Get unfiltered events for date \a qdt.
    */
    virtual Event::List rawEventsForDate( const TQDateTime &qdt ) = 0;

    /**
      Get unfiltered events in a range of dates. If inclusive is set to true,
      only events which are completely included in the range are returned.
    */
    virtual Event::List rawEvents( const TQDate &start, const TQDate &end,
                                   bool inclusive = false ) = 0;

    /**
      Sets a particular value of the resource's configuration. The possible
      keys are resource specific.

      This method is provided to make it possible
      to set resource-type specific settings without actually linking to
      the resource's library. Its use is discouraged, but in
      some situations the only possibility to avoid unwanted compiling and
      linking dependencies. E.g. if you don't want to link to the remote
      resource, but need to create a remote resource at the URL given in
      yourURL, you can use code like the following:
        KCal::ResourceCalendar *res = manager->createResource( "remote" );
        if ( res ) {
          res->setTimeZoneId( timezone );
          res->setResourceName( i18n("Test resource") );
          res->setValue( "DownloadURL", yourURL );
          manager->add( res );
        }
    */
    virtual bool setValue( const TQString &key, const TQString &value );

  signals:
    /**
      This signal is emitted when the data in the resource has changed. The
      resource has to make sure that this signal is emitted whenever any
      pointers to incidences which the resource has previously given to the
      calling code, become invalid.
    */
    void resourceChanged( ResourceCalendar * );

    /**
      This signal is emitted when loading data into the resource has been
      finished.
    */
    void resourceLoaded( ResourceCalendar * );
    /**
      This signal is emitted when saving the data of the resource has been
      finished.
    */
    void resourceSaved( ResourceCalendar * );

    /**
      This signal is emitted when an error occurs during loading.
    */
    void resourceLoadError( ResourceCalendar *, const TQString &error );
    /**
      This signal is emitted when an error occurs during saving.
    */
    void resourceSaveError( ResourceCalendar *, const TQString &error );

    /**
     This signal is emitted when a subresource is added.
    */
    void signalSubresourceAdded( ResourceCalendar *, const TQString& type,
                                 const TQString& subresource, const TQString& label );

    /**
     This signal is emitted when a subresource is removed.
    */
    void signalSubresourceRemoved( ResourceCalendar *, const TQString &,
                                   const TQString & );

  public:
    /**
      Add a todo to the todolist.
      @deprecated use addTodo(Todo *,const TQString &) instead.
    */
    virtual KDE_DEPRECATED bool addTodo( Todo *todo ) = 0;
    virtual bool addTodo( Todo *todo,  const TQString &subresource ) = 0;

    /**
      Remove a todo from the todolist.
    */
    virtual bool deleteTodo( Todo * ) = 0;
    /**
      Searches todolist for an event with this unique id.

      @return pointer to todo or 0 if todo wasn't found
    */
    virtual Todo *todo( const TQString &uid ) = 0;
    /**
      Return list of all todos.
    */
    virtual Todo::List rawTodos( TodoSortField sortField = TodoSortUnsorted, SortDirection sortDirection = SortDirectionAscending ) = 0;
    /**
      Returns list of todos due on the specified date.
    */
    virtual Todo::List rawTodosForDate( const TQDate &date ) = 0;


    /**
      Add a Journal entry to the resource.
      @deprecated use addJournal(Journal *,const TQString &) instead.
    */
    virtual KDE_DEPRECATED bool addJournal( Journal * ) = 0;
    virtual bool addJournal( Journal *journal, const TQString &subresource ) = 0;

    /**
      Remove a Journal entry from calendar.
    */
    virtual bool deleteJournal( Journal * ) = 0;

    /**
      Return Journal with given unique id.
    */
    virtual Journal *journal( const TQString &uid ) = 0;
    /**
      Return list of all journals.
    */
    virtual Journal::List rawJournals( JournalSortField sortField = JournalSortUnsorted, SortDirection sortDirection = SortDirectionAscending ) = 0;
    /**
      Returns list of journals for the given date.
    */
    virtual Journal::List rawJournalsForDate( const TQDate &date ) = 0;

    /**
      Return all alarms which occur in the given time interval.
    */
    virtual Alarm::List alarms( const TQDateTime &from,
                                const TQDateTime &to ) = 0;

    /**
      Return all alarms which occur before given date.
    */
    virtual Alarm::List alarmsTo( const TQDateTime &to ) = 0;


    /** Returns a list of all incideces */
    Incidence::List rawIncidences();

    /**
      Set time zone id used by this resource, e.g. "Europe/Berlin".
    */
    virtual void setTimeZoneId( const TQString &timeZoneId ) = 0;

    /**
      If this resource has subresources, return a TQStringList of them.
      In most cases, resources do not have subresources, so this is
      by default just empty.
    */
    virtual TQStringList subresources() const { return TQStringList(); }

    /**
      Is this subresource capable of having subresources or not?
    */
    virtual bool canHaveSubresources() const { return false; }

    /**
      Is this subresource active or not?
    */
    virtual bool subresourceActive( const TQString& ) const { return true; }

    /**
      Is this subresource writable or not?
    */
    virtual bool subresourceWritable( const TQString& ) const;

    /**
      What is the label for this subresource?
     */
    virtual const TQString labelForSubresource( const TQString& resource ) const
    {
       // the resource identifier is a sane fallback
       return resource;
    };

    /**
      Get the identifier of the subresource associated with a specified
      incidence.

      @return the identifier of the subresource or an empty string.
    */
    virtual TQString subresourceIdentifier( Incidence *incidence )
    { Q_UNUSED( incidence ); return TQString(); }



    /**
     * Remove a subresource with the id @param resource
     */
    virtual bool removeSubresource( const TQString& resource );

    /**
     * Add a subresource with the name @param resource and the parent
     * id @param parent.
     */
    virtual bool addSubresource( const TQString& resource, const TQString& parent );

    /**
     * Returns the type of the subresource: "event", "todo" or "journal",
     * TQString() if unknown/mixed.
     */
    virtual TQString subresourceType( const TQString &resource );

    /**
     * Called when we starting adding a batch of incidences.
     * So we don't show the same warnings for each incidence.
     */
    virtual void beginAddingIncidences();

    /**
     * Called when we finish adding a batch of incidences.
     * @see beginAddingIncidences()
     */
    virtual void endAddingIncidences();

  public slots:
    /**
      (De-)activate a subresource.
    */
    virtual void setSubresourceActive( const TQString &, bool active );

  protected:

    bool mResolveConflict;
    /**
      Do the actual loading of the resource data. Called by load().
    */
    virtual bool doLoad() = 0;
    /**
      Do the actual saving of the resource data. Called by save().
    */
    virtual bool doSave() = 0;

    /**
      Do the actual saving of the resource data. Called by save().
      Save one Incidence. The default implementation calls doSave() to save everything
    */
    virtual bool doSave( Incidence * );

    /**
      Add info text for concrete resources. Called by infoText().
    */
    virtual void addInfoText( TQString & ) const {};

    /**
      A resource should call this function if a load error happens.
    */
    void loadError( const TQString &errorMessage = TQString::null );
    /**
      A resource should call this function if a save error happens.
    */
    void saveError( const TQString &errorMessage = TQString::null );

  private:
    bool mReceivedLoadError;
    bool mReceivedSaveError;

    ErrorFormat *mException;

    class Private;
    Private *d;
};

typedef KRES::Manager<ResourceCalendar> CalendarResourceManager;

}

#endif
