
/***************************************************************************
                                  kinetd.h
                                ------------
    begin                : Mon Feb 11 2002
    copyright            : (C) 2002 by Tim Jansen
    email                : tim@tjansen.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef _KINETD_H_
#define _KINETD_H_

#include <kdedmodule.h>
#include <kservice.h>
#include <ksock.h>
#include <kprocess.h>
#include <tqstringlist.h>
#include <tqstring.h>
#include <tqdatetime.h>
#include <tqtimer.h>
#include <dnssd/publicservice.h>

#include "kserviceregistry.h"

class PortListener : public TQObject {
	Q_OBJECT
private:
	bool m_valid;
	TQString m_serviceName;
	TQString m_serviceURL, m_serviceAttributes;
	TQStringList m_registeredServiceURLs;
	TQString m_dnssdName, m_dnssdType;
	TQMap<TQString,TQString> m_dnssdData;
	int m_serviceLifetime;
	int m_port;
	int m_portBase, m_autoPortRange;
	int m_defaultPortBase, m_defaultAutoPortRange;
	bool m_multiInstance;
	TQString m_execPath;
	TQString m_argument;
	bool m_enabled;
	bool m_serviceRegistered, m_registerService;
	bool m_dnssdRegister, m_dnssdRegistered;
	TQDateTime m_expirationTime;
	TQDateTime m_slpLifetimeEnd;
	TQString m_uuid;

	KServerSocket *m_socket;
	KProcess m_process;

	KConfig *m_config;
	KServiceRegistry *m_srvreg;
	DNSSD::PublicService *m_dnssdreg;

	void freePort();
	void loadConfig(KService::Ptr s);
	void setEnabledInternal(bool e, const TQDateTime &ex);
	void dnssdRegister(bool enabled);
	void setServiceRegistrationEnabledInternal(bool enabled);

public:
	PortListener(KService::Ptr s, KConfig *c, KServiceRegistry *srvreg);
	~PortListener();

	bool acquirePort();
	bool isValid();
	TQString name();
	void setEnabled(bool enabled);
	void setEnabled(const TQDateTime &expiration);
	void setServiceRegistrationEnabled(bool enabled);
	bool isServiceRegistrationEnabled();
	TQDateTime expiration();
	TQDateTime serviceLifetimeEnd();
	bool isEnabled();
	int port();
	TQStringList processServiceTemplate(const TQString &a);
	bool setPort(int port = -1, int autoProbeRange = 1);
	void refreshRegistration();

private slots:
	void accepted(KSocket*);
};

class KInetD : public KDEDModule {
	Q_OBJECT
	K_DCOP

k_dcop:
	/**
	 * Returns a list of all registered services in KInetd.
	 * To add a service you need to add a .desktop file with
	 * the servicetype "KInetDModule" into the services director
	 * (see kinetdmodule.desktop in servicetypes dir).
	 * @return a list with the names of all services
	 */
	TQStringList services();

	/**
	 * Returns true if the service exists and is available.
	 * @param service name of a service as specified in its .desktop file
	 * @return true if a service with the given name exists and is enabled
	 */
	bool isEnabled(TQString service);

	/**
	 * Enables or disabled the given service. Ignored if the given service
	 * does not exist.
	 * @param service name of a service as specified in its .desktop file
	 * @param enable true to enable, false to disable.
	 */
	void setEnabled(TQString service, bool enable);

	/**
	 * Enables the given service until the given time. Ignored if the given
	 * service does not exist.
	 * @param service name of a service as specified in its .desktop file
	 * @param expiration the time the service will be disabled at
	 */
	void setEnabled(TQString service, TQDateTime expiration);

	/**
	 * Returns the port of the service, or -1 if not listening.
	 * @param service name of a service as specified in its .desktop file
	 * @return the port or -1 if no port used or service does not exist
	 */
	int port(TQString service);

	/**
	 * Sets the port of the service, and possibly a range of ports to try.
	 * It will return true if a port could be found. If it didnt find one but is
	 * enabled it will start a timer that probes that port every 30s.
	 * @param service name of a service as specified in its .desktop file
	 * @param port the first port number to try or -1 to restore defaults
	 * @param autoPortRange the number of ports to try
	 * @return true if a port could be found or service is disabled, false 
	 *          otherwise. 
	 */
	bool setPort(TQString service, int port = -1, int autoPortRange = 1);

	/**
	 * Tests whether the given service is installed..
	 * @param service name of a service as specified in its .desktop file
	 * @return true if installed, false otherwise
	 */
	bool isInstalled(TQString service);

	/**
	 * Enables or disables the SLP registration. Ignored if the service does
	 * not have a service URL. If the service is disabled the service will
	 * registered as soon as it is enabled.
	 * @param service name of a service as specified in its .desktop file
	 * @param enable true to enable, false to disable.
	 */
	void setServiceRegistrationEnabled(TQString service, bool enabled);

	/**
	 * Returns true if service registration for the given service is enabled.
	 * Note that this does not mean that the service is currently registered,
	 * because the service may be disabled.
	 * @param service name of a service as specified in its .desktop file
	 * @return true if service registration is enabled
	 */
	bool isServiceRegistrationEnabled(TQString service);

 private:
	TQDateTime getNextExpirationTime();
	void setPortRetryTimer(bool retry);
	void setReregistrationTimer();

	KConfig *m_config;
	KServiceRegistry *m_srvreg;
	TQPtrList<PortListener> m_portListeners;
	TQTimer m_expirationTimer;
	TQTimer m_portRetryTimer;
	TQTimer m_reregistrationTimer;

 private slots:
	void setExpirationTimer();
	void expirationTimer();
	void portRetryTimer();
	void reregistrationTimer();

 public:
	KInetD(TQCString &n);
	virtual ~KInetD();
	void loadServiceList();
	PortListener *getListenerByName(TQString name);
};


#endif
